using System;
using Gtk;
using Gnome;

namespace Kurush
{

public class PrintPreviewEx : ScrolledWindow
{
    static GLib.GType gtype = GLib.GType.Invalid;
    public static new GLib.GType GType
    {
        get
        {
            if (gtype == GLib.GType.Invalid)
                gtype = RegisterGType (typeof (PrintPreviewEx));
            return gtype;
        }
    }

    private Canvas canvas;
    private PrintJob job;
    private PrintPreview preview;
    private int totalPages;
    private int currentPage = -1;
    private double zoom = 1.0;
    private bool fitPage = true;

    public event EventHandler Changed;
    public event EventHandler Assigned;

    public int Pages
    {
        get
        {
            return totalPages;
        }
    }

    public int Page
    {
        get
        {
            return currentPage + 1;
        }
    }

    public PrintPreviewEx () : base (GType)
    {
        this.HscrollbarPolicy = PolicyType.Automatic;
        this.VscrollbarPolicy = PolicyType.Automatic;
        this.ShadowType = ShadowType.In;

        canvas = Canvas.NewAa ();
        Gdk.Color canvasColor = new Gdk.Color (255, 255, 255);
        canvas.ModifyBg (StateType.Normal, canvasColor);
        canvas.CenterScrollRegion = false;
        canvas.Show ();
        this.Add (canvas);
        canvas.SizeAllocated += new SizeAllocatedHandler (CanvasSizeAllocated);
    }

    ~PrintPreviewEx ()
    {
        Dispose ();
    }

    private void Fit (int width)
    {
        double pageWidth;
        double pageHeight;
        job.GetPageSize (out pageWidth, out pageHeight);

        zoom = width / pageWidth;
        canvas.PixelsPerUnit = zoom;
    }

    private  void CanvasSizeAllocated  (object o, SizeAllocatedArgs args)
    {
        if (fitPage)
            Fit (args.Allocation.Width);
    }

    public new void Dispose ()
    {
        if (job != null)
            job.Dispose ();

        if (preview != null)
            preview.Dispose ();

        base.Dispose ();
    }

    public void RenderPage (int page)
    {
        if (job == null)
            return;

        if (page == currentPage)
            return;

        if (preview != null)
            preview.Dispose ();

        preview = new PrintPreview (job.Config, canvas);
        job.RenderPage (preview, page, true);
        preview.Close ();

        currentPage = page;

        if (Changed != null)
            Changed (this, new EventArgs ());
    }

    public void RenderFirstPage ()
    {
        RenderPage (0);
    }

    public void RenderPrevPage ()
    {
        if (currentPage > 0)
            RenderPage (currentPage - 1);
    }

    public void RenderNextPage ()
    {
        if (currentPage < totalPages - 1)
            RenderPage (currentPage + 1);
    }

    public void RenderLastPage ()
    {
        RenderPage (totalPages - 1);
    }

    public double Zoom
    {
        get
        {
            return zoom;
        }
        set
        {
            if (value > 0.2 && value < 5.0)
            {
                fitPage = false;
                zoom = value;
                canvas.PixelsPerUnit = zoom;
            }
        }
    }

    public void ZoomIn ()
    {
        Zoom += 0.1;
    }

    public void ZoomOut ()
    {
        Zoom -= 0.1;
    }

    public void ZoomAll ()
    {
        Zoom = 1;
    }

    public void Print ()
    {
        if (job != null)
            job.Print ();
    }

    public bool ZoomFit
    {
        set
        {
            fitPage = value;
            Fit (canvas.Allocation.Width);
        }
    }

    public PrintJob Job
    {
        get
        {
            return this.job;
        }
        set
        {
            if (this.job != null)
            {
                this.job.Dispose ();
                currentPage = -1;
            }

            this.job = value;
            totalPages = this.job.Pages;

            if (Assigned != null)
                Assigned (this, new EventArgs ());

            RenderFirstPage ();
        }
    }



}

}
