/***************************************************************************
 *   Copyright (C) 2005 by Raul Fernandes                                  *
 *   rgfbr@yahoo.com.br                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ktranslatordictionary.h"
#include "dictionarydialog.h"
#include "ktradddictionarywizard.h"

#include <qlineedit.h>
#include <qcombobox.h>
#include <qmessagebox.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>
#include <klocale.h>

#ifndef NDEBUG

#include <kdebug.h>

#endif


KTranslatorDictionary::KTranslatorDictionary()
{
}

KTranslatorDictionary::~KTranslatorDictionary()
{
}

KTranslatorDictionaryManager::KTranslatorDictionaryManager() : QObject()
{
  offersPlugins = KTrader::self()->query("KTranslator/DictionaryPlugin");
  list.clear();
}

KTranslatorDictionaryManager::~KTranslatorDictionaryManager()
{
}

void KTranslatorDictionaryManager::loadPlugins()
{
  // dictionaries plugin system
  KTranslatorDictionary *dic = NULL;
  KService::Ptr service;
  KTrader::OfferList::ConstIterator iter;
  int errCode;

  // Load dictionaries from config file
  KConfig *config = kapp->config();
  config->setGroup( "general" );
  QStringList dictionaries = config->readListEntry( "dictionaries", ',' );

#ifndef NDEBUG

  kdDebug() << "Dictionaries " << dictionaries << endl;

#endif

  // Add a dictionary if none is configured
  if( dictionaries.count() == 0 )
  {
    QMessageBox::information( 0, i18n( "No Dictionaries" ), i18n( "There are no dictionaries configured.\nMaybe this is the first time you use KTranslator.\nYou should configure one dictionary now to use the program." ) );
    KtrAddDictionaryWizard *dialog = new KtrAddDictionaryWizard();
    if( dialog->exec() != QDialog::Accepted )
    {
      dialog->deleteLater();
      return;
    }

    // Get dictionary name
    QString name = dialog->dicName();

    // Get dictionary file
    QString file = dialog->dicFile();

    // Get dictionary plugin
    QString plugin = dialog->dicPlugin();

    // Write entry to config file
    config->writeEntry( "dictionaries", name );
    config->setGroup( name );
    config->writeEntry( "name", name );
    config->writeEntry( "file", file );
    config->writeEntry( "plugin", plugin );
    config->sync();

    // Append dictionary to dictionary list to be loaded
    dictionaries.append( name );

    // Delete the dialog
    dialog->deleteLater();
  }

  // Load dictionaries
  QStringList args;
  for( QStringList::Iterator it = dictionaries.begin(); it != dictionaries.end(); ++it )
  {
    dic = new KTranslatorDictionary();
    config->setGroup( *it );

#ifndef NDEBUG

    kdDebug() << "Dictionary " << *it << endl;

#endif

    dic->setName( config->readEntry( "name", "" ) );
    dic->setFilename( config->readEntry( "file", "" ) );
    dic->setPluginName( config->readEntry( "plugin", "" ) );

    // Load plugin for this dictionary
    for(iter = offersPlugins.begin(); iter != offersPlugins.end(); ++iter)
    {
      service = *iter;
      if( dic->pluginName() != service->library() ) continue;
      errCode = 0;
      args.clear();
      args.append( dic->filename() );
      args.append( dic->name() );
      dic->plugin = KParts::ComponentFactory::createInstanceFromService<KTranslator::DictionaryPlugin>( service, this, 0, args, &errCode );

      if ( dic->plugin )
      {
       #ifndef NDEBUG

        kdDebug() << "Loaded " << dic->name() << " with plugin " << dic->plugin->name() << endl;

        #endif

        list.append( dic );
        break;
      }

    }
  }
}


void KTranslatorDictionaryManager::addDictionary()
{
  KTranslatorDictionary *dic = NULL;
  KService::Ptr service;
  KTrader::OfferList::ConstIterator iter;
  int errCode;
  KtrAddDictionaryWizard *dialog = new KtrAddDictionaryWizard();

  if( dialog->exec() != QDialog::Accepted )
  {
    dialog->deleteLater();
    return;
  }

  dic = new KTranslatorDictionary();
  dic->setName( dialog->dicName() );
  dic->setFilename( dialog->dicFile() );
  dic->setPluginName( dialog->dicPlugin() );

  // Load plugin for this dictionary
  QStringList args;
  for(iter = offersPlugins.begin(); iter != offersPlugins.end(); ++iter)
  {
    service = *iter;
    if( dic->pluginName() != service->library() ) continue;
    errCode = 0;
    args.clear();
    args.append( dic->filename() );
    args.append( dic->name() );
    dic->plugin = KParts::ComponentFactory::createInstanceFromService<KTranslator::DictionaryPlugin>( service, this, 0, args, &errCode );

    if ( dic->plugin )
    {
      #ifndef NDEBUG

      kdDebug() << "KTranslator:: Loaded " << dic->name() << " with plugin " << dic->plugin->name() << endl;

      #endif

      list.append( dic );

      // Add dictionary to configuration file
      KConfig *config = kapp->config();
      config->setGroup( "general" );
      QStringList dictionaries = config->readListEntry( "dictionaries", ',' );
      dictionaries.append( dic->name() );
      config->writeEntry( "dictionaries", dictionaries );
      config->setGroup( dic->name() );
      config->writeEntry( "name", dic->name() );
      config->writeEntry( "file", dic->filename() );
      config->writeEntry( "plugin", dic->pluginName() );
      config->sync();
      break;
    }

  }

  // Delete the dialog
  dialog->deleteLater();
}

void KTranslatorDictionaryManager::editDictionary( uint item )
{
  KTranslatorDictionary *dic = list.at( item );

  DictionaryDialog *dialog = new DictionaryDialog();
  dialog->setCaption( "Edit Dictionary" );
  dialog->setDicName( dic->name() );
  dialog->nameEdit->setReadOnly( true );
  dialog->setDicFile( dic->filename() );
  dialog->setDicPlugin( dic->pluginName() );

  if( dialog->exec() != QDialog::Accepted )
  {
    dialog->deleteLater();
    return;
  }

  // Change the dictionary group in the configuration file
  KConfig *config = kapp->config();
  config->setGroup( dic->name() );
  config->writeEntry( "name", dialog->dicName() );
  config->writeEntry( "file", dialog->dicFile() );
  config->writeEntry( "plugin", dialog->dicPlugin() );
  config->sync();

  // Code to reload the plugin

  // Delete the dialog
  dialog->deleteLater();
}

void KTranslatorDictionaryManager::removeDictionary( uint item )
{
  if( item > list.count() - 1 ) return;
  KTranslatorDictionary *dic = list.at( item );

  if( QMessageBox::question( NULL, i18n( "Remove Dictionary" ), i18n( "Do you really want to remove the dictionary %1 ?" ).arg( dic->name() ), QMessageBox::Yes, QMessageBox::No ) != QMessageBox::Yes ) return;

  // Remove the dictionary from list
  list.remove( dic );

  // Remove the dictionary from config file
  KConfig *config = kapp->config();
  config->setGroup( "general" );
  QStringList dictionaries = config->readListEntry( "dictionaries", ',' );
  dictionaries.remove( dic->name() );
  if( dictionaries.count() == 0 ) config->deleteEntry( "dictionaries" );
  else config->writeEntry( "dictionaries", dictionaries );
  config->deleteGroup( dic->name() );
  config->sync();
}
