/***************************************************************************
 *   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qstring.h>
#include <qslider.h>
#include <qradiobutton.h>
#include <qlabel.h>

#include <kapplication.h>
#include <dcopclient.h>
#include <kmessagebox.h>
#include <kcombobox.h>
#include <kdebug.h>
#include <klocale.h>
#include <kprocess.h>

#include <synaptics.h>

#include "myframe.h"
#include "synconfig.h"
#include "synconfigwidget.h"

SynConfigWidget::SynConfigWidget( QWidget *parent, const char *name )
  : SynConfigWidgetBase( parent, name ), m_borderWizard(0)
{
    KLocale::setMainCatalogue( "ksynaptics" );

    connect( cmbTapFinger, SIGNAL( activated( int ) ), this, SLOT( tapFingerChanged( int ) ) );
    connect( cmbCircTrigger, SIGNAL( activated( int ) ), this, SLOT( scrollTriggerChanged( int ) ) );
    connect( generalSwitchOffRB, SIGNAL( clicked() ), this, SLOT( disableNearlyEverything() ) );

    // update version information
    QString libText = libVersionTL->text();
    QString drvText = driverVersionTL->text();
    libText.append( " <b>" + QString( Synaptics::Pad::libraryStr().c_str() ) + "</b>" );
    
    if ( Pad::driverKind() == DV_OUTDATED )
        drvText.append( " <b>" + QString( "Outdated" ) + "</b>" );
    else
        drvText.append( " <b>" + QString( Synaptics::Pad::driverStr().c_str() ) + "</b>" );
    
    libVersionTL->setText( libText );
    driverVersionTL->setText( drvText );

    kdDebug() << k_funcinfo << "Has SHM: " << Synaptics::Pad::hasShm() << endl;
    // check whether the driver is operateable
    if ( !Synaptics::Pad::hasShm() )
    {
        KMessageBox::information(this,
            i18n("Shared Memory is not accessible.\n" 
                "Please add the option 'UseShm ''true''' into the " 
                "touch pad section of /etc/X11/xorg.conf\n"),
            i18n("Touch pad configuration inaccessible")
        );
    }
    else
    {
        kdDebug() << k_funcinfo << "Has driver: " << Synaptics::Pad::hasDriver() << endl;
        if ( !Synaptics::Pad::hasDriver() )
        {
            KMessageBox::information(this,
                tr2i18n(
                    "No usable driver has been found!\n"
                    "Either your driver is not correctly installed or "
                    "you are using a wrong version.\n"
                    "You need at least driver version 0.14.4!\n"
                ),
                tr2i18n("No driver found")
            );
        }
    }
}

void SynConfigWidget::useTappingChanged( bool b )
{
  lblTapTime->setEnabled( b );
  lblShort1->setEnabled( b );
  slTapTime->setEnabled( b );
  lblLong1->setEnabled( b );
  cbFastTaps->setEnabled( b );

  emit changed();
}

void SynConfigWidget::useSmartModeChanged( bool b )
{
  lblDelay->setEnabled( b );
  lblShort2->setEnabled( b );
  slSmartModeDelay->setEnabled( b );
  lblLong2->setEnabled( b );

  emit changed();
}

void SynConfigWidget::useDockingChanged( bool b )
{
  grpSmartMode->setEnabled( b );

  emit changed();
}

void SynConfigWidget::tapFingerChanged( int index )
{
  kdDebug() << k_funcinfo << endl;

  myFrame::PadMode padMode;
  
  switch ( index )
  {
      case 0:
          padMode = myFrame::RightTopCorner;
          break;
      case 1:
          padMode = myFrame::RightBottomCorner;
          break;
      case 2:
          padMode = myFrame::LeftTopCorner;
          break;
      case 3:
          padMode = myFrame::LeftBottomCorner;
          break;
      case 4:
          padMode = myFrame::OneFinger;
          break;
      case 5:
          padMode = myFrame::TwoFingers;
          break;
      case 6:
          padMode = myFrame::ThreeFingers;
          break;
      default:
          padMode = myFrame::NoTrigger;
          break;
  }

  tappingStateFrame->displayState( padMode );
  
  bgTapButton->setButton( finger[index] );

  emit changed();
}

void SynConfigWidget::tapButtonChanged( int id )
{
  finger[cmbTapFinger->currentItem()] = id;

  emit changed();
}

void SynConfigWidget::scrollTriggerChanged( int index )
{
    kdDebug() << k_funcinfo << endl;
    scrollingStateFrame->displayState( index );
    
    emit changed();
}

void SynConfigWidget::horizScrollChanged( bool b )
{
  lblSlow1->setEnabled( b );
  slHorizScrollDelta->setEnabled( b );
  lblFast1->setEnabled( b ); 

  emit changed();
}

void SynConfigWidget::vertScrollChanged( bool b )
{
  lblSlow2->setEnabled( b );
  slVertScrollDelta->setEnabled( b );
  lblFast2->setEnabled( b );

  emit changed();
}

void SynConfigWidget::circScrollChanged( bool b )
{
  lblSlow3->setEnabled( b );
  slCircularDelta->setEnabled( b );
  lblFast3->setEnabled( b );
  lblUse->setEnabled( b );
  cmbCircTrigger->setEnabled( b );
  scrollingStateFrame->setEnabled( b );

  emit changed();
}

void SynConfigWidget::alpsOptionClicked( )
{
    // DONE switch defaults for ALPS/non-ALPS!
    int really = 
        KMessageBox::questionYesNo( 
            this, 
            i18n("This will override the currently set edge borders! Change really?"),
            i18n("Switching Edge Border Defaults"), i18n("Change"), i18n("Do Not Change")
    );

    if ( really == KMessageBox::No )
    {
        // undo the changes!
        cbTreatAsALPS->toggle();
    return;
    }

    if ( cbTreatAsALPS->isChecked() )
    {
      QRect edges = QRect( QPoint( 120, 120 ), QPoint( 830, 650 ) );
      SynConfig::setEdges( edges );
      kdDebug() << k_funcinfo << "use ALPS edges" << endl;
    } 
    else
    {
      QRect edges = QRect( QPoint( 1700, 1700 ), QPoint( 5300, 4200 ) );
      SynConfig::setEdges( edges );
      kdDebug() << k_funcinfo << "use Synaptics edges" << endl;
    }

    emit changed();

    KMessageBox::information(
        this,
    i18n("The new settings have been loaded. Press apply to activate!"),
    i18n("Switching Edge Border Defaults")
    );
}

void SynConfigWidget::widgetModified()
{
    emit changed();
}

void SynConfigWidget::invokeBorderWizard()
{
  if (m_borderWizard)  {
    // we must've canceled before, restart from beginning
    kdDebug() << k_funcinfo << "reusing old wizard..." << endl;
    m_borderWizard->showPage( m_borderWizard->page( 0 ) ); 
  }
  else {
    m_borderWizard = new AdjBorderWizard();
    connect(m_borderWizard, SIGNAL( changed() ), SLOT( widgetModified() ) );
  }

  m_borderWizard->show();
  kdDebug() << k_funcinfo << "invoke wizard..." << endl;
}

void SynConfigWidget::load()
{
  // Docking
  generalDockingCB->setChecked( SynConfig::enableDocking() );
  grpSmartMode->setEnabled( SynConfig::enableDocking() );

  // General Tab
  int tpState = SynConfig::enableTouchPad();

  if ( tpState == SynConfig::EnumEnableTouchPad::NoTaps ) tpState = SynConfig::EnumEnableTouchPad::Enabled;
  kdDebug() << k_funcinfo << tpState << endl;

  if ( tpState  == SynConfig::EnumEnableTouchPad::Disabled )
      disableNearlyEverything();
      
  bgTouchPadSwitch->setButton( tpState );
  slSensitivity->setValue( SynConfig::sensitivity() );
  cbSmartMode->setChecked( SynConfig::enableSmartMode() );
  slSmartModeDelay->setValue( SynConfig::smartModeDelay() );

  useSmartModeChanged( cbSmartMode->isChecked() );
  cbTreatAsALPS->setChecked( SynConfig::treatAsALPS() );
  
  // Tapping Tab
  cbEnableTapping->setChecked( SynConfig::enableTapping() );
  slTapTime->setValue( SynConfig::tapTime() );
  finger[0] = SynConfig::tapRightTop();
  finger[1] = SynConfig::tapRightBottom();
  finger[2] = SynConfig::tapLeftTop();
  finger[3] = SynConfig::tapLeftBottom();
  finger[4] = SynConfig::tapOneFinger();
  finger[5] = SynConfig::tapTwoFingers();
  finger[6] = SynConfig::tapThreeFingers();

  tapFingerChanged( cmbTapFinger->currentItem() );
  useTappingChanged( cbEnableTapping->isChecked() );
  cbFastTaps->setChecked( SynConfig::fastTaps() );

  // Scrolling Tab
  cbHorizScrolling->setChecked( SynConfig::enableHorizontalScrolling() );
  slHorizScrollDelta->setValue( SynConfig::horizontalScrollDelta()*-1 );
  cbVertScrolling->setChecked( SynConfig::enableVerticalScrolling() );
  slVertScrollDelta->setValue( SynConfig::verticalScrollDelta()*-1 );
  cbCircScrolling->setChecked( SynConfig::enableCircularScrolling() );
  slCircularDelta->setValue( SynConfig::circularScrollDelta()*-1 );
  cmbCircTrigger->setCurrentItem( SynConfig::circularScrollTrigger() );
  cbEdgeMotion->setChecked( SynConfig::enableEdgeMotion() );
  cbCoasting->setChecked( SynConfig::enableCoasting() );

  horizScrollChanged( cbHorizScrolling->isChecked() );
  vertScrollChanged( cbVertScrolling->isChecked() );
  circScrollChanged( cbCircScrolling->isChecked() );

  // initialize the edge widget with the proper value
  scrollingStateFrame->displayState( SynConfig::circularScrollTrigger() );
  
  if (!( Pad::hasDriver() and Pad::hasShm() ))
      disableEverything();
}

void SynConfigWidget::save()
{
  kdDebug() << k_funcinfo << "here?" << endl;
  // Docking
  enableDocking( generalDockingCB->isChecked() );
  SynConfig::setEnableDocking( generalDockingCB->isChecked() );

    
  // General Tab
  SynConfig::setEnableTouchPad( bgTouchPadSwitch->selectedId() );
  SynConfig::setSensitivity( slSensitivity->value() );
  SynConfig::setEnableSmartMode( cbSmartMode->isChecked() );
  SynConfig::setSmartModeDelay( slSmartModeDelay->value() );
  SynConfig::setTreatAsALPS( cbTreatAsALPS->isChecked() );

  // Tapping Tab
  SynConfig::setEnableTapping( cbEnableTapping->isChecked() );
  SynConfig::setTapTime( slTapTime->value() );
  SynConfig::setTapRightTop( finger[0] );
  SynConfig::setTapRightBottom( finger[1] );
  SynConfig::setTapLeftTop( finger[2] );
  SynConfig::setTapLeftBottom( finger[3] );
  SynConfig::setTapOneFinger( finger[4] );
  SynConfig::setTapTwoFingers( finger[5] );
  SynConfig::setTapThreeFingers( finger[6] );
  SynConfig::setFastTaps( cbFastTaps->isChecked() );
  SynConfig::setEnableCoasting( cbCoasting->isChecked() );

  // Scrolling Tab
  SynConfig::setEnableHorizontalScrolling( cbHorizScrolling->isChecked() );
  SynConfig::setHorizontalScrollDelta( slHorizScrollDelta->value()*-1 );
  SynConfig::setEnableVerticalScrolling( cbVertScrolling->isChecked() );
  SynConfig::setVerticalScrollDelta( slVertScrollDelta->value()*-1 );
  SynConfig::setEnableCircularScrolling( cbCircScrolling->isChecked() );
  SynConfig::setCircularScrollDelta( slCircularDelta->value()*-1 );
  SynConfig::setCircularScrollTrigger( cmbCircTrigger->currentItem() );
  SynConfig::setEnableEdgeMotion( cbEdgeMotion->isChecked() );

  SynConfig::writeConfig();
}


void SynConfigWidget::enableDocking( bool yesNo )
{
    kdDebug() << k_funcinfo << "switch docking " << (yesNo ? "on" : "off") << endl;
    // set autostart link or remove it!
    if (yesNo)
    {
        KApplication::kdeinitExec("syndock", QStringList(), 0, 0);
    }
    else
    {
        KApplication::dcopClient()->send("syndock", "MainApplication-Interface",
            "quit()", QString::null);
    }
    kdDebug() << k_funcinfo << "done" << endl;
}

void SynConfigWidget::disableEverything( bool onOff )
{
    kdDebug() << k_funcinfo << endl;
    disableNearlyEverything( onOff );
    bgTouchPadSwitch->setDisabled( onOff );
}

void SynConfigWidget::disableNearlyEverything( bool onOff)
{
    kdDebug() << k_funcinfo << endl;
    grpSmartMode->setDisabled( onOff );
    grpSensitivity->setDisabled( onOff );
    grpTouchPadBorders->setDisabled( onOff );
    grpTapEmulation->setDisabled( onOff );
    grpFingers->setDisabled( onOff );
    cbFastTaps->setDisabled( onOff );
    grpScrolling->setDisabled( onOff );
    grpCircScrolling->setDisabled( onOff );
    grpOnAnEdge->setDisabled( onOff );
    generalDockingCB->setDisabled( onOff );
}

#include "synconfigwidget.moc"

