/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifndef KSNIFFER_H_
#define KSNIFFER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kmainwindow.h>
#include <kio/job.h>

#include "sniffer.h"
#include "options/captureoptions.h"
#include "options/snifferoptiontab.h"
//#include "ksnifferiface.h"

class KToggleAction;
class KRecentFilesAction;
class KURL;
class KSnifferMainWidget;
class KSystemTray;
class KTempFile;
class KProcess;
class PacketManager;

class QString;
class QTimer;

class ProtocolNameList;

/**
 * This class serves as the main window for KSniffer. It handles the
 * menus, toolbars, and status bars.
 *
 * @short Main window class
 * @author Giovanni Venturi <giovanni@ksniffer.org>
 */

class KSniffer : public KMainWindow//, virtual public KSnifferIface
{
  Q_OBJECT

public:
  /**
   * Default Constructor
   */
  KSniffer();

  /**
   * Default Destructor
   */
  virtual ~KSniffer();

  KSnifferMainWidget* getMainView() { return m_mainWidgetUI; }
  KSystemTray* getSysTray() { return m_sysTray; }

  /**
   * Use this method to load whatever file/URL you have
   */
  void load(const KURL& url);

protected:
  KSnifferMainWidget* m_mainWidgetUI;
  KSystemTray* m_sysTray;
  /**
   * Overridden virtuals for Qt drag 'n drop (XDND)
   */
  virtual void dragEnterEvent(QDragEnterEvent *event);
  virtual void dropEvent(QDropEvent *event);

protected:
  /**
   * This function is called when it is time for the app to save its
   * properties for session management purposes.
   */
  void saveProperties(KConfig *);

  /**
   * This function is called when this app is restored.  The KConfig
   * object points to the session management config file that was saved
   * with @ref saveProperties
   */
  void readProperties(KConfig *);

  void readSettings();
  void saveSettings();

  /**
   * This function is called when you close main view.
   * It reimplements the virtual bool queryClose from KMainWindow
   * to leave open the application in the system tray.
   * It's a virtual protected function.
  */
  bool queryClose();

private slots:
  void slotNewCapture();
  void slotOpen( const KURL& urlParam = QString::null );
  void slotSave();
  void slotSaveAs();
  void slotPrint();
  void slotPauseContinue();
  void slotStop();
  void slotQuit();

  /**
   * check if the sniffing is ended or if you are still loading a file, so it can
   * enable/disable command and related icons
   */
  void checkIfSniffing();

  void copyingJobFinished( KIO::Job *job );

  void optionsShowStatusbar();
  void optionsConfigureKeys();
  void optionsConfigureToolbars();
  void optionsSettings();
  void applyNewToolbarConfig();

  void changeStatusbar( const QString& text );
  void changeCaption( const QString& text );
  void readyStatusbar();

  // show and hide the tray bar
  void showTrayBar( bool );

private:
  void setupAccel();
  void setupActions();

  void stopSniffing();
  void showAfterDisplay( const KURL& urlParam );

  void prepareXMLConfigFile();
  void startLiveSniffing( const QString& interface );

  /**
   * prepare loading packets from file you disable "New Capture",
   * "Open", "Open Recent", ... enable "Stop Capture", ...
   */
  void startSniffingFromFile();
  void resetOptions();

private:
  KToggleAction *m_statusbarAction;

  KAction *m_actNew;                    // the action to start new network packet-capture
  KAction *m_actOpen;                   // the action to open a file
  KAction *m_actSave;                   // the action to save a file with all network captured packets
  KAction *m_actSaveAs;                 // the action to "save as" a file with all network captured packets
  KAction *m_actPrint;                  // the action to print captured packets
  KAction *m_actPauseContinue;          // the action to pause / continue capturing packets
  KAction *m_actStop;                   // the action to stop network packet-capture
  QString m_whatsthisPause;             // the string for the what's this help for Pause Capture
  KRecentFilesAction *m_actOpenRecent;  // the action to open recent URL
  bool m_doingCapture;                  // true if it's sniffing
  bool m_doingPause;                    // true if capture is in pause

  ProtocolNameList *m_protocolNameList;
  Sniffer *m_sniffer;
  CaptureOptions *m_options;
  SnifferOptionTab *m_snifferOption;

  // is needed a PacketManager to store packets and manage them for the View
  PacketManager *m_packets;

  // needed to know if sniffing is from file or not
  bool m_loadFromFile;

  /**
   * needed by slotStop to understand if it's showing captured packets
   * after capture was stopped (true)
   * or if it's displaying packets from a selected file (false)
   */
  bool m_startedShowing;

  /**
   * need to remove temporary file when I load a non local file
   * and packets are available: to avoid it doesn't remove temporary file not needed
   */
  bool m_wroteTmpLocalFile;

  /**
   * Constant action to known if KSniffer has to be closed with special conditions
   */
  enum {ACT_DO_NOTHING = 0, ACT_QUIT_APPL = 1, ACT_SAVING_PACKETS = 2};

  // the action after Job
  int m_action;

  QString m_strFilename;       // the filename to save
  KURL m_strSavedFilename;     // the filename saved
  bool m_needToSave;           // true if data needs to be saved before quit
  KTempFile *m_tmpFile;        // temporary file needed when loading a not local file
  KConfig *m_config;           // global KSniffer configuration
  bool m_optionChanged;        // during capture option can't take effect so this
                               // remember to apply the modify when starting a new capture
  bool m_mustShowPacketsNow;   // true is started capturing with the 'after show packets' option enabled
  bool m_showTrayBar;
  QTimer *m_timer;

  long m_filesize;             // size of the sniffing file
  KProcess *m_sniffProcess;      // the suid process to sniff packets on file
};

#endif // KSNIFFER_H_
