/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifndef SNIFFER_H
#define SNIFFER_H

#if !defined(__FreeBSD__) && !defined(__NetBSD__) && !defined(__OpenBSD__) && \
    !defined(__bsdi__) && !defined(__APPLE__) && !defined(_WIN32) && \
    !defined(__CYGWIN__)
# define MUST_DO_SELECT
#endif

#include <pcap.h>

#include <qobject.h>
#include <qthread.h>

#include "options/captureoptions.h"
#include "packet.h"
#include "version.h"

class QString;
class PacketManager;

/**
 * This class implements the sniffer functions
 *
 * @author Giovanni Venturi
 */
class Sniffer : public QObject, public QThread
{
  Q_OBJECT

public:
  /**
   * Constructs a new Sniffer setting all the options at once.
   * @param captureOption the pointer to the class containing all the
   * sniffing options.
   */
  Sniffer( CaptureOptions *options );

  virtual ~Sniffer();

  /**
   * Set the capture options.
   */
  void setCaptureOptions( CaptureOptions *captOpt );

  /**
   * Start sniffing according the wanted capture options.
   */
  void startSniffing();

  /**
   * Stop sniffing.
   */
  void stopSniffing();

  /**
   * Error constant for \ref getPacketFromFile().
   */
  enum {BAD_FORMAT_FILE = -10, NO_RESULT_AVAILABLE = 0, NO_FILE_SELECTED = 10, RIGHT_FORMAT_FILE = 20};

  /**
   * Get packets from a libpcap format file.
   *
   * @param url the libpcap format file to open
   *
   * @return
   *   - BAD_FORMAT_FILE    --> the url is not a libpcap format file
   *   - BAD_PROTOCOL_FILE  --> the url is not a local file
   *   - CORRECT_FILE       --> all ok file can be processed
   */
  int checkFile( QString strFile );

  /**
   * Let you known if sniffing thread is active.
   *
   * @return
   *   - true    --> the sniffing thread is active
   *   - false   --> the sniffing thread is not active
   */
  bool isSniffing();

  void pauseSniffing();
  void continueSniffing();
  bool pausedSniffing() const;

  // return the sniffed packets size
  long packetsTotalSize() const;

  // return the number of sniffed packets
  long countPackets() const;

protected:
  // don't use this member to start the thread
  virtual void run();

signals:
  void gotPacket( ptrPacketType ptrPacket, struct pcap_pkthdr, long ord, int frameType, int hdrLen );

private:
  /**
   * Get packets from a libpcap format file as defined
   * from setCaptureOptions method.
   */
  void startSniffingFromFile();

  /**
   * Get packets from a network interface as defined
   * from setCaptureOptions method.
   */
  void startSniffingInterface();

//FIXME: make sharable the following method to reduce code and errors
  // detect the kind and the length of frame refered by pHandler and set them into frameType and hdrLen
  void setHeaderData( pcap_t *pHandler, int &frameType, int &hdrLen );

private:
  bool m_canSniff;           // true/false ---> start/stop sniffing thread
  QString m_strInterface;    // string to set the sniffing interface
  QString m_strTmpFilePath;  // string for temporary path file

  bool m_bSniffing;          // true if the sniffing thread is active

  bool m_bFromFile;          // true if packets have got from file / false if it's a live capture
  QString m_strFilename;     // file to open

  /*long m_packetsNumber;  // options type for stopping sniffing after X packets
  long m_packetsSize;    // options type for stopping sniffing after X bytes
  long m_packetsTime;    // options type for stopping sniffing after X seconds
  int m_sniffType;*/

  long m_frameNumber;        // the ordinal number of a frame
  bool m_bPauseSniffing;     // true if you have to pause capture
  pcap_t *m_pcapfp;          // a pointer to pcap_t structure
  pcap_dumper_t *m_pdump;    // a pointer to dump packets on file

#ifdef MUST_DO_SELECT
#define CAP_READ_TIMEOUT 250
    int m_pcap_fd;
    fd_set m_fdset;
    struct timeval m_fdtimeout;
#endif

  long m_packetsTotalSize;
  long m_countPackets;
};

#endif  // SNIFFER_H
