/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <netinet/in.h>

#include <qstring.h>
#include <qstringlist.h>

#include <klocale.h>

#include "tcp-protocol.h"
#include "protocolname.h"
#include "../packet.h"
#include "../packetstructures.h"

TcpProtocol::TcpProtocol(Packet *packet)
{
  m_packet = packet;
}


uint16_t TcpProtocol::sourcePort() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;

  return ntohs(tcp->th_sport);
}


uint16_t TcpProtocol::destinationPort() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;

  return ntohs(tcp->th_dport);
}


QString TcpProtocol::flags() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;
  char *flagVector = "FSRPAU", p[6];
  int bit = 0;
  QString result;

  while(bit < 6) {
    if (tcp->th_flags & (1 << bit))
    {
      p[bit] = flagVector[bit];
      switch (p[bit])
      {
        case 'F':
          result += "FIN, ";
          break;
        case 'S':
          result += "SYN, ";
          break;
        case 'R':
          result += "RST, ";
          break;
        case 'P':
          result += "PSH, ";
          break;
        case 'A':
          result += "ACK, ";
          break;
        case 'U':
          result += "URG, ";
          break;
      }
    }
    bit++;
  }
  result = result.left(result.length() - 2 );

  return result;
}


uint16_t TcpProtocol::win() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;

  return ntohs(tcp->th_win);
}


uint16_t TcpProtocol::sequenceNumber() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;

  return ntohl(tcp->th_seq);
}


uint16_t TcpProtocol::dataLength() const
{
  struct TcpHdr *tcp = (struct TcpHdr *) m_packet;

  return tcp->th_off << 2;
}


QStringList TcpProtocol::headerLines() const
{
  QStringList list;
  QString protocol;

  protocol = ProtocolName::detectProtocol( sourcePort() );
  if ( protocol.isEmpty() )
    list << i18n("TCP protocol field", "Source port: %1").arg( sourcePort() );
  else
    list << i18n("TCP protocol field", "Source port: %1 (%2)").arg( sourcePort() ).arg( protocol );

  protocol = ProtocolName::detectProtocol( destinationPort() );
  if ( protocol.isEmpty() )
    list << i18n("TCP protocol field", "Destination port: %1").arg( destinationPort() );
  else
    list << i18n("TCP protocol field", "Destination port: %1 (%2)").arg( destinationPort() ).arg( protocol );

  list << i18n("TCP protocol field", "Window size: %1").arg( win() );
  list << i18n("TCP protocol field", "Sequence number: %1").arg( sequenceNumber() );
  list << i18n("TCP protocol field", "Header length: %1 bytes").arg( dataLength() );
  list << i18n("TCP protocol field", "Flags: %1").arg( flags() );

  return list;
}
