/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kurlrequester.h>
#include <kuniqueapplication.h>
#include <kconfig.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <klocale.h>

#include <kdebug.h>

#include "snifferoptiontab.h"
#include "captureoptions.h"

SnifferOptionTab::SnifferOptionTab(QWidget *parent, const char *name)
  : base_snifferOptionTab(parent,name)
{
  m_stopButtons << m_checkManually;
  m_stopButtons << m_checkPackets;
  m_stopButtons << m_checkSize;
  m_stopButtons << m_checkTime;
  m_editTempDir->setMode( KFile::Directory );

  connect( m_checkWarning, SIGNAL(clicked()), this, SLOT(checkWarning()) );

  connect( m_checkPackets, SIGNAL(clicked()), this, SLOT(checkSelectPackets()) );
  connect( m_checkSize, SIGNAL(clicked()), this, SLOT(checkSelectSize()) );
  connect( m_checkTime, SIGNAL(clicked()), this, SLOT(checkSelectTime()) );
  connect( m_checkManually, SIGNAL(clicked()), this, SLOT(checkStopManually()) );

  connect( m_packetsNum, SIGNAL(valueChanged(int)), this, SLOT(checkSelectPackets()) );
  connect( m_packetsSize, SIGNAL(valueChanged(int)), this, SLOT(checkSelectSize()) );
  connect( m_packetsTime, SIGNAL(valueChanged(int)), this, SLOT(checkSelectTime()) );

  QToolTip::add( m_editTempDir, i18n("The folder where KSniffer stores a temporary file") );
  QWhatsThis::add( m_editTempDir,
    i18n("<p>This is the default temporary folder where KSniffer will store a temporary "\
         "file for packet-captured session.</p><p>If you want the temporary folder "\
         "can be changed here before starting a new capture session.</p>") );

  QToolTip::add( m_afterDisplay, i18n("The packets will be displayed just when you stop capturing them") );
  QWhatsThis::add( m_afterDisplay,
    i18n("<p>The default option is <i>showing packets when you capture it</i>. "\
         "If you check this, when KSniffer will capture packets it will not display them. "\
         "Just when you stop capturing all captured packets will be displayed.</p>"
         "<p>You can change this option during sniffing, but the effect will "\
         "take effect after you make a <b>New capture</b>. If you want simply "\
         "to stop packet-capture for a while you can use the <i>Pause Capture</i> "\
         "feature from the <i>Capture</i> menu.</p>") );

  QToolTip::add( m_trayBar, i18n("KSniffer can be added to the KDE tray bar clicking here") );
  QWhatsThis::add( m_trayBar,
    i18n("<p>The default option is <i>not showing KSniffer icon in the tray bar</i>."\
         "If you want to add KSniffer to the tray bar you have to click here. "\
         "In this way you can access some useful KSniffer functions like start a new "\
         "capure, stop it, pause/continue it directly from the tray bar.</p>") );

  QWhatsThis::add( m_checkPackets,
    i18n("<p>If you check this, you can capture just a certain packets number. "\
         "After that number, capture will stop automatically.</p>"\
         "<p>You can active just a kind of \"<b>Stop capture after</b>\". "\
         "For <i>packets number</i>, for <i>packets size</i> or for "\
         "<i>packets time</i>.</p>") );

  QToolTip::add( m_packetsNum, i18n("Packets number after you want to stop capturing") );

  QWhatsThis::add( m_checkSize,
    i18n("<p>If you check this, you can capture just a certain packets number "\
         "for a total maximum packets size. After that size, capture will stop "\
         "automatically.</p>"\
         "<p>You can active just a kind of \"<b>Stop capture after</b>\". "\
         "For <i>packets number</i>, for <i>packets size</i> or for "\
         "<i>packets time</i>.</p>") );

  QToolTip::add( m_packetsSize, i18n("Total packets size after you want to stop capturing") );

  QWhatsThis::add( m_checkTime,
    i18n("<p>If you check this, you can capture just a certain packets number for a total "\
         "specific maximum time. When you get all packets for a certain time, capture "\
         "will stop automatically.</p>"\
         "<p>You can active just a kind of \"<b>Stop capture after</b>\". "\
         "For <i>packets number</i>, for <i>packets size</i> or for "\
         "<i>packets time</i>.</p>") );

  QToolTip::add( m_packetsTime, i18n("Packets time after you want to stop capturing") );

  QWhatsThis::add( m_checkWarning,
     i18n("<p>If you check this, you will be warn when the capture stops and you got no packets. "\
          "You can choose if you want to be warned with a modal window or with a pop up one "\
          "checking or not the following sub option.</p>") );

  QToolTip::add( m_checkWarning, i18n("Warning when the capture stops and you got no packets") );

  QWhatsThis::add( m_checkPassiveWarn,
     i18n("<p>If you check this, you will be warn when the capture stops and you got no packets "\
          "with a passive pop up, otherwise you will warn by a modal window that stops the "\
          "application till when you click on the OK button.</p>") );

  QToolTip::add( m_checkPassiveWarn, i18n("Warning with a passive pop up when the capture stops and you got no packets") );
}


SnifferOptionTab::~SnifferOptionTab()
{
}


void SnifferOptionTab::checkWarning()
{
  m_checkPassiveWarn->setEnabled(m_checkWarning->isChecked());
}


void SnifferOptionTab::checkStopManually()
{
  exclusiveSelectOf( 0 );
}


void SnifferOptionTab::checkSelectPackets()
{
  exclusiveSelectOf( 1 );
}


void SnifferOptionTab::checkSelectSize()
{
  exclusiveSelectOf( 2 );
}


void SnifferOptionTab::checkSelectTime()
{
  exclusiveSelectOf( 3 );
}


void SnifferOptionTab::exclusiveSelectOf( int which )
{
  int size = static_cast<int>( m_stopButtons.count() );

  for ( int i = 0; i < size; i++ )
    if ( i != which )
      m_stopButtons[i]->setChecked( false );
    else
      m_stopButtons[i]->setChecked( true );

  if (which == 0)
    m_stopButtonSelected = "manually";
  else if (which == 1)
    m_stopButtonSelected = "by packets";
  else if (which == 2)
    m_stopButtonSelected = "by size";
  else if (which == 3)
    m_stopButtonSelected = "by time";
  kdDebug() << "which: " << which << " selected: '" << m_stopButtonSelected << "'" << endl;
}


void SnifferOptionTab::getShowAfterOptions( CaptureOptions *options )
{
  // get just the Show After configuration
  KConfig* c = kapp->config();
  c->setGroup( "General Options" );
  options->setShowAfter( c->readBoolEntry( "Show packets after stopped capture" , false ) );
}


void SnifferOptionTab::getOptions( CaptureOptions *options )
{
  KConfig* c = kapp->config();
  c->setGroup( "General Options" );
  //m_checkShowSplash->setChecked( c->readBoolEntry("Show splash", true) );
  options->setShowAfter( c->readBoolEntry( "Show packets after stopped capture" , false ) );

  options->setTmpDir( c->readPathEntry( "Temp dir", KGlobal::dirs()->resourceDirs( "tmp" ).first() ) );
  options->setTrayBar( c->readBoolEntry( "Show Tray Bar", false ) );
  options->setWarning( c->readBoolEntry( "Warn with no packets", true ) );
  options->setPassiveWarn( c->readBoolEntry( "Passive pop up warning with no packets", false ) );

  // load stopping capture type: manually, ...
  c->setGroup( "Capture Type Options" );
  QString strWhichStopType = c->readEntry( "Stopping Type", "manually" );
  options->setSniffingType( strWhichStopType );

  c->setGroup( "Capture After Packets" );
  options->setPacketsNumber( c->readLongNumEntry( "packets number", 0 ) );

  c->setGroup( "Capture After Size" );
  options->setPacketsSize( c->readLongNumEntry( "size", 0 ) );
  options->setPacketsInfoSizeUnits( c->readNumEntry( "unit", 0 ) );

  c->setGroup( "Capture After Time" );
  options->setPacketsTime( c->readLongNumEntry( "time", 0 ) );
  options->setPacketsInfoTimeUnits( c->readNumEntry( "unit", 0 ) );
}


void SnifferOptionTab::readSettings()
{
  KConfig* c = kapp->config();
  c->setGroup( "General Options" );
  //m_checkShowSplash->setChecked( c->readBoolEntry("Show splash", true) );
  m_afterDisplay->setChecked( c->readBoolEntry( "Show packets after stopped capture" , false ) );
  m_trayBar->setChecked( c->readBoolEntry( "Show Tray Bar", false ) );

  QString tempdir = c->readPathEntry( "Temp dir", KGlobal::dirs()->resourceDirs( "tmp" ).first() );
  m_editTempDir->setURL( tempdir );

  m_checkWarning->setChecked( c->readBoolEntry( "Warn with no packets", true ) );
  if (!m_checkWarning->isChecked())
    m_checkPassiveWarn->setEnabled( false );
  else
    m_checkPassiveWarn->setChecked( c->readBoolEntry( "Passive pop up warning with no packets", false ) );

  // load stopping capture type: manually, ...
  c->setGroup( "Capture Type Options" );
  m_stopButtonSelected = c->readEntry( "Stopping Type", "manually" );
  int whichStopType;

  if (m_stopButtonSelected == "manually")
    whichStopType = 0;
  else if (m_stopButtonSelected == "by packets")
    whichStopType = 1;
  else if (m_stopButtonSelected == "by size")
    whichStopType = 2;
  else if (m_stopButtonSelected == "by time")
    whichStopType = 3;

  c->setGroup( "Capture After Packets" );
  m_packetsNum->setValue( c->readLongNumEntry( "packets number", 0 ) );

  c->setGroup( "Capture After Size" );
  m_packetsSize->setValue( c->readLongNumEntry( "size", 0 ) );
  m_sizeType->setCurrentItem( c->readNumEntry( "unit", 0 ) );

  c->setGroup( "Capture After Time" );
  m_packetsTime->setValue( c->readLongNumEntry( "time", 0 ) );
  m_timeType->setCurrentItem( c->readNumEntry( "unit", 0 ) );

  exclusiveSelectOf( whichStopType );
}


bool SnifferOptionTab::saveSettings()
{
  KConfig* c = kapp->config();
  c->setGroup( "General Options" );
  //c->writeEntry( "Show splash", m_checkShowSplash->isChecked() );

  // saving temporary directory URL
  QString tempDir = m_editTempDir->url();
  QFileInfo fi( tempDir );

  if( fi.isRelative() )
    fi.setFile( fi.absFilePath() );

  if( !fi.exists() )
  {
    if( KMessageBox::questionYesNo( this, i18n("Folder %1 does not exist. Do you want to create it?").arg(tempDir),
          i18n("Create Folder") ) == KMessageBox::Yes )
    {
      if( !KStandardDirs::makeDir( fi.absFilePath() ) )
      {
        KMessageBox::error( this, i18n("Unable to create folder %1").arg(tempDir) );
        return false;
      }
    }
    else
      // the directory does not exist and the user doesn't want to create it
      return false;
  }

  if( fi.isFile() )
  {
    KMessageBox::information( this, i18n("You specified a file for the temporary folder. "
      "KSniffer will use its base path as the temporary folder."),
      i18n("Warning"), i18n("Don't show again.") );
    fi.setFile( fi.dirPath() );
  }

  // check for writing permission
  if( !fi.isWritable() )
  {
    KMessageBox::error( this, i18n("You have not permission to write to %1.").arg(fi.absFilePath()) );
    return false;
  }

  m_editTempDir->setURL( fi.absFilePath() );
  c->writeEntry( "Version", "0.2" );
  c->writeEntry( "Show packets after stopped capture", m_afterDisplay->isChecked() );
  c->writeEntry( "Show Tray Bar", m_trayBar->isChecked() );
  c->writePathEntry( "Temp dir", m_editTempDir->url() );

  c->writeEntry( "Warn with no packets", m_checkWarning->isChecked() );
  c->writeEntry( "Passive pop up warning with no packets", m_checkPassiveWarn->isChecked() );

  emit trayBarShown( m_trayBar->isChecked() );

  // save stopping capture type: manually, ...
  c->setGroup( "Capture Type Options" );
  c->writeEntry( "Stopping Type", m_stopButtonSelected );

  c->setGroup( "Capture After Packets" );
  c->writeEntry( "packets number", m_packetsNum->value() );

  c->setGroup( "Capture After Size" );
  c->writeEntry( "size", m_packetsSize->value() );
  c->writeEntry( "unit", m_sizeType->currentItem());

  c->setGroup( "Capture After Time" );
  c->writeEntry( "time", m_packetsTime->value() );
  c->writeEntry( "unit", m_timeType->currentItem());

  return true;
}

#include "snifferoptiontab.moc"
