/***************************************************************************
                          skymapwindow.cpp  -  description
                             -------------------
    begin                : Sun Jun 3 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlayout.h>
#include <qlabel.h>
#include <qmovie.h>
#include <qpainter.h>
#include <qscrollview.h>
#include <qtooltip.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <kurllabel.h>

#include "resultsplotview.h"

#include "skymapwindow.h"

const QString SkyPixmap      = "skymap";
const QString TargetMovie    = "ksetispy/pics/target-anim.mng";

SkyMapWindow::SkyMapWindow(const char *name)
             : KSetiSpyView(Pixmap, 0, name)
{
  KIconLoader loader;

  const QSize borders = QSize(4, 4);
  const QPixmap background = KIconLoader().loadIcon(SkyPixmap, KIcon::User);
  const QMovie movie = QMovie(kapp->dirs()->findResource("data", TargetMovie));

  setMaximumSize(background.size() + borders);

  QBoxLayout *layout = new QVBoxLayout(this);

  QScrollView *scrollView = new QScrollView(this);
  layout->addWidget(scrollView, 1);

  QLabel *box = new QLabel(scrollView->viewport(), "SkyMapWindow::box");
  box->installEventFilter(this);
  box->setPixmap(background);
  QToolTip::add(box, i18n("Right-click for options"));
  scrollView->addChild(box);

  QLabel *target = new QLabel(box, "SkyMapWindow::target");
  target->installEventFilter(this);
  target->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
  target->setBackgroundPixmap(background);
  target->setBackgroundOrigin(ParentOrigin);
  target->setMovie(movie);

  initLegend();

  telescopeWindow = new TelescopePathWindow(0, "Telescope Path Window");

  history = false;

  popup->setCheckable(true);
  popup->insertItem(i18n("Show &history"), this, SLOT(toggleHistoryVisible()), CTRL+Key_H, 0, 0);
  popup->setItemChecked(0, false);
  popup->insertItem(i18n("&Legend"), legend, SLOT(show()), CTRL+Key_L, 1, 1);
  popup->insertItem(i18n("&Telescope path"), telescopeWindow->topLevelWidget(),
                                             SLOT(show()), CTRL+Key_T, 2, 2);
  popup->insertSeparator(3);

  cache.clear();
  targets = 0;

  updateContent();
}

bool SkyMapWindow::isHistoryVisible() const
{
  return(history);
}

void SkyMapWindow::setHistoryVisible(bool set)
{
  if(set == history)
    return;

  history = set;

  updateContent();
}

SkyMapWindow::~SkyMapWindow()
{
}

void SkyMapWindow::readConfig(bool readGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("SkyMap Window");

  if(readGeometry) {
    QWidget *dialog;
    QRect rect;
    
    dialog = this->topLevelWidget();
    rect = dialog->geometry();
    dialog->setGeometry(config->readRectEntry("Geometry", &rect));
  }
  
  setHistoryVisible(config->readBoolEntry("Show history", history));

  if(readGeometry) {
    QRect rect;

    config->setGroup("SkyMap Window Legend");

    rect = legend->geometry();
    legend->setGeometry(config->readRectEntry("Geometry", &rect));
  }

  telescopeWindow->readConfig(readGeometry);
}

void SkyMapWindow::saveConfig(bool saveGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("SkyMap Window");

  if(saveGeometry)
    config->writeEntry("Geometry", this->topLevelWidget()->geometry());
    
  config->writeEntry("Show history", history);

  if(saveGeometry) {

    config->setGroup("SkyMap Window Legend");

    config->writeEntry("Geometry", legend->geometry());
  }

  telescopeWindow->saveConfig(saveGeometry);
}

void SkyMapWindow::updateContent()
{
  const SetiClientMonitor::State state = kdoc->setiMonitorState();

  QLabel *target = (QLabel *) child("SkyMapWindow::target", "QLabel");

  if(state >= SetiClientMonitor::Idle)
  {
    SetiClientMonitor *monitor = kdoc->setiMonitor();
    const seti_data *data = monitor->setiData();

    const double ra = data->wu.start.ra;
    const double dec = data->wu.start.dec;

    this->setCaption(i18n("Work Unit Position: %1 RA, %2 Dec").arg(SetiClientMonitor::raToString(ra))
                                                              .arg(SetiClientMonitor::decToString(dec)));

    QSize target_size = target->movie()->framePixmap().size();
    QPoint target_loc = position(ra, dec) - QPoint(target_size.width()/2, target_size.height()/2);

    target->move(target_loc);

    if(!target->isVisibleTo(this)) target->show();

    QValueList<log_data> log = monitor->logData();
    if(log == cache)
      for(uint i = 0; i < log.count(); i++)
        if(history) getTarget(i)->show();
        else getTarget(i)->hide();
    else
    {
      KIconLoader loader;

      const QPixmap normal = loader.loadIcon(GaussianIcon, KIcon::User);
      const QPixmap interesting = loader.loadIcon(InterestingGaussianIcon, KIcon::User);
      const QPixmap background = loader.loadIcon(SkyPixmap, KIcon::User);

      for(uint i = 0; i < log.count(); i++)
      {
        QPixmap icon = log[i].interesting ? interesting : normal;
        QPoint loc = position(log[i].start.ra, log[i].start.dec) - QPoint(icon.width()/2, icon.height()/2);

        QLabel *target = getTarget(i);

        if(history) target->show();
        else target->hide();

        target->move(loc);
        target->resize(icon.size());
        target->setPixmap(icon);
        target->setBackgroundPixmap(background);
        target->setBackgroundOrigin(ParentOrigin);

        KLocale *locale = KGlobal::locale();
        QString tooltip;

        tooltip  = i18n("WU name: %1").arg(log[i].name) + "\n";
        tooltip += i18n("Returned: %1").arg(locale->formatDateTime(log[i].done)) + "\n";
        tooltip += i18n("Position: %1 RA, %2 Dec").arg(SetiClientMonitor::raToString(log[i].start.ra))
                                                  .arg(SetiClientMonitor::decToString(log[i].start.dec)) + "\n";
        tooltip += i18n("Angle Range: %1 (%2 TeraFLOPs)").arg(locale->formatNumber(log[i].angle_range, 3) + "")
                                                         .arg(locale->formatNumber(log[i].teraFLOPs, 2)) + "\n";
        const double hours = log[i].cpu / (60 * 60);
        tooltip += i18n("Process time: %1 hours (%2%)").arg(locale->formatNumber(hours, 3))
                                                       .arg(locale->formatNumber(1e2 * log[i].progress, 2)) + "\n";
        const double speed = (log[i].cpu > 0) ? 1e4 * log[i].teraFLOPs * log[i].progress / log[i].cpu : 0.0;
        tooltip += i18n("Process speed: %1 MegaFLOPs / sec").arg(locale->formatNumber(speed, 1)) + "\n";
        tooltip += i18n("Returned spikes: %1").arg(log[i].spikes.returned) + "\n";
        tooltip += i18n("Best spike signal ratio: %1").arg(log[i].spikes.best_score) + "\n";
        tooltip += i18n("Returned gaussians: %1").arg(log[i].gaussians.returned) + "\n";
        tooltip += i18n("Best gaussian signal ratio: %1").arg(log[i].gaussians.best_score) + "\n";
        tooltip += i18n("Returned pulses: %1").arg(log[i].pulses.returned) + "\n";
        tooltip += i18n("Best pulse score: %1").arg(log[i].pulses.best_score) + "\n";
        tooltip += i18n("Returned triplets: %1").arg(log[i].triplets.returned) + "\n";
        tooltip += i18n("Best triplet score: %1").arg(log[i].triplets.best_score);

        QToolTip::remove(target);
        QToolTip::add(target, tooltip);

        cache = log;
      }
      for(uint i = log.count(); i < targets; i++)
        getTarget(i)->hide();

      target->raise();
    }
  }
  else
  {
    setCaption(i18n("Sky Map"));

    if(target->isVisibleTo(this)) target->hide();
  }
}

void SkyMapWindow::handleURL(const QString& url)
{
  kapp->invokeBrowser(url);
}

void SkyMapWindow::pixmapToClipboard()
{
  setPixmap();
  KSetiSpyView::pixmapToClipboard();
}

void SkyMapWindow::pixmapToFile()
{
  setPixmap();
  KSetiSpyView::pixmapToFile();
}

void SkyMapWindow::setPixmap()
{
  pixmap = KIconLoader().loadIcon(SkyPixmap, KIcon::User);

  QPainter painter(&pixmap);

  QLabel *target = (QLabel *) child("SkyMapWindow::target", "QLabel");
  if(target->isVisibleTo(this))
  {
    QSize target_size = target->movie()->framePixmap().size();
    const int x = target->x() + (target_size.width()/2);
    const int y = target->y() + (target_size.height()/2);

    painter.setPen(red);
    painter.setBrush(red);
    painter.drawPie(x + 11, y + 11, 5, 5, 0, 5760);
    painter.drawArc(x + 5, y + 5, 17, 17, 0, 5760);
  }

  for(uint i = 0; i < targets; i++)
  {
    target = getTarget(i);
    QPixmap *pixmap = target->pixmap();
    
    if(target->isVisibleTo(this) && pixmap != NULL)
      painter.drawPixmap(target->pos(), *pixmap);
  }
}

void SkyMapWindow::initLegend()
{
  legend = new KDialog(this->topLevelWidget(), "Sky Map Legend");
  legend->setCaption(i18n("Sky Map Legend"));

  QBoxLayout *layout = new QVBoxLayout(legend);
  layout->setSpacing(2);

  QLabel *label;
  KURLLabel *urllabel;

  label = new QLabel(i18n("Internet links courtesy of"), legend);
  label->setAlignment(AlignCenter);
  layout->addWidget(label);

  urllabel = new KURLLabel(i18n(ConstellationsSiteHomeURL), i18n(ConstellationsSiteName), legend);
  urllabel->setAlignment(AlignCenter);
  urllabel->setUseTips();
  urllabel->setTipText(i18n("Click to visit web site"));
  layout->addWidget(urllabel);

  connect(urllabel, SIGNAL(leftClickedURL(const QString &)), this, SLOT(handleURL(const QString &)));

  QScrollView *scrollView = new QScrollView(legend);
  scrollView->setResizePolicy(QScrollView::AutoOneFit);
  layout->addWidget(scrollView, 1);

  QWidget *grid = new QWidget(scrollView->viewport());
  scrollView->addChild(grid);

  QGridLayout *gridLayout = new QGridLayout(grid, N_CONSTELLATIONS, 2, 2);
  gridLayout->setMargin(5);
  gridLayout->addColSpacing(0, 40);
  gridLayout->setColStretch(1, 1);

  for(int i = 0; i < N_CONSTELLATIONS; i++)
  {
    label = new QLabel(Constellations[i].abbrev, grid);
    gridLayout->addWidget(label, i, 0);

    urllabel = new KURLLabel(i18n(ConstellationsSiteQueryURL).arg(i18n(Constellations[i].id)),
                             i18n(Constellations[i].name), grid);
    urllabel->setUseTips();
    urllabel->setTipText(i18n("Click to visit website for more information"));
    gridLayout->addWidget(urllabel, i, 1);

    connect(urllabel, SIGNAL(leftClickedURL(const QString &)), this, SLOT(handleURL(const QString &)));
  }
}

QLabel *SkyMapWindow::getTarget(uint n)
{
  QString name;
  QLabel *out = NULL;

  if(n < targets)
  {
    name = QString("SkyMapWindow::target[%1]").arg(n);
    out = (QLabel *) child(name.latin1(), "QLabel");
  }
  else
  {
    for(uint i = targets; i <= n; i++)
    {
      QLabel *box = (QLabel *) child("SkyMapWindow::box", "QLabel");
      name = QString("SkyMapWindow::target[%1]").arg(i);

      out = new QLabel(box, name.latin1());
    }
    targets = n + 1;
  }

  return(out);
}

QPoint SkyMapWindow::position(double ra, double dec)
{
  ra -= 24.0 * int(ra / 24.0); // normalize RAs > 24

  const int x = (ra <= 12.0) ? 481 - int(40.1 * ra) : 481 + int(40.1 * (24.0 - ra));
  const int y = 240 - int(8.0/3.0 * dec);

  return(QPoint(x, y));
}

void SkyMapWindow::toggleHistoryVisible()
{
  bool set = !popup->isItemChecked(0);

  popup->setItemChecked(0, set);
  setHistoryVisible(set);
}

#include "skymapwindow.moc"

