/***************************************************************************
                          progressview.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlabel.h>
#include <qlayout.h>
#include <qwidgetstack.h>

#include <kglobal.h>
#include <klocale.h>
#include <kprogress.h>

#include "progressview.h"

ProgressView::ProgressView(QWidget *parent, const char *name)
             : KSetiSpyView(Text, parent, name)
{
  QGridLayout *layout = new QGridLayout(this, 8, 2);
  layout->setSpacing(8);

  layout->addColSpacing(0, fontMetrics().width(i18n("Speed: ________ MFLOPs")));
  layout->addColSpacing(1, fontMetrics().width(i18n("Estimated total time: __________")));
  layout->setColStretch(0, 1);
  layout->setColStretch(1, 1);

  layout->addWidget(addField(this, "ProgressView::time_spent"), 0, 0);
  setFieldName("ProgressView::time_spent", i18n("CPU time:"));

  layout->addWidget(addField(this, "ProgressView::time_total"), 0, 1);
  setFieldName("ProgressView::time_total", i18n("Estimated total time:"));

  layout->addWidget(addField(this, "ProgressView::speed"), 1, 0);
  setFieldName("ProgressView::speed", i18n("Speed:"));

  layout->addWidget(addField(this, "ProgressView::time_left"), 1, 1);
  setFieldName("ProgressView::time_left", i18n("Estimated time left:"));

  layout->addWidget(addField(this, "ProgressView::rate"), 2, 0);
  setFieldName("ProgressView::rate", i18n("Rate:"));

  layout->addWidget(addField(this, "ProgressView::time_done"), 2, 1);
  setFieldName("ProgressView::time_done", i18n("Est. time done:"));

  layout->addWidget(addField(this, "ProgressView::efficiency"), 3, 0);
  setFieldName("ProgressView::efficiency", i18n("Efficiency:"));

  layout->addWidget(addField(this, "ProgressView::drift"), 3, 1);
  setFieldName("ProgressView::drift", i18n("Drift rate:"));

  layout->addWidget(addField(this, "ProgressView::completed"), 4, 0);
  setFieldName("ProgressView::completed", i18n("Completed:"));

  layout->addWidget(addField(this, "ProgressView::FFT"), 4, 1);
  setFieldName("ProgressView::FFT", i18n("FFT number:"));

  layout->setRowStretch(5, 1);

  QWidgetStack *progressBarContainer = new QWidgetStack(this, "ProgessView::bar_container");
  layout->addMultiCellWidget(progressBarContainer, 6, 6, 0, 1);

  QLabel *label = new QLabel(this);
  label->installEventFilter(this);
  progressBarContainer->addWidget(label, 0);

  KProgress *progressBar = new KProgress(this, "ProgressView::bar");
  progressBar->installEventFilter(this);
  progressBarContainer->addWidget(progressBar, 1);

  layout->addRowSpacing(6, 25);

  layout->setRowStretch(7, 2);

  setMinimumSize(layout->minimumSize());

  updateContent(false);
}

ProgressView::~ProgressView()
{
}

void ProgressView::updateContent(bool force)
{
  // optimization: do not update if this is not the visible widget
  const QWidgetStack *views = (QWidgetStack *) this->parent();
  if(views->visibleWidget() != this && !force) return;

  const SetiClientMonitor::State state = kdoc->setiMonitorState();
  SetiClientMonitor *monitor = kdoc->setiMonitor();

  QWidgetStack *progressBarContainer = (QWidgetStack *) child("ProgessView::bar_container", "QWidgetStack");

  KLocale *locale = KGlobal::locale();

  if(state == SetiClientMonitor::Running)
  {
    const seti_data *data = monitor->setiData();
    const double teraFLOPs = SetiDataMonitor::teraFLOPs(*data);
    const sys_info *info = monitor->sysInfo();
    const int cpu = monitor->currentCPU();

    const double secs = data->state.cpu;
    const double done = data->state.progress;
    const double drift_rate = data->state.cr;
    const int fft = data->state.ncfft;

    setFieldContentText("ProgressView::time_spent", SetiClientMonitor::timeToString(secs));

    const double time_total = (done > 0.0) ? secs/done : 0.0;
    setFieldContentText("ProgressView::time_total", SetiClientMonitor::timeToString(time_total));

    const double speed = (secs > 0.0) ? (1e6 * teraFLOPs * done)/secs : 0.0;
    setFieldContentText("ProgressView::speed",
                        i18n("%1 MFLOPS").arg(locale->formatNumber(speed, 2)));

    const double time_left = time_total - secs;
    setFieldContentText("ProgressView::time_left", SetiClientMonitor::timeToString(time_left));

    const double rate = (secs > 0.0) ? 1e2 * (done/secs) * 60.0 * 60.0 : 0.0;
    setFieldContentText("ProgressView::rate",
                        i18n("%1% / hour").arg(locale->formatNumber(rate, 2)));

    const QDateTime date_time_done = QDateTime::currentDateTime().addSecs(int(time_left));
    const int days = QDateTime::currentDateTime().daysTo(date_time_done);
    QTime time_done = date_time_done.time();
    QString time_done_str = locale->formatTime(time_done);
    if(days > 1) time_done_str += QString(" (+%1)").arg(days-1);
    setFieldContentText("ProgressView::time_done", time_done_str);

    if(info != NULL)
    {
      const double efficiency = (speed > 0.0) ? info->cpus[cpu].MHz / speed : 0.0;
      setFieldContentText("ProgressView::efficiency",
                        i18n("%1 CpF").arg(locale->formatNumber(efficiency, 2)));
    }
    else
      setFieldContentText("ProgressView::efficiency", i18n(unknownContent));

    setFieldContentText("ProgressView::drift",
                        i18n("%1 Hz/s").arg(locale->formatNumber(drift_rate, 4)));

    const double completed = 1e2 * done;
    const double teraFLOPs_completed = teraFLOPs * done;
    setFieldContentText("ProgressView::completed",
                        QString("%1%").arg(locale->formatNumber(completed, 2)));
    setFieldToolTip("ProgressView::completed",
                    i18n("%1 of %2 TeraFLOPs").arg(locale->formatNumber(teraFLOPs_completed, 3))
                                              .arg(locale->formatNumber(teraFLOPs, 3)));

    setFieldContentText("ProgressView::FFT", QString::number(fft));

    KProgress *progressBar = (KProgress *) child("ProgressView::bar", "KProgress");
    progressBar->setValue(int(completed));

    progressBarContainer->raiseWidget(1);
  }
  else
  {
    switch(state) {
      case SetiClientMonitor::No_Data:
      case SetiClientMonitor::No_WU:
        setFieldContentText("ProgressView::time_spent", i18n(unknownContent));
        break;
      case SetiClientMonitor::Loading:
        {
          const double loaded = monitor->percentLoaded();
          setFieldContentText("ProgressView::time_spent", i18n("(loading - %1%)").arg(int(loaded)));
        }
        break;
      default:
        setFieldContentText("ProgressView::time_spent", i18n("(idle)"));
        break;
    }

    setFieldContentText("ProgressView::time_total", i18n(unknownContent));
    setFieldContentText("ProgressView::speed", i18n(unknownContent));
    setFieldContentText("ProgressView::time_left", i18n(unknownContent));
    setFieldContentText("ProgressView::rate", i18n(unknownContent));
    setFieldContentText("ProgressView::time_done", i18n(unknownContent));
    setFieldContentText("ProgressView::efficiency", i18n(unknownContent));
    setFieldContentText("ProgressView::drift", i18n(unknownContent));

    setFieldContentText("ProgressView::completed", i18n(unknownContent));
    setFieldToolTip("ProgressView::completed", QString::null);

    setFieldContentText("ProgressView::FFT", i18n(unknownContent));

    progressBarContainer->raiseWidget(0);
  }

  updateGeometry();
}

