/***************************************************************************
                          performanceitemreader.cpp  -  description
                             -------------------
    begin                : Sat Jun 9 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>

#include "performanceitemreader.h"

PerformanceItemReader::PerformanceItemReader(QListView *list) : list(list)
{
}

PerformanceItemReader::~PerformanceItemReader()
{
}

bool PerformanceItemReader::startElement(const QString &, const QString &,
                                         const QString &qName, const QXmlAttributes &attributes)
{
  if(qName == "entry")
  {
    // initialize data
    data.name = data.cache = QString::null;
    data.min = data.max = 1.0;
    for(int i = 0; i < N_COEFF; i++)
      data.coeff[i] = 0.0;
  }
  else if(qName == "name")
  {
    // read core type
    core = attributes.value("core");
    // read cpu number
    cpus = parseCPUs(attributes.value("cpus"));
    // read number of clients per CPU (used for SMT-capable cpus)
    clients = parseClients(attributes.value("client-ratio"));
    // read mode (for cpus emulating other instruction sets)
    mode = parseMode(attributes.value("mode"));
    // read memory type
    memory = attributes.value("memory");
  }
  else if(qName == "cache")
  {
    // read cache level
    QString levelStr = attributes.value("level");
    level = levelStr.isEmpty() ? 2 : levelStr.toInt();
    // read size unit
    QString unitStr = attributes.value("unit");
    unit = unitStr.isEmpty() ? QString("KB") : unitStr.upper();
  }
  else if(qName == "coeff")
  {
    // read degree of the polynomial coefficient
    QString degStr = attributes.value("deg");
    deg = degStr.isEmpty() ? 0 : degStr.toInt();
  }
  return(true);
}

bool PerformanceItemReader::endElement(const QString &, const QString &, const QString &qName)
{
  if(qName == "entry")
  {
    data.min = (data.min > data.max) ? data.max : data.min;
    new PerformanceItem(list, data);
  }
  else if(qName == "name")
  {
    QString ascription = QString::null;

    if(!cpus.isEmpty())
      ascription = cpus;
    if(!clients.isEmpty())
      if(ascription.isEmpty())
        ascription = clients;
      else
        ascription += " - " + clients;
    if(!mode.isEmpty())
      if(ascription.isEmpty())
        ascription = mode;
      else
        ascription += " - " + mode;
    if(!memory.isEmpty())
      if(ascription.isEmpty())
        ascription = memory;
      else
        ascription += " - " + memory;

    data.name = cdata;
    if(!core.isEmpty())
      data.name += " " + core;
    if(!ascription.isEmpty())
            data.name += " (" + ascription + ")";
  }
  else if(qName == "cache")
  {
    if(level == 2)
      data.cache = i18n("%1 %2").arg(cdata).arg(unit);
  }
  else if(qName == "min")
  {
    data.min = cdata.toDouble();
    data.min = (data.min < 1.0 || data.min > 20.0) ? 1.0 : data.min;
  }
  else if(qName == "max")
  {
    data.max = cdata.toDouble();
    data.max = (data.max < 1.0 || data.max > 20.0) ? 1.0 : data.max;
  }
  else if(qName == "coeff")
  {
    bool parseOK;
    double coeffDouble = cdata.toDouble(&parseOK);
    if(parseOK && deg >= 0 && deg < N_COEFF)
      data.coeff[deg] = coeffDouble;
  }
  return(true);
}

bool PerformanceItemReader::characters(const QString &cdata)
{
  this->cdata = cdata;
  return(true);
}

QString PerformanceItemReader::parseCPUs(const QString &cpusStr)
{
  bool parseOK;
  int cpusInt = cpusStr.toInt(&parseOK);

  cpusInt = (!parseOK || cpusInt < 1) ? 1 : cpusInt;

  switch(cpusInt) {
    case 1:
      return(QString::null);
    case 2:
      return(i18n("dual CPUs"));
    case 4:
      return(i18n("quad CPUs"));
    default:
      return(i18n("%1 CPUs").arg(cpusInt));
  }
  return(QString::null);
}

QString PerformanceItemReader::parseClients(const QString &clientsStr)
{
  bool parseOK;
  int clientsInt = clientsStr.toInt(&parseOK);

  clientsInt = (!parseOK || clientsInt < 1) ? 1 : clientsInt;

  switch(clientsInt) {
    case 1:
      return(QString::null);
    case 2:
      return(i18n("dual clients"));
    case 4:
      return(i18n("quad clients"));
    default:
      return(i18n("%1 clients").arg(clientsInt));
  }
  return(QString::null);
}

QString PerformanceItemReader::parseMode(const QString &modeStr)
{
  return(modeStr.isEmpty() ? modeStr : i18n("%1 mode").arg(modeStr));
}

