/***************************************************************************
                          performancedialog.cpp  -  description
                             -------------------
    begin                : Wed May 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qclipboard.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qvaluelist.h>
#include <qxml.h>

#include <kapplication.h>
#include <kcombobox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstatusbar.h>
#include <kstandarddirs.h>

#include "performanceitemreader.h"

#include "performancedialog.h"

const QString ArrowIcon = "arrow";

const QString PerfCompXML = "ksetispy/xml/perfcomp.xml";

PerformanceDialog::PerformanceDialog(QWidget *parent, const char *name)
                  : KSetiSpyView(Text, parent, name, WStyle_Customize | WStyle_DialogBorder | WStyle_Dialog)
{
  setCaption(i18n("Peak Processing Efficiency"));

  QBoxLayout *layout = new QBoxLayout(this, QBoxLayout::TopToBottom);
  layout->setSpacing(8);

  KListView *performanceItemList = new KListView(this, "PerformanceDialog::list");
  layout->addWidget(performanceItemList, 1);

  performanceItemList->addColumn(i18n("Processor"), 220);
  performanceItemList->addColumn(i18n("L2 Cache"));
  performanceItemList->addColumn(i18n("CpF*"));
  performanceItemList->setSorting(2, true);
  performanceItemList->setShowSortIndicator(true);
  performanceItemList->setSelectionMode(QListView::NoSelection);

  connect(performanceItemList, SIGNAL(contextMenu(KListView *, QListViewItem *, const QPoint &)),
          this, SLOT(handleContextMenu(KListView*, QListViewItem*, const QPoint &)));

  PerformanceItemReader handler(performanceItemList);
  QFile xmlFile(kapp->dirs()->findResource("data", PerfCompXML));
  QXmlInputSource source(xmlFile);
  QXmlSimpleReader reader;
  reader.setContentHandler(&handler);
  reader.parse(&source);

  QBoxLayout *hbox = new QBoxLayout(layout, QBoxLayout::LeftToRight);
  hbox->setMargin(5);
  hbox->setSpacing(8);

  QLabel *label = new QLabel(i18n("CPU/RAM speed ratio (for CpF estimate only):"), this);
  label->installEventFilter(this);
  hbox->addWidget(label);

  KLocale *locale = KGlobal::locale();

  KComboBox *multiplierMenu = new KComboBox(this, "PerformanceDialog::multiplier");
  multiplierMenu->installEventFilter(this);
  for (double d = 1.0; d < 20.1; d += 0.1)
    multiplierMenu->insertItem(locale->formatNumber(d, 1));
  hbox->addWidget(multiplierMenu);
  connect(multiplierMenu, SIGNAL(activated(const QString &)),
          this, SLOT(handleMultiplierMenu(const QString &)));

  hbox->addStretch(1);

  KStatusBar *status = new KStatusBar(this);
  status->installEventFilter(this);
  status->insertItem(i18n("*CPU Cycles/FLOP (lower is better)"), 1, 1, true);
  status->setItemAlignment(1, AlignLeft | AlignVCenter);
  layout->addWidget(status);

  instant = false;
  old.wu_name = current.wu_name = QString::null;
  thisItem = NULL;
}

PerformanceDialog::~PerformanceDialog()
{
}

void PerformanceDialog::readConfig(bool readGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("Performance Dialog");

  if(readGeometry)
  {
    QWidget *dialog = this->topLevelWidget();
    const QRect rect = dialog->geometry();
    dialog->setGeometry(config->readRectEntry("Geometry", &rect));
  }

  KListView *performanceItemList = (KListView *) child("PerformanceDialog::list", "KListView");
  QValueList<int> l = config->readIntListEntry("Column widths");
  for(uint i = 0; i < l.count(); i++)
    performanceItemList->setColumnWidth(i, l[i]);

  KComboBox *multiplierMenu = (KComboBox *) child("PerformanceDialog::multiplier", "KComboBox");
  multiplierMenu->setCurrentItem(config->readNumEntry("Multiplier"));

}

void PerformanceDialog::saveConfig(bool saveGeometry)
{
  KConfig *config = kapp->config();

  config->setGroup("Performance Dialog");

  if(saveGeometry)
    config->writeEntry("Geometry", this->topLevelWidget()->geometry());

  KListView *performanceItemList = (KListView *) child("PerformanceDialog::list", "KListView");
  QValueList<int> l;
  l.append(performanceItemList->columnWidth(0));
  l.append(performanceItemList->columnWidth(1));
  l.append(performanceItemList->columnWidth(2));
  config->writeEntry("Column widths", l);

  KComboBox *multiplierMenu = (KComboBox *) child("PerformanceDialog::multiplier", "KComboBox");
  config->writeEntry("Multiplier", multiplierMenu->currentItem());
}

bool PerformanceDialog::isInstant() const
{
  return(instant);
}

void PerformanceDialog::setInstant(bool set)
{
  instant = set;
  updateContent();
}

void PerformanceDialog::showEvent(QShowEvent *)
{
  updateContent();
}

void PerformanceDialog::updateContent()
{
  const SetiClientMonitor::State state = kdoc->setiMonitorState();
  SetiClientMonitor *monitor = kdoc->setiMonitor();

  if(state >= SetiClientMonitor::Idle)
  {
    const seti_data *data = monitor->setiData();

    if(data->wu.name != current.wu_name || data->state.progress < current.progress) {
      old.wu_name = QString::null;
      current.wu_name = data->wu.name;
      current.cpu = data->state.cpu;
      current.progress = data->state.progress;
    } else if(data->state.progress > current.progress) {
      old = current;
      current.cpu = data->state.cpu;
      current.progress = data->state.progress;
    }
  }

  if(!this->isVisible())
    return;

  const sys_info *info = (monitor != NULL) ? monitor->sysInfo() : NULL;

  KListView *performanceItemList = (KListView *) child("PerformanceDialog::list", "KListView");

  if(thisItem != NULL)
  {
    performanceItemList->takeItem(thisItem);
    delete thisItem;
    thisItem = NULL;
  }

  if(info != NULL && state == SetiClientMonitor::Running && !(instant && old.wu_name.isEmpty()))
  {
    const seti_data *data = monitor->setiData();
    const double teraFLOPs = SetiDataMonitor::teraFLOPs(*data);
    const int cpu = monitor->currentCPU();

    const double secs = instant ? (current.cpu - old.cpu) : current.cpu;
    const double done = instant ? (current.progress - old.progress) : current.progress;

    const double efficiency = (done > 0) ? (info->cpus[cpu].MHz * secs) / (1e6 * teraFLOPs * done) : 0.0;

    performanceItemStruct thisData;

    thisData.name = instant ? i18n("This processor (instantaneous)") : i18n("This processor (average)");
    thisData.cache = (info->cpus[cpu].cpuFamily > 5) ? SysInfoMonitor::kbToString(info->cpus[cpu].cacheSize)
                                                     : i18n("Unknown");
    thisData.min = thisData.max = 1.0;
    thisData.coeff[0] = efficiency;
    for(int i = 1; i < N_COEFF; i++)
      thisData.coeff[i] = 0.0;

    thisItem = new PerformanceItem(performanceItemList, thisData);
    thisItem->setPixmap(0, KIconLoader().loadIcon(ArrowIcon, KIcon::User));
  }
}

void PerformanceDialog::handleContextMenu(KListView *, QListViewItem *, const QPoint& p)
{
  popup->popup(p);
}

void PerformanceDialog::handleMultiplierMenu(const QString &string)
{
  double multiplier = string.toDouble();
  KListView *performanceItemList = (KListView *) child("PerformanceDialog::list", "KListView");
  PerformanceItem *item = (PerformanceItem *) performanceItemList->firstChild();

  if(item != NULL)
    do item->setMultiplier(multiplier);
    while((item = (PerformanceItem *) item->nextSibling()) != NULL);
  performanceItemList->sort();
}

void PerformanceDialog::textToClipboard()
{
  QString out;

  KListView *performanceItemList = (KListView *) child("PerformanceDialog::list", "KListView");

  for(int i = 0; i < performanceItemList->columns(); i++)
  {
    if(i > 0) out += "\t";
    out += performanceItemList->columnText(i);
  }
  out+= "\n";

  for(QListViewItem *performanceItem = performanceItemList->firstChild();
      performanceItem != NULL;
      performanceItem = performanceItem->nextSibling())
  {
    for(int i = 0; i < performanceItemList->columns(); i++)
    {
      if(i > 0) out += "\t";
      out += performanceItem->text(i);
    }
    out+= "\n";
  }

  KApplication::clipboard()->setText(out);
}

#include "performancedialog.moc"

