/***************************************************************************
                          ksetispydoc.cpp  -  description
                             -------------------
    begin                : Mon Jun 18 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qregexp.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <krfcdate.h>
#include <kdeversion.h>

#include "ksetispydoc.h"

const QString LinuxSysInfoURL = "/proc/";

const statsURLStruct defaultStatsURLs[N_STATS_URLS] =
{
  {false, "setiathome2.ssl.berkeley.edu", "/fcgi-bin/fcgi", "user_stats", ""},
  {false, "setiathome2.ssl.berkeley.edu", "/fcgi-bin/fcgi", "user_stats_new", ""},
  {false, "setiathome2.ssl.berkeley.edu", "/fcgi-bin/fcgi", "team_lookup", ""},
  {false, "setiathome2.ssl.berkeley.edu", "/fcgi-bin/fcgi", "team_lookup", ""},
  {false, "setiathome2.ssl.berkeley.edu", "/cgi-bin/cgi2", "print_cert", ""}
};

const int N_CERTIFICATES = 13;

const int Certificates[N_CERTIFICATES] =
{ 100, 250, 500, 750, 1000, 2500, 5000, 7500, 10000, 25000, 50000, 75000, 100000 };

KSetiSpyDoc::KSetiSpyDoc(QObject *parent, const char *name)
            : DCOPObject("Profiles-Interface")
            , QObject(parent, name)
{
  monitors.setAutoDelete(false);
  current = QString::null;

  defaultSysInfoURL = LinuxSysInfoURL;

  SetiClientMonitor::loadCalibrationPresets();

  cacheManager = new CacheManager(this);
  connect(cacheManager, SIGNAL(notify(int, int)), this, SLOT(handleCacheNotify(int, int)));

  userStats.email = QString::null;
  groupStats.email = groupStatsHistory[0].email = groupStatsHistory[1].email = QString::null;  

  statsStartup = statsUpload = false;
  statsInterval = 0;
  statsTimer = new QTimer(this);
  connect(statsTimer, SIGNAL(timeout()), this, SLOT(updateStats()));

  for(int i = 0; i < N_STATS_URLS; i++)
    statsURLs[i] = defaultStatsURLs[i];

  // save config every hour
  configTimer = new QTimer(this);
  configTimer->start(60 * 60 * 1000);
  connect(configTimer, SIGNAL(timeout()), this, SLOT(saveConfig()));
}

KSetiSpyDoc::~KSetiSpyDoc()
{
  for(QStringList::Iterator name = monitor_names.begin(); name != monitor_names.end(); name++)
  {
    SetiClientMonitor *monitor = monitors.find(*name);
    if(monitor->killOnExit())
      monitor->kill();
  }
}

SetiClientMonitor *KSetiSpyDoc::setiMonitor()
{
  return(current.isEmpty() ? NULL : monitors.find(current));
}

SetiClientMonitor::State KSetiSpyDoc::setiMonitorState()
{
  return(current.isEmpty() ? SetiClientMonitor::No_Data : monitors.find(current)->currentState());
}

QStringList KSetiSpyDoc::profiles()
{
  return(monitor_names);
}

QString KSetiSpyDoc::currentProfile()
{
  return(current);
}

profileInfoStruct KSetiSpyDoc::getProfile(const QString& name)
{
  profileInfoStruct out;

  out.name = name;

  SetiClientMonitor *monitor = name.isEmpty() ? NULL : monitors.find(name);
  if(monitor != NULL)
  {
    out.url = monitor->url();
    out.email = monitor->defaultEMail();
    out.sysInfoURL = monitor->sysInfoURL();
    out.timeout = monitor->clientTimeOut();
    out.interval = monitor->getInterval();

    out.clientURL.regular = monitor->command();
    out.clientURL.VLAR = monitor->command(SetiClientMonitor::VLAR);
    out.clientArgs = monitor->arguments();

    out.launchOnStartup = monitor->launchOnStartup();
    out.killOnExit = monitor->killOnExit();
    out.keepAlive = monitor->keepAlive();

    out.usesCache = monitor->usesCache();

    out.usesSETISpy = monitor->usesSETISpyLog();
    out.writesSETISpy = monitor->writesSETISpyLog();
    out.writesCSV = monitor->writesCSVLog();
    out.logURL = monitor->logURL();

    out.bestGaussian = monitor->bestGaussianLog();
    out.returnedGaussians = monitor->returnedGaussiansLog();

    out.calibration = monitor->calibration();
  }
  else
  {
    // not found: set some reasonable defaults
    out.url = QString::null;
    out.email = cacheManager->getEMail();
    out.sysInfoURL = defaultSysInfoURL;
    out.timeout = 2*60;
    out.interval = 30;

    out.clientURL.regular = out.clientURL.VLAR = cacheManager->getClientURL();
    out.clientArgs = QString::null;

    out.launchOnStartup = true;
    out.killOnExit = out.keepAlive = false;

    out.usesCache = false;

    out.usesSETISpy = true;
    out.writesSETISpy = out.writesCSV = false;

    const imageLogStruct defaultImageLog =
    {
      imageLogStruct::None,
      0.0,
      imageLogStruct::PNG,
      imageLogStruct::Default,
      QString::null
    };
    out.bestGaussian = out.returnedGaussians = defaultImageLog;

    out.calibration = defaultCalibration();
  }

  return(out);
}

QString KSetiSpyDoc::getProfileName(const SetiClientMonitor *monitor)
{
  for(QStringList::Iterator name = monitor_names.begin(); name != monitor_names.end(); name++)
  {
    SetiClientMonitor *current = monitors.find(*name);
    if(monitor == current)
      return(*name);
  }
  return(QString::null);
}

calibrationDataStruct KSetiSpyDoc::defaultCalibration()
{
  calibrationDataStruct out;

  out.cpu = QString::null;

  const double low_defaults[7] = {5.0, 25.0, 45.0, 50.0, 85.0, 95.0, 100.0};
  out.low.clear();
  for(uint i = 0; i < 7; i++)
  {
    calibrationItemStruct item;

    item.reported = item.actual = low_defaults[i] * 1e-2;
    out.low += item;
  }

  const double medium_defaults[7] = {5.0, 15.0, 45.0, 50.0, 80.0, 90.0, 100.0};
  out.medium.clear();
  for(uint i = 0; i < 7; i++)
  {
    calibrationItemStruct item;

    item.reported = item.actual = medium_defaults[i] * 1e-2;
    out.medium += item;
  }

  const double high_defaults[7] = {1.0, 10.0, 45.0, 50.0, 90.0, 95.0, 100.0};
  out.high.clear();
  for(uint i = 0; i < 7; i++)
  {
    calibrationItemStruct item;

    item.reported = item.actual = high_defaults[i] * 1e-2;
    out.high += item;
  }

  return(out);
}

CacheManager *KSetiSpyDoc::getCacheManager() const
{
  return(cacheManager);
}

bool KSetiSpyDoc::statsAutoStartup() const
{
  return(statsStartup);
}

int KSetiSpyDoc::statsTimerInterval() const
{
  return(statsInterval);
}

bool KSetiSpyDoc::statsAutoUpload() const
{
  return(statsUpload);
}

statsURLStruct KSetiSpyDoc::statsURL(StatsURLType type) const
{
  return statsURLs[type];
}

QString KSetiSpyDoc::userStatsServer() const
{
  return statsURLs[UserData].host;
}

QString KSetiSpyDoc::groupStatsServer() const
{
  return statsURLs[GroupData].host;
}

QString KSetiSpyDoc::userDataURL(const QString& email) const
{
  if(email.isEmpty())
    return QString::null;

  if(statsURLs[UserData].custom)
    return statsURLs[UserData].URL;

  const QString command = QString("http://%1%2?cmd=%3").arg(statsURLs[UserData].host)
                                                       .arg(statsURLs[UserData].path)
                                                       .arg(statsURLs[UserData].command);
  const QString args = QString("&email=%1").arg(KURL::encode_string(email));

  return (command + args);
}

QString KSetiSpyDoc::userStatsURL(const QString& email) const
{
  if(email.isEmpty())
    return QString::null;

  if(statsURLs[UserLink].custom)
    return statsURLs[UserLink].URL;

  const QString command = QString("http://%1%2?cmd=%3").arg(statsURLs[UserLink].host)
                                                       .arg(statsURLs[UserLink].path)
                                                       .arg(statsURLs[UserLink].command);
  const QString args = QString("&email=%1").arg(KURL::encode_string(email));

  return (command + args);
}

QString KSetiSpyDoc::groupDataURL(const QString& group) const
{
  if(group.isEmpty())
    return QString::null;

  if(statsURLs[GroupData].custom)
    return statsURLs[GroupData].URL;

  const QString command = QString("http://%1%2?cmd=%3").arg(statsURLs[GroupData].host)
                                                       .arg(statsURLs[GroupData].path)
                                                       .arg(statsURLs[GroupData].command);
  const QString args = QString("&name=%1").arg(KURL::encode_string(group));

  return (command + args);
}

QString KSetiSpyDoc::groupStatsURL(const QString& group) const
{
  if(group.isEmpty())
    return QString::null;

  if(statsURLs[GroupLink].custom)
    return statsURLs[GroupLink].URL;

  const QString command = QString("http://%1%2?cmd=%3").arg(statsURLs[GroupLink].host)
                                                       .arg(statsURLs[GroupLink].path)
                                                       .arg(statsURLs[GroupLink].command);
  const QString args = QString("&name=%1").arg(KURL::encode_string(group));

  return (command + args);
}

QString KSetiSpyDoc::certificateURL(const QString& email, int wus) const
{
  if(email.isEmpty())
    return QString::null;

  if(statsURLs[CertificateLink].custom)
    return statsURLs[CertificateLink].URL;

  int i;

  for(i = 0; i < N_CERTIFICATES; i++)
    if(Certificates[i] > wus) break;

  if(i == 0)
    return QString::null;

  const QString command = QString("http://%1%2?cmd=%3").arg(statsURLs[CertificateLink].host)
                                                       .arg(statsURLs[CertificateLink].path)
                                                       .arg(statsURLs[CertificateLink].command);
  const QString args = QString("&certnum=%1&email=%2&size=0").arg(Certificates[i-1])
                                                             .arg(KURL::encode_string(email));
  return(command + args);
}

QCStringList KSetiSpyDoc::functions()
{
  QCStringList functions = DCOPObject::functions();

  functions << "QString current()";
  functions << "void setCurrent(QString profile)";

  functions << "void start()";
  functions << "void stop()";

  return functions;
}

bool KSetiSpyDoc::process(const QCString& fun, const QByteArray& data,
                          QCString& replyType, QByteArray& replyData)
{
  if(fun == "current()") {
    replyType = "QString";
    QDataStream out(replyData, IO_WriteOnly);
    out << currentProfile();
  } else if(fun == "setCurrent(QString)") {
    QDataStream in(data, IO_ReadOnly);
    QString name;
    in >> name;

    replyType = "void";
    setCurrentProfile(name, true);
  } else if(fun == "start()") {
    replyType = "void";
    
    SetiClientMonitor *monitor = setiMonitor();
    if(monitor != NULL)
      monitor->exec();
  } else if(fun == "stop()") {
    replyType = "void";

    SetiClientMonitor *monitor = setiMonitor();
    if(monitor != NULL && monitor->isRunning())
      monitor->kill();
  } else
    return DCOPObject::process(fun, data, replyType, replyData);

  return true;
}

void KSetiSpyDoc::readConfig()
{
  KConfig *config = kapp->config();

  cacheManager->readConfig(config);

  config->setGroup("KSetiSpy");

  defaultSysInfoURL = config->readEntry("Default system info URL", LinuxSysInfoURL);

  const uint n_profiles = config->readUnsignedNumEntry("Profiles", 0);
  const QString profile = config->readEntry("Current profile");

  for(uint i = 0; i < n_profiles; i++)
  {
    profileInfoStruct profile;

    config->setGroup(QString("Profile %1").arg(i));

    profile.name = config->readEntry("Name");
    profile.url = config->readEntry("URL");
    profile.email = config->readEntry("E-mail");

    profile.sysInfoURL = config->readEntry("System info URL", defaultSysInfoURL);

    profile.clientURL.regular = config->readEntry("Client URL",
                                                  cacheManager->getClientURL().prettyURL(-1, KURL::StripFileProtocol));
    profile.clientURL.VLAR = config->readEntry("Client URL (VLAR)",
                                               profile.clientURL.regular.prettyURL(-1, KURL::StripFileProtocol));
    profile.clientArgs = config->readEntry("Client command line arguments");

    profile.timeout = config->readNumEntry("Timeout", 5 * 60);
    profile.interval = config->readNumEntry("Poll interval", 30);

    profile.launchOnStartup = config->readBoolEntry("Launch client on startup", true);
    profile.killOnExit = config->readBoolEntry("Kill client on exit", false);
    profile.keepAlive = config->readBoolEntry("Keep client alive", false);

    profile.usesCache = config->readBoolEntry("Use cache", false);

    profile.usesSETISpy = config->readBoolEntry("Use SETI Spy log format", true);
    profile.writesSETISpy = config->readBoolEntry("Record WUs in SETI Spy log", false);
    profile.writesCSV = config->readBoolEntry("Record WUs in SetiWatch log", false);
    profile.logURL = config->readEntry("Log file directory URL");

    profile.bestGaussian.filter = imageLogStruct::Filter(config->readNumEntry("Best gaussian log filter", 0));
    profile.bestGaussian.threshold = config->readDoubleNumEntry("Best gaussian log threshold", 2.0);
    profile.bestGaussian.format = imageLogStruct::Format(config->readNumEntry("Best gaussian log image format", 1));
    profile.bestGaussian.size = imageLogStruct::Size(config->readNumEntry("Best gaussian log image size", 0));
    profile.bestGaussian.url = config->readEntry("Best gaussian log URL",
                                                 profile.logURL.prettyURL(+1, KURL::StripFileProtocol));

    profile.returnedGaussians.filter = imageLogStruct::Filter(config->readNumEntry("Returned gaussians log filter", 0));
    profile.returnedGaussians.threshold = config->readDoubleNumEntry("Returned gaussians log threshold", 3.5);
    profile.returnedGaussians.format = imageLogStruct::Format(config->readNumEntry("Returned gaussians log image format", 1));
    profile.returnedGaussians.size = imageLogStruct::Size(config->readNumEntry("Returned gaussians log image size", 0));
    profile.returnedGaussians.url = config->readEntry("Returned gaussians log URL",
                                                      profile.logURL.prettyURL(+1, KURL::StripFileProtocol));

    calibrationItemStruct item;

    QStringList low = QStringList::split(" ", config->readEntry("Calibration (low)"));
    profile.calibration.low.clear();
    for(uint i = 0; i < low.count(); i++)
    {
      if(i % 2 == 0)
        item.reported = low[i].toDouble() * 1e-2;
      else {
        item.actual = low[i].toDouble() * 1e-2;
        profile.calibration.low += item;
      }
    }

    QStringList medium = QStringList::split(" ", config->readEntry("Calibration (medium)"));
    profile.calibration.medium.clear();
    for(uint i = 0; i < medium.count(); i++)
    {
      if(i % 2 == 0)
        item.reported = medium[i].toDouble() * 1e-2;
      else {
        item.actual = medium[i].toDouble() * 1e-2;
        profile.calibration.medium += item;
      }
    }

    QStringList high = QStringList::split(" ", config->readEntry("Calibration (high)"));
    profile.calibration.high.clear();
    for(uint i = 0; i < high.count(); i++)
    {
      if(i % 2 == 0)
        item.reported = high[i].toDouble() * 1e-2;
      else {
        item.actual = high[i].toDouble() * 1e-2;
        profile.calibration.high += item;
      }
    }

    if(!profile.name.isEmpty() && !profile.url.isEmpty())
      addProfile(profile, false);
  }

  setCurrentProfile(profile, false);

  config->setGroup("Last User Stats");

  userStats.name = config->readEntry("Name", "");
  userStats.email = config->readEntry("E-mail");
  userStats.group = config->readEntry("Group");
  userStats.n_results = config->readNumEntry("Total results");
  userStats.total_cpu = config->readDoubleNumEntry("Total CPU time");
  userStats.average_time = config->readDoubleNumEntry("Average time");
  userStats.last_result_time = config->readDateTimeEntry("Last result time");
  userStats.rank.pos = config->readNumEntry("User rank");
  userStats.rank.total = config->readNumEntry("Total users");
  userStats.n_rank = config->readNumEntry("Users with same rank");
  userStats.country = config->readEntry("Country");

  for(int i = 0; i < 2; i++)
  {
    config->setGroup(QString("Group Stats %1").arg(i));

    groupStatsHistory[i].email = config->readEntry("E-mail");
    groupStatsHistory[i].description = config->readEntry("Description");
    groupStatsHistory[i].founder = config->readEntry("Founder");
    groupStatsHistory[i].n_members = config->readNumEntry("Members");
    groupStatsHistory[i].n_results = config->readNumEntry("Results returned");
    groupStatsHistory[i].total_cpu = config->readDoubleNumEntry("CPU time");
    groupStatsHistory[i].timestamp = config->readDateTimeEntry("Timestamp");
  }

  config->setGroup("CGI Servers");

  for(int i = 0; i < N_STATS_URLS; i++)
  {
    QString name;

    name = QString("Use custom URL %1").arg(i);
    statsURLs[i].custom = config->readBoolEntry(name, defaultStatsURLs[i].custom);

    name = QString("Host %1").arg(i);
    statsURLs[i].host = config->readEntry(name, defaultStatsURLs[i].host);

    name = QString("Path %1").arg(i);
    statsURLs[i].path = config->readEntry(name, defaultStatsURLs[i].path);

    name = QString("Command %1").arg(i);
    statsURLs[i].command = config->readEntry(name, defaultStatsURLs[i].command);

    name = QString("URL %1").arg(i);
    statsURLs[i].URL = config->readEntry(name, defaultStatsURLs[i].URL);
  }

  emit updatedProfiles();
  emit updatedLog();
  emit updated();

  if(!userStats.email.isEmpty())
    emit updatedUserStats(&userStats);
  if(!groupStatsHistory[1].email.isEmpty())
    emit updatedGroupStats(groupStatsHistory, groupStatsHistory+1);
}

void KSetiSpyDoc::saveConfig()
{
  KConfig *config = kapp->config();

  cacheManager->saveConfig(config);

  config->setGroup("KSetiSpy");

  config->writeEntry("Default system info URL", defaultSysInfoURL);

  config->writeEntry("Profiles", monitor_names.count());
  config->writeEntry("Current profile", current);

  for(uint i = 0; i < monitor_names.count(); i++)
  {
    profileInfoStruct profile = getProfile(monitor_names[i]);

    config->setGroup(QString("Profile %1").arg(i));

    config->writeEntry("Name", profile.name);
    config->writeEntry("URL",
                       profile.url.prettyURL(+1, KURL::StripFileProtocol));
    config->writeEntry("E-mail", profile.email);

    config->writeEntry("System info URL",
                       profile.sysInfoURL.prettyURL(+1, KURL::StripFileProtocol));

    config->writeEntry("Client URL",
                       profile.clientURL.regular.prettyURL(-1, KURL::StripFileProtocol));
    config->writeEntry("Client URL (VLAR)",
                       profile.clientURL.VLAR.prettyURL(-1, KURL::StripFileProtocol));
    config->writeEntry("Client command line arguments", profile.clientArgs);

    config->writeEntry("Timeout", profile.timeout);
    config->writeEntry("Poll interval", profile.interval);

    config->writeEntry("Launch client on startup", profile.launchOnStartup);
    config->writeEntry("Kill client on exit", profile.killOnExit);
    config->writeEntry("Keep client alive", profile.keepAlive);

    config->writeEntry("Use cache", profile.usesCache);

    config->writeEntry("Use SETI Spy log format", profile.usesSETISpy);
    config->writeEntry("Record WUs in SETI Spy log", profile.writesSETISpy);
    config->writeEntry("Record WUs in SetiWatch log", profile.writesCSV);
    config->writeEntry("Log file directory URL",
                       profile.logURL.prettyURL(+1, KURL::StripFileProtocol));

    config->writeEntry("Best gaussian log filter", profile.bestGaussian.filter);
    config->writeEntry("Best gaussian log threshold", profile.bestGaussian.threshold);
    config->writeEntry("Best gaussian log image format", profile.bestGaussian.format);
    config->writeEntry("Best gaussian log image size", profile.bestGaussian.size);
    config->writeEntry("Best gaussian log URL",
                       profile.bestGaussian.url.prettyURL(+1, KURL::StripFileProtocol));

    config->writeEntry("Returned gaussians log filter", profile.returnedGaussians.filter);
    config->writeEntry("Returned gaussians log threshold", profile.returnedGaussians.threshold);
    config->writeEntry("Returned gaussians log image format", profile.returnedGaussians.format);
    config->writeEntry("Returned gaussians log image size", profile.returnedGaussians.size);
    config->writeEntry("Returned gaussians log URL",
                       profile.returnedGaussians.url.prettyURL(+1, KURL::StripFileProtocol));

    QString low;
    for(uint i = 0; i < profile.calibration.low.count(); i++)
    {
      low += " " + QString::number(profile.calibration.low[i].reported * 1e2, 'f', 2);
      low += " " + QString::number(profile.calibration.low[i].actual * 1e2, 'f', 2);
    }
    low.stripWhiteSpace();
    config->writeEntry("Calibration (low)", low);

    QString medium;
    for(uint i = 0; i < profile.calibration.medium.count(); i++)
    {
      medium += " " + QString::number(profile.calibration.medium[i].reported * 1e2, 'f', 2);
      medium += " " + QString::number(profile.calibration.medium[i].actual * 1e2, 'f', 2);
    }
    medium.stripWhiteSpace();
    config->writeEntry("Calibration (medium)", medium);

    QString high;
    for(uint i = 0; i < profile.calibration.high.count(); i++)
    {
      high += " " + QString::number(profile.calibration.high[i].reported * 1e2, 'f', 2);
      high += " " + QString::number(profile.calibration.high[i].actual * 1e2, 'f', 2);
    }
    high.stripWhiteSpace();
    config->writeEntry("Calibration (high)", high);
  }

  config->setGroup("Last User Stats");

  config->writeEntry("Name", userStats.name);
  config->writeEntry("E-mail", userStats.email);
  config->writeEntry("Group", userStats.group);
  config->writeEntry("Total results", userStats.n_results);
  config->writeEntry("Total CPU time", userStats.total_cpu);
  config->writeEntry("Average time", userStats.average_time);
  config->writeEntry("Last result time", userStats.last_result_time);
  config->writeEntry("User rank", userStats.rank.pos);
  config->writeEntry("Total users", userStats.rank.total);
  config->writeEntry("Users with same rank", userStats.n_rank);
  config->writeEntry("Country", userStats.country);

  for(int i = 0; i < 2; i++)
  {
    config->setGroup(QString("Group Stats %1").arg(i));

    config->writeEntry("E-mail", groupStatsHistory[i].email);
    config->writeEntry("Description", groupStatsHistory[i].description);
    config->writeEntry("Founder", groupStatsHistory[i].founder);
    config->writeEntry("Members", groupStatsHistory[i].n_members);
    config->writeEntry("Results returned", groupStatsHistory[i].n_results);
    config->writeEntry("CPU time", groupStatsHistory[i].total_cpu);
    config->writeEntry("Timestamp", groupStatsHistory[i].timestamp);
  }

  config->setGroup("CGI Servers");

  for(int i = 0; i < N_STATS_URLS; i++)
  {
    QString name;

    name = QString("Use custom URL %1").arg(i);
    config->writeEntry(name, statsURLs[i].custom);

    name = QString("Host %1").arg(i);
    config->writeEntry(name, statsURLs[i].host);

    name = QString("Path %1").arg(i);
    config->writeEntry(name, statsURLs[i].path);

    name = QString("Command %1").arg(i);
    config->writeEntry(name, statsURLs[i].command);

    name = QString("URL %1").arg(i);
    config->writeEntry(name, statsURLs[i].URL);
  }
}

void KSetiSpyDoc::setCurrentProfile(const QString& name, bool signal)
{
  if(name.isEmpty()) return;

  SetiClientMonitor *monitor = monitors.find(name);
  if(monitor != NULL)
  {
    current = name;
    if(signal)
    {
      emit updatedProfiles();
      emit updatedLog();
      emit updated();
    }
  }
}

void KSetiSpyDoc::setStatsAutoStartup(bool set)
{
  statsStartup = set;
}

void KSetiSpyDoc::setStatsTimerInterval(int hours)
{
  if(hours > 0)
  {
    statsInterval = hours;
    statsTimer->start(hours * 60 * 60 * 1000);
  }
  else
  {
    statsInterval = 0;
    statsTimer->stop();
  }
}

void KSetiSpyDoc::setStatsAutoUpload(bool set)
{
  statsUpload = set;
}

void KSetiSpyDoc::setStatsURL(StatsURLType type, const statsURLStruct& datum)
{
  if(type < 0 || type > N_STATS_URLS) return;

  statsURLs[type] = datum;
}

void KSetiSpyDoc::addProfile(const profileInfoStruct& profile, bool signal)
{
  if(profile.name.isEmpty() || profile.url.isEmpty())
    return;

  monitor_names.append(profile.name);
  monitor_names.sort();

  SetiClientMonitor *monitor = new SetiClientMonitor(profile.url, profile.sysInfoURL, profile.logURL, profile.usesSETISpy, this);

  monitor->setDefaultEMail(profile.usesCache ? cacheManager->getEMail() : profile.email);

  monitor->setClientTimeOut(profile.timeout);
  monitor->setInterval(profile.interval);

  monitor->setLaunchOnStartup(profile.launchOnStartup);
  monitor->setKillOnExit(profile.killOnExit);
  monitor->setKeepAlive(profile.keepAlive);

  monitor->setCommand(profile.clientURL.regular.path(-1));
  monitor->setCommand(profile.clientURL.VLAR.path(-1), SetiClientMonitor::VLAR);
  monitor->setArguments(profile.clientArgs);

  monitor->useCache(profile.usesCache);

  monitor->writeSETISpyLog(profile.writesSETISpy);
  monitor->writeCSVLog(profile.writesCSV);

  monitor->setBestGaussianLog(profile.bestGaussian);
  monitor->setReturnedGaussiansLog(profile.returnedGaussians);

  monitor->setCalibration(profile.calibration);

  monitors.insert(profile.name, monitor);

  connect(monitor, SIGNAL(updated(QObject *)), this, SLOT(handleSetiDataChange(QObject *)));
  connect(monitor, SIGNAL(logUpdated(QObject *)), this, SLOT(handleSetiLogChange(QObject *)));
  connect(monitor, SIGNAL(notify(QObject *, int)), this, SLOT(handleClientNotify(QObject *, int)));

  if(current.isEmpty())
    setCurrentProfile(profile.name, signal);
  else if(signal)
    emit updatedProfiles();

  if(monitor->launchOnStartup())
    monitor->exec();
}

void KSetiSpyDoc::changeProfile(const QString& name, const profileInfoStruct& profile, bool signal)
{
  if(name.isEmpty()) return;

  SetiClientMonitor *monitor = monitors.find(name);

  if(monitor == NULL || profile.name.isEmpty())
    return;

  if(name != profile.name)
  {
    monitor_names.remove(name);
    monitor_names.append(profile.name);

    monitors.remove(name);
    monitors.insert(profile.name, monitor);
    if(name == current) current = profile.name;
  }

  monitor->setDefaultEMail(profile.usesCache ? cacheManager->getEMail() : profile.email);

  monitor->setClientTimeOut(profile.timeout);
  monitor->setInterval(profile.interval);

  monitor->setLaunchOnStartup(profile.launchOnStartup);
  monitor->setKillOnExit(profile.killOnExit);
  monitor->setKeepAlive(profile.keepAlive);

  monitor->setCommand(profile.clientURL.regular.path(-1));
  monitor->setCommand(profile.clientURL.VLAR.path(-1), SetiClientMonitor::VLAR);
  monitor->setArguments(profile.clientArgs);

  monitor->useCache(profile.usesCache);

  monitor->useSETISpyLog(profile.usesSETISpy);
  monitor->writeSETISpyLog(profile.writesSETISpy);
  monitor->writeCSVLog(profile.writesCSV);
  monitor->setLogURL(profile.logURL);

  monitor->setBestGaussianLog(profile.bestGaussian);
  monitor->setReturnedGaussiansLog(profile.returnedGaussians);

  monitor->setCalibration(profile.calibration);

  if(name != profile.name && signal)
  {
    if(current == profile.name) {
      emit updatedLog();
      emit updated();
    }
    emit updatedProfiles();
  }
}

void KSetiSpyDoc::deleteProfile(const QString& name, bool signal)
{
  if(name.isEmpty()) return;

  bool isCurrent = name == current;
  monitor_names.remove(name);
  if(isCurrent)
    current = (monitor_names.count() > 0) ? monitor_names[0] : QString::null;

  SetiClientMonitor *monitor = monitors.find(name);
  if(monitor != NULL) {
    disconnect(monitor);
    delete monitor;
  }
  monitors.remove(name);

  if(signal)
  {
    if(isCurrent) {
      emit updatedLog();
      emit updated();
    }
    emit updatedProfiles();
  }
}

void KSetiSpyDoc::updateUserStats()
{
  SetiClientMonitor *monitor = setiMonitor();
  const seti_data *data = (monitor != NULL) ? monitor->setiData() : NULL;

  if(data != NULL && !data->user.email.isEmpty())
    userStats.email = data->user.email;
  else if (!cacheManager->getEMail().isEmpty())
    userStats.email = cacheManager->getEMail();
  else return;

  const KURL url(userDataURL(userStats.email));

  KIO::TransferJob *job = KIO::get(url, true, false);
  connect(job, SIGNAL(data(KIO::Job *, const QByteArray&)),
          this, SLOT(slotUserData(KIO::Job *, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job *)),
          this, SLOT(slotUserResult(KIO::Job *)));

  userCData = QString::null;
}

void KSetiSpyDoc::updateGroupStats()
{
  if(userStats.email.isEmpty() || userStats.group.isEmpty())
    return;

  groupStats.email = userStats.email;

  const KURL url(groupDataURL(userStats.group));

  KIO::TransferJob *job = KIO::get(url, true, false);
  connect(job, SIGNAL(data(KIO::Job *, const QByteArray&)),
          this, SLOT(slotGroupData(KIO::Job *, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job *)),
          this, SLOT(slotGroupResult(KIO::Job *)));

  groupCData = QString::null;
}

void KSetiSpyDoc::updateStats()
{
  updateUserStats();
  updateGroupStats();
}

void KSetiSpyDoc::handleSetiDataChange(QObject *object)
{
  if(!current.isEmpty() && object == monitors.find(current))
    emit updated();
}

void KSetiSpyDoc::handleSetiLogChange(QObject *object)
{
  if(!current.isEmpty() && object == monitors.find(current))
    emit updatedLog();
}

void KSetiSpyDoc::handleClientNotify(QObject *sender, int notify)
{
  const SetiClientMonitor *monitor = (SetiClientMonitor *) sender;
  const QString profile = getProfileName(monitor);
  QString event, message;

  switch(notify) {
    case SetiClientMonitor::WUCompleted:
      event = "WUCompleted";
      message = i18n("work unit completed");
      break;
    case SetiClientMonitor::InterestingGaussian:
      event = "InterestingGaussian";
      message = i18n("an interesting gaussian has been found");
      break;
    default:
      event = "Generic";
      message = i18n("generic notification #%1.").arg(notify);
  }

  #if KDE_IS_VERSION(3,2,0)
  KNotifyClient::event(qApp->mainWidget()->winId(), event,
                       i18n("Profile \"%1\": %2.").arg(message).arg(profile));
  #else
  KNotifyClient::event(event, i18n("Profile \"%1\": %2.").arg(message).arg(profile));
  #endif
}

void KSetiSpyDoc::handleCacheNotify(int notify, int count)
{
  QString event, message;

  switch(notify) {
    case CacheManager::WUsDownloaded:
      event = "WUsDownloaded";
      message = i18n("%1 new work units have been downloaded from the server").arg(count, 0);
      break;
    case CacheManager::ResultsUploaded:
      event = "ResultsUploaded";
      message = i18n("%1 results have been uploaded into the server").arg(count, 0);
      if(statsUpload)
        updateStats();
      break;
    default:
      event = "Generic";
      message = i18n("Generic notification #%1.").arg(notify);
  }

  #if KDE_IS_VERSION(3,2,0)
  KNotifyClient::event(qApp->mainWidget()->winId(), event, message);
  #else
  KNotifyClient::event(event, message);
  #endif
}

void KSetiSpyDoc::slotUserData(KIO::Job *, const QByteArray &data)
{
  userCData += QString(data);
}

void KSetiSpyDoc::slotGroupData(KIO::Job *job, const QByteArray &data)
{
  groupCData += QString(data);

  // abort the job after reading the first table
  if(groupCData.contains("</table>", false))
    job->kill(false);
}

void KSetiSpyDoc::slotUserResult(KIO::Job *job)
{

  if(!job->error())
  {
    const QRegExp anyTag("\\s*<[^>]*>\\s*");

    userCData.replace(QRegExp("&nbsp;"), " ");
    userCData.replace(QRegExp("<[Bb][Rr]>"), " ");

    {
      // parse the main table

      const QRegExp tableStartTag("\\s*<[Tt][Aa][Bb][Ll][Ee][^>]*>\\s*");
      const QRegExp tableEndTag("\\s*</[Tt][Aa][Bb][Ll][Ee][^>]*>\\s*");
      const QRegExp rowStartTag("\\s*<[Tt][Rr][^>]*>\\s*");
      const QRegExp headerStartTag("\\s*<[Tt][HDhd][^>]*>\\s*");
      const QRegExp datumStartTag("\\s*<[Tt][Dd][^>]*>\\s*");
      
      const int start = userCData.find(tableStartTag);
      if(start < 0) goto error;

      const int end = userCData.find(tableEndTag);
      if(end < 0 || end <= start) goto error;

      QString table = userCData.mid(uint(start), end-start);
      table.replace(tableStartTag, "");
      
      const QStringList rows = QStringList::split(rowStartTag, table);
      if(rows.count() < 2) goto error;

      QStringList headers = QStringList::split(headerStartTag, rows[0]);
      for(uint i = 0; i < headers.count(); i++)
        headers[i].replace(anyTag, "");

      QStringList data = QStringList::split(datumStartTag, rows[1]);
      for(uint i = 0; i < data.count(); i++)
        data[i].replace(anyTag, "");

      QStringList header;
      int pos;

      header = headers.grep(QRegExp("^Name"));
      pos = (header.count() > 0) ? headers.findIndex(header[0]) : -1;
      userStats.name = (pos >= 0 && pos < int(data.count())) ? data[pos] : QString(" ");

      header = headers.grep(QRegExp("^Results"));
      if(header.count() == 0) goto error;
      pos = headers.findIndex(header[0]);
      if(pos < 0 || pos >= int(data.count())) goto error;
      userStats.n_results = data[pos].toInt();

      header = headers.grep(QRegExp("^Total CPU"));
      if(header.count() == 0) goto error;
      pos = headers.findIndex(header[0]);
      if(pos < 0 || pos >= int(data.count())) goto error;
      userStats.total_cpu = convertTotalTime(data[pos]);

      header = headers.grep(QRegExp("^Average CPU"));
      if(header.count() == 0) goto error;
      pos = headers.findIndex(header[0]);
      if(pos < 0 || pos >= int(data.count())) goto error;
      userStats.average_time = convertAvgTime(data[pos]);

      header = headers.grep(QRegExp("^Country"));
      pos = (header.count() > 0) ? headers.findIndex(header[0]) : -1;
      userStats.country = (pos >= 0 && pos < int(data.count())) ? data[pos] : QString::null;
    }
    
    {
      // parse the remaining data

      const QStringList data = QStringList::split(anyTag, userCData);
      int pos = 0;

      pos = data.findIndex("Last result returned:");
      if(pos < 0 || pos+1 >= int(data.count())) goto error;
      userStats.last_result_time =  convertLastReturnedTime(data[pos+1]);

      pos = data.findIndex("You belong to the group named:");
      userStats.group = (pos >= 0 && pos+1 < int(data.count())) ? data[pos+1] : QString::null;

      pos = data.findIndex("Your rank out of");
      if(pos < 0 || pos+3 >= int(data.count())) goto error;
      userStats.rank.total = data[pos+1].toInt();
      userStats.rank.pos = data[pos+3].toInt();

      pos = data.findIndex("The number of users who have this rank:");
      if(pos < 0 || pos+1 >= int(data.count())) goto error;
      userStats.n_rank = data[pos+1].toInt();
    }

    emit updatedUserStats(&userStats);
    return;
  }

  error:
    emit updatedUserStats(NULL);
    return;
}

void KSetiSpyDoc::slotGroupResult(KIO::Job *job)
{
  int pos = groupCData.find("</table>", 0, false);

  if(!job->error() || pos >= 0)
  {
    groupCData = groupCData.left(uint(pos));
    groupCData.replace(QRegExp("&nbsp;"), " ");
    groupCData.replace(QRegExp("<br>"), " ");
    QStringList data = QStringList::split(QRegExp("\\s*<[^>]*>\\s*"), groupCData);

    pos = data.findIndex("Description");
    if(pos < 0 || pos >= int(data.count())-1)
      groupStats.description = QString::null;
    else
      groupStats.description = data[pos+1];

    pos = data.findIndex("Members");
    if(pos < 0 || pos >= int(data.count())-1) goto error;
    groupStats.n_members = data[pos+1].toInt();

    pos = data.findIndex("Results received");
    if(pos < 0 || pos >= int(data.count())-1) goto error;
    groupStats.n_results = data[pos+1].toInt();
    
    pos = data.findIndex("Total CPU time");
    if(pos < 0 || pos >= int(data.count())-1) goto error;
    groupStats.total_cpu = convertTotalTime(data[pos+1]);
    
    pos = data.findIndex("Founder");
    if(pos < 0 || pos >= int(data.count())-1) goto error;
    groupStats.founder = data[pos+1];

    groupStats.timestamp = QDateTime::currentDateTime();

    if(groupStats.email != groupStatsHistory[1].email)
      groupStatsHistory[0] = groupStatsHistory[1] = groupStats;
    else if (groupStatsHistory[1].timestamp.daysTo(groupStats.timestamp) >= 1)
    {
      groupStatsHistory[0] = groupStatsHistory[1];
      groupStatsHistory[1] = groupStats;
    }
    else if(groupStats.total_cpu == groupStatsHistory[0].total_cpu)
      groupStats.timestamp = groupStatsHistory[0].timestamp;

    emit updatedGroupStats(groupStatsHistory, &groupStats);
    return;
  }
  
  error:
    emit updatedGroupStats(NULL, NULL);
    return;
}

double KSetiSpyDoc::convertTotalTime(const QString& string)
{
  int pos;

  pos = string.find(" years");
  if(pos != -1)
    return(string.left(pos).toDouble() * 365 * 24 * 60 * 60);

  pos = string.find(" days");
  if(pos != -1)
    return(string.left(pos).toDouble() * 24 * 60 * 60);

  pos = string.find(" hr");
  if(pos != -1) {
    double out = string.left(pos).toDouble() * 60 * 60;
    QString tmp = string.mid(pos+3);

    pos = tmp.find(" min");
    if(pos != -1) {
      out += tmp.left(pos).toDouble() * 60;
      tmp = tmp.mid(pos+4);
    }

    pos = tmp.find(" sec");
    if(pos != -1)
      out += tmp.left(pos).toDouble();

    return(out);
  }

  return(0.0);
}

double KSetiSpyDoc::convertAvgTime(const QString& string)
{
  QString tmp = string;
  double out = 0.0;
  int pos;

  pos = tmp.find(" hr");
  if(pos != -1)
  {
    out += tmp.left(pos).toDouble() * 60 * 60;
    tmp = tmp.mid(pos+3);
  }
  pos = tmp.find(" min");
  if(pos != -1)
  {
    out += tmp.left(pos).toDouble() * 60;
    tmp = tmp.mid(pos+4);
  }
  pos = tmp.find(" sec");
  if(pos != -1)
    out += tmp.left(pos).toDouble();

  return(out);
}

QDateTime KSetiSpyDoc::convertLastReturnedTime(const QString& string)
{
  QDateTime out;

  QStringList list = QStringList::split(" ", string);
  if(list.count() != 6) return(out);

  const int year = list[4].toInt();

  const QString monthStr = list[1];
  int month;

  if(monthStr == "Jan") month = 1;
  else if(monthStr == "Feb") month = 2;
  else if(monthStr == "Mar") month = 3;
  else if(monthStr == "Apr") month = 4;
  else if(monthStr == "May") month = 5;
  else if(monthStr == "Jun") month = 6;
  else if(monthStr == "Jul") month = 7;
  else if(monthStr == "Aug") month = 8;
  else if(monthStr == "Sep") month = 9;
  else if(monthStr == "Oct") month = 10;
  else if(monthStr == "Nov") month = 11;
  else /* monthStr == "Dec" */ month = 12;

  int day = list[2].toInt();

  out.setDate(QDate(year, month, day));

  list = QStringList::split(":", list[3]);
  if(list.count() != 3) return(out);

  int hours = list[0].toInt();
  int minutes = list[1].toInt();
  int seconds = list[2].toInt();

  out.setTime(QTime(hours, minutes, seconds));
  out = out.addSecs(KRFCDate::localUTCOffset() * 60);

  return(out);
}

#include "ksetispydoc.moc"

