/***************************************************************************
                          datamonitor.cpp  -  description
                             -------------------
    begin                : Sun Jun 17 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <unistd.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtextstream.h>

#include <kio/job.h>

#include "datamonitor.h"

DataMonitor::DataMonitor(const KURL& url, QObject *parent, const char *name)
            : QObject(parent, name),
              baseURL(url), timer(new QTimer(this)), interval(0)
{
  baseURL.adjustPath(+1);

  connect(timer, SIGNAL(timeout()), this, SLOT(checkFiles()));
}

DataMonitor::~DataMonitor()
{
}

KURL DataMonitor::url() const
{
  return(baseURL);
}

int DataMonitor::getInterval() const
{
  return(interval);
}

void DataMonitor::setInterval(int secs)
{
  if(secs > 0) {
    interval = secs;
    if(url().isLocalFile()) timer->start(1000); // poll local files every sec.
    else timer->start(interval * 1000);
    checkFiles();
  } else {
    interval = 0;
    timer->stop();
  }
}

void DataMonitor::checkFiles()
{
  bool modified = false;

  for(int i = 0; i < int(files.count()); i++)
  {
    QDateTime backup = files[i].timestamp;

    checkFile(i, false);

    if(files[i].timestamp != backup)
      modified = true;
  }

  if(modified) updateData();
}

QStringList DataMonitor::readFile(const QString& fileName, bool& ok, const QString& delimiter)
{
  QFile file(fileName);
  QStringList out;

  ok = false;
  if(file.open(IO_ReadOnly))
  {
    const bool delimited = !delimiter.isEmpty();
    QTextStream text(&file);

    while (!text.atEnd())
    {
      QString line = text.readLine().stripWhiteSpace();

      if(delimited && line == delimiter)
        break;

      out.append(line);
    }

    file.close();
    ok = true;
  }

  return(out);
}

bool DataMonitor::findIntEntry(const QString& key, QStringList& list, const QString& sep, int& value)
{
  bool found = false;

  value = 0;
  for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    int pos = (*it).find(sep);

    if(pos == -1) continue;

    QString lhs = *it;
    lhs.truncate(pos);
    lhs = lhs.stripWhiteSpace();

    if(key != lhs) continue;

    found = true;

    QString rhs = *it;
    rhs.remove(0, pos + sep.length());

    value = rhs.toInt();

    list.remove(it);

    break;
  }

  return(found);
}

bool DataMonitor::findLongEntry(const QString& key, QStringList& list, const QString& sep, long& value)
{
  bool found = false;

  value = 0L;
  for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    int pos = (*it).find(sep);

    if(pos == -1) continue;

    QString lhs = *it;
    lhs.truncate(pos);
    lhs = lhs.stripWhiteSpace();

    if(key != lhs) continue;

    found = true;

    QString rhs = *it;
    rhs.remove(0, pos + sep.length());

    value = rhs.toLong();

    list.remove(it);

    break;
  }

  return(found);
}

bool DataMonitor::findFloatEntry(const QString& key, QStringList& list, const QString& sep, float& value)
{
  bool found = false;

  value = 0.0f;
  for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    int pos = (*it).find(sep);

    if(pos == -1) continue;

    QString lhs = *it;
    lhs.truncate(pos);
    lhs = lhs.stripWhiteSpace();

    if(key != lhs) continue;

    found = true;

    QString rhs = *it;
    rhs.remove(0, pos + sep.length());

    value = rhs.toFloat();

    list.remove(it);

    break;
  }

  return(found);
}

bool DataMonitor::findDoubleEntry(const QString& key, QStringList& list, const QString& sep, double& value)
{
  bool found = false;

  value = 0.0;
  for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    int pos = (*it).find(sep);

    if(pos == -1) continue;

    QString lhs = *it;
    lhs.truncate(pos);
    lhs = lhs.stripWhiteSpace();

    if(key != lhs) continue;

    found = true;

    QString rhs = *it;
    rhs.remove(0, pos + sep.length());

    value = rhs.toDouble();

    list.remove(it);

    break;
  }

  return(found);
}

bool DataMonitor::findStringEntry(const QString& key, QStringList& list, const QString& sep, QString& value)
{
  bool found = false;

  value = QString::null;
  for(QStringList::Iterator it = list.begin(); it != list.end(); ++it)
  {
    int pos = (*it).find(sep);

    if(pos == -1) continue;

    QString lhs = *it;
    lhs.truncate(pos);
    lhs = lhs.stripWhiteSpace();

    if(key != lhs) continue;

    found = true;

    QString rhs = *it;
    rhs.remove(0, pos + sep.length());

    value = rhs;

    list.remove(it);

    break;
  }

  return(found);
}

void DataMonitor::addFile(const QString& fileName, int& index)
{
  KURL fileURL(url(), fileName);

  index = files.count();

  data_info empty;

  empty.fileName = fileName;
  empty.exists = false;
  empty.ok = false;
  empty.job = NULL;
  empty.tmp = NULL;

  files += empty;

  checkFile(index, false);
}

int DataMonitor::index(const QString& fileName) const
{
  for(int i = 0; i < int(files.count()); i++)
    if(files[i].fileName == fileName)
      return(i);
  return(-1);
}

int DataMonitor::index(KIO::Job *job) const
{
  for(int i = 0; i < int(files.count()); i++)
    if(files[i].job == job)
      return(i);
  return(-1);
}

QString DataMonitor::fileName(int index)
{
  return((index < 0 || index > int(files.count())) ? QString::null : files[index].fileName);
}

bool DataMonitor::exists(int index)
{
  return((index < 0 || index > int(files.count())) ? false : files[index].exists);
}

bool DataMonitor::exists(const QString& fileName)
{
  return(exists(index(fileName)));
}

QDateTime DataMonitor::timestamp(int index)
{
  return((index < 0 || index > int(files.count())) ? QDateTime() : files[index].timestamp);
}

uint DataMonitor::size(const QString& fileName)
{
  return(size(index(fileName)));
}

uint DataMonitor::size(int index)
{
  return((index < 0 || index > int(files.count())) ? 0 : files[index].size);
}

QDateTime DataMonitor::timestamp(const QString& fileName)
{
  return(timestamp(index(fileName)));
}

bool DataMonitor::isOK(int index)
{
  return((index < 0 || index > int(files.count())) ? false : files[index].ok);
}

bool DataMonitor::isOK(const QString& fileName)
{
  return(isOK(index(fileName)));
}

void DataMonitor::checkFile(const QString& file)
{
  if(url().path(-1) == file || url().path(+1) == file)
    checkFiles();
  else {
    const QFileInfo fileInfo(file);
    const int i = index(fileInfo.fileName());
    if(i >= 0) checkFile(i, true);
  }
}

bool DataMonitor::parseFile(int, const QString&)
{
  return(false);
}

void DataMonitor::updateData()
{
}

void DataMonitor::checkFile(int index, bool signal)
{
  KURL fileURL(url(), files[index].fileName);

  if(fileURL.isLocalFile())
  {
    bool modified = false;

    data_info info = fileInfo(files[index].fileName);

    if(info.exists != files[index].exists) {
      modified = true;
    }
    if(info.exists && info.timestamp != files[index].timestamp)
      modified = true;

    if(info.exists)
      if(modified)
        info.ok = parseFile(index, fileURL.path(-1));
      else
        info.ok = files[index].ok;
    else
      info.ok = false;

    files[index] = info;
  
    if(modified && signal) updateData();
  }
  else if(files[index].job == NULL)
  {
    files[index].job = KIO::stat(fileURL, true, 4, false);
    
    connect(files[index].job, SIGNAL(result(KIO::Job *)),
            this, SLOT(statResult(KIO::Job *)));
  }
}

data_info DataMonitor::fileInfo(const QString& fileName)
{
  data_info out;
  out.fileName = fileName;

  KURL fileURL(url(), fileName);

  QFileInfo fileInfo(fileURL.path(-1));

  out.exists = fileInfo.exists();

  if(out.exists) {
    out.timestamp = fileInfo.lastModified();
    out.size = fileInfo.size();
  }

  return out;
}

void DataMonitor::statResult(KIO::Job *job)
{
  int index = this->index(job);
  if(index < 0) return;
  
  data_info file;
  bool modified = false;

  file.fileName = files[index].fileName;

  KIO::StatJob *statJob = (KIO::StatJob *) job;
  
  if(!job->error())
  {
    KIO::UDSEntry entry = statJob->statResult();

    file.exists = true;
    file.size = 0L;
    file.timestamp = QDateTime::currentDateTime();
    
    for(KIO::UDSEntry::iterator it = entry.begin(); it != entry.end(); ++it)
      switch((*it).m_uds) {
        case KIO::UDS_SIZE:
          file.size = (*it).m_long;
          break;
        case KIO::UDS_MODIFICATION_TIME:
          file.timestamp.setTime_t((*it).m_long);
          break;
        default:
          break;
      }
  }
  else
    file.exists = false;

  file.ok = file.exists && files[index].ok;

  if(file.exists != files[index].exists)
    modified = true;

  if(file.exists && (file.size != files[index].size))
    modified = true;

  if(file.exists && (file.timestamp != files[index].timestamp))
    modified = true;

  if(file.exists && modified)
  {
    file.tmp = new KTempFile();
    file.tmp->setAutoDelete(true);
    
    file.job = KIO::file_copy(statJob->url(), file.tmp->name(), -1, true, false, false);

    connect(file.job, SIGNAL(result(KIO::Job *)), this, SLOT(copyResult(KIO::Job *)));
  }
  else
    file.job = NULL;

  files[index] = file;

  if(file.job == NULL && modified)
    updateData();
}

void DataMonitor::copyResult(KIO::Job *job)
{
  int index = this->index(job);
  if(index < 0) return;

  if(!job->error())
  {
    KIO::FileCopyJob *copyJob = (KIO::FileCopyJob *) job;
    QString tmpFile = copyJob->destURL().path(-1);

    files[index].ok = parseFile(index, tmpFile);
  }
  else
    files[index].ok = false;

  files[index].job = NULL;
  delete files[index].tmp;

  updateData();
}

#include "datamonitor.moc"

