/***************************************************************************
                          querywidget.cpp  -  description
                             -------------------
    begin                : Thu May 22 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qtoolbutton.h>
#include <qtooltip.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "querywidget.h"
#include "kscopepixmaps.h"
#include "cscopefrontend.h"
#include "kscopeconfig.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
QueryWidget::QueryWidget(QWidget* pParent, const char* szName) :
	QueryWidgetLayout(pParent, szName),
	m_pHistPage(NULL),
	m_bHistEnabled(true),
	m_nQueryPages(0)
{
	QIconSet is;
	
	// Pages can be closed by clicking their tabs
	m_pQueryTabs->setHoverCloseButton(true);
	
	// Initialise the "new query" button
	m_pNewButton->setIconSet(
		Pixmaps().getPixmap(KScopePixmaps::ButtonNewQuery));
	QToolTip::add(m_pNewButton, i18n("New Query Page"));
	
	// Create a new query page when the "New Query" button is clicked
	connect(m_pNewButton, SIGNAL(clicked()), this, 
		SLOT(slotNewQueryPage()));
	
	// Create the icon set for the "lock query" button
	is.setPixmap(Pixmaps().getPixmap(KScopePixmaps::ButtonLockQuery),
		QIconSet::Automatic, QIconSet::Normal, QIconSet::Off);
	is.setPixmap(Pixmaps().getPixmap(KScopePixmaps::ButtonUnlockQuery),
		QIconSet::Automatic, QIconSet::Normal, QIconSet::On);
	
	// Initialise the "lock query" toggle button
	m_pLockButton->setIconSet(is);
	QToolTip::add(m_pLockButton, i18n("Lock/Unlock Page"));
	
	// Lock/unlock a query when the toggle button changes state
	connect(m_pLockButton, SIGNAL(toggled(bool)), this, 
		SLOT(slotLockCurrent(bool)));
	
	// Lock/unlock a query when the toggle button changes state
	connect(m_pQueryTabs, SIGNAL(currentChanged(QWidget*)), this, 
		SLOT(slotCurrentChanged(QWidget*)));
	
	// Initialise the "refresh query" button
	m_pRefreshButton->setIconSet(
		Pixmaps().getPixmap(KScopePixmaps::ButtonRefreshQuery));
	QToolTip::add(m_pRefreshButton, i18n("Refresh Query"));
	
	// Create a new query page when the "New Query" button is clicked
	connect(m_pRefreshButton, SIGNAL(clicked()), this, 
		SLOT(slotRefreshCurrent()));
	
	// Initialise the "close query button"
	m_pCloseButton->setIconSet(
		Pixmaps().getPixmap(KScopePixmaps::ButtonCloseQuery));
	QToolTip::add(m_pCloseButton, i18n("Close Page"));
	
	// Close a query when the close button is clicked
	connect(m_pCloseButton, SIGNAL(clicked()), this, 
		SLOT(slotCloseCurrent()));
	
	// Close query when its tab button is clicked
	connect(m_pQueryTabs, SIGNAL(closeRequest(QWidget*)), this,
		SLOT(slotClosePage(QWidget*)));
	
	// Hide the buttons, if not needed
	if (!Config().getShowQueryButtons())
		((QWidget*)m_pButtonGroup)->setShown(false);
	
	// Create a popup menu to control query page operations
	m_pPageMenu = new QPopupMenu(this);
	m_pPageMenu->insertItem(i18n("New Query Page"), this, 
		SLOT(slotNewQueryPage()));
	m_nLockMenuId = m_pPageMenu->insertItem(i18n("Lock/Unlock Page"), this, 
		SLOT(slotLockCurrent()));
	m_nRefreshMenuId = m_pPageMenu->insertItem(i18n("Refresh Query"), this, 
		SLOT(slotRefreshCurrent()));
	m_pPageMenu->insertSeparator();
	m_nCloseMenuId = m_pPageMenu->insertItem(i18n("Close Page"), this, 
		SLOT(slotCloseCurrent()));

	// Show the menu when requested
	connect(m_pQueryTabs, SIGNAL(contextMenu(const QPoint&)), this,
		SLOT(slotContextMenu(const QPoint&)));
	connect(m_pQueryTabs, SIGNAL(contextMenu(QWidget*, const QPoint&)), this,
		SLOT(slotContextMenu(QWidget*, const QPoint&)));
}

/**
 * Class destructor.
 */
QueryWidget::~QueryWidget()
{
}

/**
 * Runs a query in a query page.
 * A query page is first selected, with a new one created if required. The
 * method then creates a Cscope process and runs the query.
 * @param	nType	The query's numeric type code
 * @param	sText	The query's text, as entered by the user
 */
void QueryWidget::initQuery(uint nType, const QString& sText)
{
	QueryPage* pPage;
	CscopeFrontend* pCscope;
	
	// Make sure we have a query page
	findQueryPage();
	pPage = currentPage();
			
	// Use the current page, or a new page if the current one is locked
	if (pPage->isLocked()) {
		slotAddQueryPage();
		pPage = currentPage();
	}

	// Reset the page's results list
	pPage->clear();
	pPage->setQueryData(nType, sText);
	
	// Set the page's tab text according to the new query
	setPageCaption(pPage);
	
	// Create a self-detsructing Cscope process, and set this page as the
	// target for Cscope's output
	pCscope = new CscopeFrontend(true);
	pPage->attach(pCscope);
	
	// Run the query
	pCscope->query(nType, sText);
}

/**
 * Applies the user's colour and font preferences to all pages.
 */
void QueryWidget::applyPrefs()
{
	QueryPage* pPage;
	int nPages, i;

	// Iterate query pages
	nPages = m_pQueryTabs->count();
	for (i = 0; i < nPages; i++) {
		pPage = (QueryPage*)m_pQueryTabs->page(i);
		pPage->applyPrefs();
		setPageCaption(pPage);
	}
}

/**
 * Loads all queries that were locked when the project was closed.
 * @param	sProjPath		The full path of the project directory
 * @param	slQueryFiles	The list of query file names to load
 */
void QueryWidget::loadLockedQueries(const QString& sProjPath,
	const QStringList& slQueryFiles)
{
	QStringList::ConstIterator itr;
	QueryPage* pPage;
	QString sName;
	
	// Iterate through query files
	for (itr = slQueryFiles.begin(); itr != slQueryFiles.end(); ++itr) {
		// Set the target page, based on the file type (query or history)
		if ((*itr).startsWith("History")) {
			findHistoryPage();
			pPage = m_pHistPage;
		}
		else {
			findQueryPage();
			pPage = currentPage();
		}

		// Load a query file to this page, and lock the page
		if (pPage->load(sProjPath, *itr)) {
			setPageCaption(pPage);
			setPageLocked(pPage, true);
		}
	}
}

/**
 * Saves all locked queries into files in the project directory.
 * @param	sProjPath		The full path of the project directory
 * @param	slQueryFiles	Holds a list of query file names, upon return
 */
void QueryWidget::saveLockedQueries(const QString& sProjPath, 
	QStringList& slQueryFiles)
{
	int nPageCount, i;
	QueryPage* pPage;
	QString sFileName;
	
	// Iterate pages
	nPageCount = m_pQueryTabs->count();
	for (i = 0; i < nPageCount; i++) {
		pPage = (QueryPage*)m_pQueryTabs->page(i);
		if (pPage->isLocked()) {
			// Store this query page
			sFileName = pPage->store(sProjPath);
			
			// A successful store operation returns a non-empty file name
			if (!sFileName.isEmpty())
				slQueryFiles.append(sFileName);
		}
	}
}

/**
 * Shows/hides the query buttons.
 */
void QueryWidget::showHideButtons()
{
	bool bShow;
	
	bShow = !Config().getShowQueryButtons();
	((QWidget*)m_pButtonGroup)->setShown(bShow);
	Config().setShowQueryButtons(bShow);
}

/**
 * Adds a new position record to the active history page.
 * @param	sFile	The file path
 * @param	nLine	The line number
 * @param	sText	The contents of the line pointed to by the file path and
 * 					line number
 */
void QueryWidget::addHistoryRecord(const QString& sFile, uint nLine, 
	const QString& sText)
{
	// Validate file name and line number
	if (sFile.isEmpty() || nLine == 0)
		return;
	
	// Do nothing if history logging is disabled
	if (!m_bHistEnabled)
		return;

	// Make sure there is an active history page	
	findHistoryPage();
			
	// Add the position entry to the active page
	m_pHistPage->addRecord(sFile, nLine, sText);
}

/**
 * Sets the tab caption and tool-tip for the given page.
 * @param	pPage	The page whose tab needs to be changed
 */
void QueryWidget::setPageCaption(QueryPage* pPage)
{
	m_pQueryTabs->changeTab(pPage, 
		pPage->getCaption(Config().getUseBriefQueryCaptions()));
	m_pQueryTabs->setTabToolTip(pPage, pPage->getCaption());
}

/**
 * Creates a new query page, and adds it to the tab widget.
 * The new page is set as the current one.
 */
void QueryWidget::slotAddQueryPage()
{
	QueryPage* pPage;
	QString sTitle;

	// Create the page
	pPage = new QueryPage(this);

	// Add the page, and set it as the current one
	m_pQueryTabs->insertTab(pPage, 
		Pixmaps().getPixmap(KScopePixmaps::TabUnlocked), "Query",
		m_nQueryPages++);
	setCurrentPage(pPage);
	
	// Emit the lineRequested() signal when a query record is selected on 
	// this page
	connect(pPage, SIGNAL(recordSelected(QListViewItem*)), this, 
		SLOT(slotRequestLine(QListViewItem*)));
	
	// Emit the queryFinished() signal when all records have been added to
	// the page
	connect(pPage, SIGNAL(queryFinished()), this, SIGNAL(queryFinished()));
}

/**
 * Locks or unlocks a query.
 * This slot is connected to the toggled() signal of the lock query button.
 * @param	bOn	true if the new state of the button is "on", false if it is
 *				"off"
 */
void QueryWidget::slotLockCurrent(bool bOn)
{
	QueryPage* pPage;
	
	pPage = currentPage();
	
	if (pPage != NULL)
		setPageLocked(currentPage(), bOn);
}

/**
 * Locks or unlocks a query, by toggling the current state.
 */
void QueryWidget::slotLockCurrent()
{
	QueryPage* pPage;
	
	pPage = currentPage();
	if (pPage != NULL)
		setPageLocked(pPage, !pPage->isLocked());
}

/**
 * Reruns the query whose results are displayed in the current page.
 */
void QueryWidget::slotRefreshCurrent()
{
	QueryPage* pPage;
	CscopeFrontend* pCscope;
	uint nType;
	QString sText;
	
	// Make sure the current page is a valid, non-empty one
	pPage = currentPage();
	if ((pPage == NULL) || isHistoryPage(pPage) ||
		(pPage->getQueryType() == CscopeFrontend::None)) {
		return;
	}

	// Get the current page parameters (before they are deleted by clear())
	nType = pPage->getQueryType();
	sText = pPage->getQueryText();
	
	// Clear the current page contents
	pPage->clear();
	pPage->setQueryData(nType, sText);
	
	// Create a self-destructing Cscope process, and run it with the page's
	// parameters
	pCscope = new CscopeFrontend(this);	
	pPage->attach(pCscope);
	pCscope->query(nType, sText);
}

/**
 * Closes the current query page.
 */
void QueryWidget::slotCloseCurrent()
{
	QWidget* pPage;

	// Close the current page
	pPage = currentPage();
	if (pPage != NULL)
		slotClosePage(pPage);
}

/**
 * Closes all query pages.
 */
void QueryWidget::slotCloseAll()
{
	int nPageCount, i;
	QueryPage* pPage;
	
	// Close all pages
	nPageCount = m_pQueryTabs->count();
	for (i = 0; i < nPageCount; i++) {
		pPage = (QueryPage*)m_pQueryTabs->page(0);
		m_pQueryTabs->removePage(pPage);
		delete pPage;
	}
	
	m_pHistPage = NULL;
}

/**
 * Creates a new history page.
 */
void QueryWidget::slotAddHistoryPage()
{
	QString sTitle;
		
	// Create the page
	m_pHistPage = new HistoryPage(this);

	// Add the page, and set it as the current one
	m_pQueryTabs->insertTab(m_pHistPage, 
		Pixmaps().getPixmap(KScopePixmaps::TabUnlocked), "");
	setPageCaption(m_pHistPage);

	// Emit the lineRequested() signal when a query record is selected on 
	// this page
	connect(m_pHistPage, SIGNAL(recordSelected(QListViewItem*)), this, 
		SLOT(slotRequestLine(QListViewItem*)));
}

/**
 * Handles the "Go->Back" menu command.
 * Moves to the previous position in the position history.
 */
void QueryWidget::slotHistoryPrev()
{
	if (m_pHistPage != NULL) {
		m_bHistEnabled = false;
		m_pHistPage->selectPrev();
		m_bHistEnabled = true;
	}
}

/**
 * Handles the "Go->Forward" menu command.
 * Moves to the next position in the position history.
 */
void QueryWidget::slotHistoryNext()
{
	if (m_pHistPage != NULL) {
		m_bHistEnabled = false;
		m_pHistPage->selectNext();
		m_bHistEnabled = true;
	}
}

/**
 * Sets the active history page, if any, as the current page.
 */
void QueryWidget::selectActiveHistory()
{
	if (m_pHistPage)
		setCurrentPage(m_pHistPage);
}

/**
 * Emits a signal indicating a certain source file and line number are
 * requested.
 * This slot is connected to the recordSelected() signal emitted by any of
 * the query pages. The signal emitted by this slot is used to display an
 * editor page at the requested line.
 * @param	pItem	The list item selected by the user
 */
void QueryWidget::slotRequestLine(QListViewItem* pItem)
{
	QString sFileName, sLine;

	// Get the file name and line number
	sFileName = pItem->text(0);
	sLine = pItem->text(2);

	// Do not process the "No results" item
	if (sLine.isEmpty())
		return;
	
	// Disable history if the request came from the active history page
	if (currentPage() == m_pHistPage)
		m_bHistEnabled = false;
		
	// Emit the signal
	emit lineRequested(sFileName, sLine.toUInt());
	
	// Re-enable history
	if (currentPage() == m_pHistPage)
		m_bHistEnabled = true;
}

/**
 * Update the lock button when the current query page changes.
 * @param	pWidget	The new current page
 */
void QueryWidget::slotCurrentChanged(QWidget* pWidget)
{
	QueryPage* pPage;
	
	pPage = (QueryPage*)pWidget;
	m_pLockButton->setOn(pPage->isLocked());
}

/**
 * Removes the given page from the tab widget.
 * This slot is connected to the closeRequest() signal of the KTabBar object.
 * @param	pPage	The page to close
 */
void QueryWidget::slotClosePage(QWidget* pPage)
{
	// Prompt the user before closing a locked query
	if (((QueryPage*)pPage)->isLocked()) {
		if (KMessageBox::questionYesNo(NULL, i18n("You about about to close"
			" a locked page.\nAre you sure?")) == KMessageBox::No) {
			return;
		}
	}

	// Check if the closed page is the active history page
	if (pPage == m_pHistPage)
		m_pHistPage = NULL;
	// Update the number of open history pages
	else if (dynamic_cast<HistoryPage*>(pPage) == NULL)
		m_nQueryPages--;
	
	// Remove the page and delete the object
	m_pQueryTabs->removePage(pPage);
	delete pPage;
}

/**
 * Displays a context menu for page operations.
 * This slot is connected to the contextMenu() signal, emitted by 
 * m_pQueryTabs.
 * @param	pt	The point over which the mouse was clicked in request for the
 *				context menu
 */
void QueryWidget::slotContextMenu(const QPoint& pt)
{
	// Disable the lock and close operations (clicked outside any widget)
	m_pPageMenu->setItemEnabled(m_nLockMenuId, false);
	m_pPageMenu->setItemEnabled(m_nRefreshMenuId, false);
	m_pPageMenu->setItemEnabled(m_nCloseMenuId, false);
	
	// Show the menu
	m_pPageMenu->popup(pt);
}

/**
 * Displays a context menu for page operations.
 * This slot is connected to the contextMenu() signal, emitted by 
 * m_pQueryTabs.
 * @param	pWidget	The page under the mouse
 * @param	pt		The point over which the mouse was clicked in request for 
 *					the	context menu
 */
void QueryWidget::slotContextMenu(QWidget* pWidget, const QPoint& pt)
{
	// Operations are on the current page, so we must ensure the clicked
	// tab becomes the current one
	setCurrentPage(pWidget);
	
	// Enable the lock and close operations
	m_pPageMenu->setItemEnabled(m_nLockMenuId, true);
	m_pPageMenu->setItemEnabled(m_nRefreshMenuId,
		dynamic_cast<HistoryPage*>(pWidget) == NULL);
	m_pPageMenu->setItemEnabled(m_nCloseMenuId, true);
	
	// Show the menu
	m_pPageMenu->popup(pt);
}

/**
 * Locks/unlocks the give page.
 * @param	pPage	The page to lock or unlock
 * @param	bLock	true to lock the page, false to unlock it
 */
void QueryWidget::setPageLocked(QueryPage* pPage, bool bLock)
{
	// Set the locking state of the current page
	pPage->setLocked(bLock);
	m_pQueryTabs->setTabIconSet(pPage,
		bLock ? Pixmaps().getPixmap(KScopePixmaps::TabLocked) : 
		Pixmaps().getPixmap(KScopePixmaps::TabUnlocked));
		
	// There can only be one unlocked history page. Check if a non-active
	// query page is being unlocked
	if ((dynamic_cast<HistoryPage*>(pPage) != NULL) && (pPage != m_pHistPage)
		&& (!bLock)) {
		// Lock the active history page (may be NULL)
		if (m_pHistPage != NULL)
			setPageLocked(m_pHistPage, true);
		
		// Set the unlock page as the new active history page
		m_pHistPage = (HistoryPage*)pPage;
	}
}

/**
 * Ensures the current page is a query page that is ready to accept new
 * queries.
 * The function first checks the current page. If it is an unlocked query
 * page, then nothing needs to be done. Otherwise, it checks for the first
 * unlocked query page by iterating over all pages in the tab widget. If this
 * fails as well, a new query page is created.
 */
void QueryWidget::findQueryPage()
{
	QueryPage* pPage;
	HistoryPage* pHistPage;
	int nPages, i;
	
	// First check if the current page is an unlocked query page
	pPage = currentPage();
	if ((pPage != NULL) && 
		(pHistPage = dynamic_cast<HistoryPage*>(pPage)) == NULL) {
		if (!pPage->isLocked() && !pPage->isAttached())
			return;
	}
	
	// Look for the first unlocked query page
	nPages = m_pQueryTabs->count();
	for (i = 0; i < nPages; i++) {
		pPage = (QueryPage*)m_pQueryTabs->page(i);
		if ((pHistPage = dynamic_cast<HistoryPage*>(pPage)) == NULL) {
			if (!pPage->isLocked() && !pPage->isAttached()) {
				setCurrentPage(pPage);
				return;
			}
		}
	}

	// Couldn't find an unlocked query page, create a new one
	slotAddQueryPage();
}

/**
 * Ensures an active history page exists.
 * The active history page is the only unlocked history page. If one does not
 * exist, it is created.
 */
void QueryWidget::findHistoryPage()
{
	QueryPage* pPage;
	int nPages, i;
	
	// First check if the active history page is unlocked
	if (m_pHistPage != NULL && !m_pHistPage->isLocked())
		return;
	
	// Look for the first unlocked history page
	nPages = m_pQueryTabs->count();
	for (i = 0; i < nPages; i++) {
		pPage = (QueryPage*)m_pQueryTabs->page(i);
		if ((m_pHistPage = dynamic_cast<HistoryPage*>(pPage)) != NULL) {
			if (!pPage->isLocked())
				return;
		}
	}

	// Couldn't find an unlocked query page, create a new one
	slotAddHistoryPage();
}

/**
 * Creates a new query page, and emits signal about it.
 */
void QueryWidget::slotNewQueryPage()
{
	slotAddQueryPage();
	emit newQuery();
}
