/***************************************************************************
                          projectmanager.h  -  description
                             -------------------
    begin                : Sat May 10 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef PROJECTMANAGER_H
#define PROJECTMANAGER_H

#include <qobject.h>
#include <qstring.h>
#include <qdir.h>
#include <qmap.h>
#include <kconfig.h>

/**
 * Abstract base class for classes that need the list of project files.
 * Objects of classes derived from this one are used as a parameter to
 * ProjectManager::fillList(), which reads all file entries in the project,
 * and calls addItem() for each.
 * Any class that wishes to retrieve the project's file list, should inherit
 * from this class, and implement addItem().
 * @author Elad Lahav
 */

class FileListTarget
{
public:
	/**
	 * Class constructor.
	 */
	FileListTarget() {}

	/**
	 * Appends a file to the list.
	 * @param	sFilePath	The full path of the file to add
	 */
	virtual void addItem(const QString& sFilePath) = 0;
};

/**
 * Abstract base class for classes that need the list of project files.
 * Objects of classes derived from this one are used as a parameter to
 * ProjectManager::writeList(), which calls getFirstItem() and getNextItem(),
 * and writes the returned values to the project's 'cscope.files' file.
 * Any class that wishes to retrieve the project's file list, should inherit
 * from this class, and implement firstItem() and nextItem().
 * @author Elad Lahav
 */

class FileListSource
{
public:
	/**
	 * Class constructor.
	 */
	FileListSource() {}

	/**
	 * Returns the first file in the list, and initiates a new iteration.
	 * @param	sFilePath	Holds the path of the first file, upon return
	 * @return	true if there are more files, false otherwise
	 */
	virtual bool firstItem(QString& sFilePath) = 0;

	/**
	 * Returns the next file in the list.
	 * @param	sFilePath	Holds the path of the file, upon return
	 * @return	true if there are more files, false otherwise
	 */
	virtual bool nextItem(QString& sFilePath) = 0;
};

/**
 * Maintains a KScope project.
 * A project is a set of source files, on which to run Cscope, as well as
 * other programmes. The project is maintained using a set of configuration
 * files, located in a common directory. These files include:
 * - cscope.proj: A KDE configuration file that stores project parameters
 * - cscope.files: Holds the list of source files included in this project
 * To these cscope adds its symbol reference files.
 * The ProjectManager class is responsible for creating new projects, adding
 * and removing source files, and allowing other objects access to the
 * currently active project (such as filling a list object with the source 
 * files).
 * @author Elad Lahav
 */

class ProjectManager : public QObject
{
	Q_OBJECT
	
public: 
	ProjectManager();
	virtual ~ProjectManager();

	/**
	 * Configurable project options.
	 */
	struct Options {
		/** A list of MIME-types that determines which files are included in
			the project. */
		QStringList slFileTypes;
		
		/** true if the -k option for CScope should be used. */
		bool bKernel;
		
		/** true if Cscope should build an inverted index. */
		bool bInvIndex;
		
		/** The time, in milliseconds, after which the database should be
			automatically rebuilt (-1 if this option is disabled). */
		int nAutoRebuildTime;
		
		/** true to use auto-completion. */
		bool bACEnabled;
		
		/** Minimum number of characters in a symbol for auto-completion. */
		uint nACMinChars;
		
		/** Time interval, in milliseconds, before auto-completion is
			started. */
		uint nACDelay;
		
		/** Maximal number of entries for auto-completion. */
		uint nACMaxEntries;
	};
	
	bool create(const QString&, const QString&, const Options&);
	bool open(const QString&);
	bool openCscopeOut(const QString&);
	void close();
	int addFiles(const QStringList&);
	void fillList(FileListTarget*);
	void writeList(FileListSource*);
	
	bool isEmpty();
	QString getFileTypes() const;
	void getOptions(Options&) const;
	void setOptions(const Options&);
	void getOpenFiles(QStringList&) const;
	void setOpenFiles(QStringList&);
	void getQueryFiles(QStringList&) const;
	void setQueryFiles(QStringList&);
	void getCallTreeFiles(QStringList&) const;
	void setCallTreeFiles(QStringList&);
	QString getRoot() const;
	void setRoot(const QString&);
	void getSymHistory(QStringList&) const;
	void setSymHistory(QStringList&);
	bool dbExists();
	
	/**
	 * Determines whether a project is open, i.e., a project directory is
	 * associated with this object.
	 * @return	true if the project is open, false otherwise
	 */
	bool isOpen() { return m_bOpen; }

	/**
	 * Determines whether a project is based on a Cscope.out file, and is
	 * therefore considered as a temporary project.
	 * @return	true if this is a temporary project, false otherwise
	 */
	bool isTemporary() { return m_bTempProj; }

	/**
	 * @return	The name of the current project
	 */
	QString getName() const { return m_sName; }
	
	/**
	 * @return	The full path of the project's directory
	 */
	QString getPath() const { return m_dir.absPath(); }
	
	/**
	 * @return	A list of command-line arguments to pass to a Cscope object,
	 * based on the project's options
	 */
	const QStringList& getArgList() const { return m_slArgs; }
	
	/**
	 * @return	The time, in seconds, to wait before rebuilding the
     *			cross-refernce database.
	 */
	int getAutoRebuildTime() const { return m_nAutoRebuildTime; }
	
	static void getDefOptions(Options&);

signals:
	/**
	 * Emitted when a set of files is added to the current project.
	 * @param	slFiles	A list containing the full paths of the new files
	 */
	void filesAdded(const QStringList& slFiles);
	
	/**
	 * Emitted when the list of files in a project is modified.
	 */
	void fileListChanged();
	
private:
	/** The name of the project, as written in the configuration file */
	QString m_sName;

	/** The directory associated with the project */
	QDir m_dir;

	/** The file that holds the paths of all source files in this project
		("cscope.files") */
	QFile* m_pFiles;

	/** The configuration file ("cscope.proj") */
	KConfig* m_pConf;

	/** true if there is an active project, false otherwise */
	bool m_bOpen;
	
	/** true if the current project is a temporary wrapper for a Cscope.out
		file */
	bool m_bTempProj;
	
	/** A list of Cscope command-line arguments based on the project's
		options. */
	QStringList m_slArgs;

	/** The time, in seconds, to wait before rebuilding the cross-refernce
	 	database. */
	int m_nAutoRebuildTime;
	
	/** A list of symbols previously queried. */
	QStringList m_slSymHistory;
	
	static void initConfig(const QString&, QFile**, KConfig**);
	static bool isCscopeOut(const QString&);
	static void writeOptions(KConfig*, const Options&);
};

#endif
