/***************************************************************************
                          preffrontenddlg.cpp  -  description
                             -------------------
    begin                : Fri Jun 13 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpushbutton.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>
#include "preffrontend.h"
#include "kscopeconfig.h"
#include "shellfrontend.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
PrefFrontend::PrefFrontend(QWidget* pParent, const char* szName) :
	PrefFrontendLayout(pParent, szName)
{
	// Set initial values
	load();
		
	// Display a file search dialog on each of the "Find..." buttons
	connect(m_pCscopeButton, SIGNAL(clicked()), this,
		SLOT(slotFindCscope()));
	connect(m_pCtagsButton, SIGNAL(clicked()), this,
		SLOT(slotFindCtags()));

	// Attempt to guess paths based on the user's PATH environment variable
	connect(m_pGuessButton, SIGNAL(clicked()), this,
		SLOT(slotGuessPaths()));	
		
	// Emit the modified() signal when a new path is set
	connect(m_pCscopeEdit, SIGNAL(textChanged(const QString&)), this,
		SIGNAL(modified()));
	connect(m_pCtagsEdit, SIGNAL(textChanged(const QString&)), this,
		SIGNAL(modified()));
	connect(m_pVerboseCheck, SIGNAL(toggled(bool)), this,
		SIGNAL(modified()));
}

/**
 * Class destructor.
 */
PrefFrontend::~PrefFrontend()
{
}

/**
 * Reads the current settings from the configuration object, and applies them
 * the the page's widget.
 */
void PrefFrontend::load()
{
	m_pCscopeEdit->setText(Config().getCscopePath());
	m_pVerboseCheck->setChecked(Config().useVerboseCscope());
	m_pCtagsEdit->setText(Config().getCtagsPath());
}

/**
 * Commits settings changes to the configuration object.
 */
void PrefFrontend::apply()
{
	Config().setCscopePath(m_pCscopeEdit->text());
	Config().setVerboseCscope(m_pVerboseCheck->isChecked());
	Config().setCtagsPath(m_pCtagsEdit->text());
}

/**
 * Prompts the user for the full path of the Cscope executable.
 * This slot is called when the user clicks the "Find..." button besides the
 * Cscope path line edit widget. In response, it displays a file search
 * dialog, which allows the user to select the Cscope executable. Once
 * selected, this path is entered in the line edit widget.
 */
void PrefFrontend::slotFindCscope()
{
	QString sPath;

	sPath = KFileDialog::getOpenFileName(QString::null);
	if (!sPath.isEmpty())
		m_pCscopeEdit->setText(sPath);
}

/**
 * Prompts the user for the full path of the Ctags executable.
 * This slot is called when the user clicks the "Find..." button besides the
 * Cscope path line edit widget. In response, it displays a file search
 * dialog, which allows the user to select the Ctags executable. Once
 * selected, this path is entered in the line edit widget.
 */
void PrefFrontend::slotFindCtags()
{
	QString sPath;

	sPath = KFileDialog::getOpenFileName(QString::null);
	if (!sPath.isEmpty())
		m_pCtagsEdit->setText(sPath);
}

/**
 * Emits the modified() signal whenever any of the paths edit widgets changes
 * its contents.
 * This slot is connected to the textChanged() signal of each of the path
 * edit widgets. By emitting the modified() signal, the widget notifies the
 * parent dialog it should enable the "Apply" button.
 */
void PrefFrontend::slotChanged(const QString&)
{
	emit modified();
}

/**
 * Checks the user's PATH environment variable for a Cscope and Ctags
 * executables.
 * If found, the full paths to these files are set in the appropriate widgets.
 * This slot is connected to the clicked() signal of the "Guess" push button.
 */
void PrefFrontend::slotGuessPaths()
{
	KStandardDirs sd;
	QString sScript;
	ShellFrontend shell;
	
	// Find the configuration script
	sScript = sd.findResource("data", "kscope/kscope_config");
	if (sScript.isEmpty())
		return;
	
	sScript = QString("source ") + sScript;
		
	// Find the Cscope executable
	m_pCscopeEdit->setText(shell.run(sScript + " cscope"));
	
	// Determine whether Cscope has the -v option
	m_pVerboseCheck->setChecked(shell.run(sScript + " cscope_verb") ==
		"yes");
		
	// Find the Ctags executable
	m_pCtagsEdit->setText(shell.run(sScript + " ctags"));
}
