/***************************************************************************
                          openprojectdlg.cpp  -  description
                             -------------------
    begin                : Wed Jun 11 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpushbutton.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include "openprojectdlg.h"
#include "kscopeconfig.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
OpenProjectDlg::OpenProjectDlg(QWidget* pParent, const char* szName) :
	OpenProjectLayout(pParent, szName)
{
	loadRecent();
	
	connect(m_pOpenButton, SIGNAL(clicked()), this, SLOT(accept()));
	connect(m_pCancelButton, SIGNAL(clicked()), this, SLOT(reject()));

	// Browse for a project when the "Find..." button is clicked
	connect(m_pFindButton, SIGNAL(clicked()), this, SLOT(slotFindProject()));

	// Remove a project from the recent projects list when the "Remove"
	// button is clicked
	connect(m_pRemoveButton, SIGNAL(clicked()), this,
		SLOT(slotRemoveProject()));

	// Set the selected project when its path is clicked in the recent
	// projects list
	connect(m_pRecentList, SIGNAL(clicked(QListBoxItem*)), this,
		SLOT(slotSelectProject(QListBoxItem*)));

	// Open the project when its path is double-clicked in the recent
	// projects list
	connect(m_pRecentList, SIGNAL(doubleClicked(QListBoxItem*)), this,
		SLOT(slotOpenProject(QListBoxItem*)));
}

/**
 * Class destructor.
 */
OpenProjectDlg::~OpenProjectDlg()
{
}

/**
 * Fills the recent projects list box with the project paths read from the 
 * configuration file.
 */
void OpenProjectDlg::loadRecent()
{
	const QStringList& slProjects = Config().getRecentProjects();
	QStringList::const_iterator itr;

	// Create a list item for each project in the list
	for (itr = slProjects.begin(); itr != slProjects.end(); ++itr)
		m_pRecentList->insertItem(*itr);
}

/**
 * Allows the user to select a project by browsing the file system for a
 * project file (named "cscope.proj".)
 * This slot is connected to the clicked() signal of the "Find..." button.
 */
void OpenProjectDlg::slotFindProject()
{
	QString sPath;
	
	// Prompt the user for a project file
	sPath = KFileDialog::getOpenFileName(QString::null, "cscope.proj");
	if (sPath.isEmpty())
		return;

	// Set the project's path in the selection edit-box
	QFileInfo fi(sPath);
	m_pPathEdit->setText(fi.dirPath(true));
}

/**
 * Removes a project from the recent projects list.
 * This slot is connected to the clicked() signal of the "Remove" button.
 */
void OpenProjectDlg::slotRemoveProject()
{
	QListBoxItem* pItem;

	// Remove the selected item, if any
	pItem = m_pRecentList->selectedItem();
	if (pItem != NULL) {
		Config().removeRecentProject(pItem->text());
		m_pRecentList->removeItem(m_pRecentList->currentItem());
	}
}

/**
 * Selects a project for opening when an item is clicked in the recent
 * projects list.
 * This slot is connected to the clicked() signal of the recent projects
 * list box.
 * @param	pItem	The selected project item
 */
void OpenProjectDlg::slotSelectProject(QListBoxItem* pItem)
{
	if (pItem != NULL)
		m_pPathEdit->setText(pItem->text());
}

/**
 * Selects a project for opening and closes the dialogue when an item in the
 * recent projects list is double-clicked. 
 * This slot is connected to the doubleClicked() signal of the recent 
 * projects list box.
 * @param	pItem	The selected project item
 */
void OpenProjectDlg::slotOpenProject(QListBoxItem* pItem)
{
	if (pItem != NULL) {
		m_pPathEdit->setText(pItem->text());
		accept();
	}
}
