/***************************************************************************
                          kscopeconfig.h  -  description
                             -------------------
    begin                : Fri Jun 13 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KSCOPECONFIG_H
#define KSCOPECONFIG_H

#include <qobject.h>
#include <qstringlist.h>
#include <qcolor.h>
#include <kdockwidget.h>

typedef QValueList<int> SPLIT_SIZES;

/**
 * Loads and stores global configuration parameters.
 * @author Elad Lahav
 */

class KScopeConfig : public QObject
{
	Q_OBJECT
	
public:
	KScopeConfig();
	~KScopeConfig();

	/** The child widgets whose visual characteristics can be modified. */
	enum ConfWidget { FileList = 0, TagList, QueryList };
	
	/** Defines values for the colour characteristics of a widget. */
	enum ColorType { Background = 0, Foreground = 1 };
	
	/** Sort order values for the tags list. */
	enum CtagSort { NameAsc = 0, NameDes, LineAsc, LineDes, TypeAsc,
		TypeDes };
	
	/** Types of systems that determine certain aspects in KScope's
		behaviour. 
		For fast systems, certain time-consuming operations, such as
		rebuilding the database, may be performed automatically. Such
		behaviour, however, is not desired on slow systems, in which the user
		should handle such operations manually. */
	enum SysProfile { Fast, Slow };	
		
	/** The different options for a popup menu to be installed in the editor
		parts. */
	enum EditorPopup { Embedded, KScopeOnly };	
	
	void load();
	void loadDefault();
	void loadWorkspace(KDockMainWindow*);
	void store();
	void storeWorkspace(KDockMainWindow*);
	bool isFirstTime();
	
	const QString& getCscopePath() const;
	void setCscopePath(const QString&);
	bool useVerboseCscope() const;
	void setVerboseCscope(bool);
	const QString& getCtagsPath() const;
	void setCtagsPath(const QString& sPath);
	const QStringList& getRecentProjects() const;
	void addRecentProject(const QString&);
	void removeRecentProject(const QString&);
	bool getShowTagList() const;
	void setShowTagList(bool);
	const SPLIT_SIZES& getEditorSizes() const;
	void setEditorSizes(const SPLIT_SIZES&);
	const QColor& getColor(ConfWidget, ColorType) const;
	void setColor(ConfWidget, ColorType, const QColor&);
	const QFont& getFont(ConfWidget) const;
	void setFont(ConfWidget, const QFont&);
	uint getMaxPosHistory();
	CtagSort getCtagSortOrder();
	void setCtagSortOrder(CtagSort);
	bool getReadOnlyMode();
	void setReadOnlyMode(bool);
	bool getLoadLastProj();
	void setLoadLastProj(bool);
	bool getAutoTagHl();
	void setAutoTagHl(bool);
	bool getUseBriefQueryCaptions();
	void setUseBriefQueryCaptions(bool);
	bool getShowQueryButtons();
	void setShowQueryButtons(bool);
	bool getWarnModifiedOnDisk();
	void setWarnModifiedOnDisk(bool);
	const QString& getExtEditor();
	void setExtEditor(const QString&);
	bool useExtEditor();
	SysProfile getSysProfile() const;
	void setSysProfile(SysProfile);
	EditorPopup getEditorPopup() const;
	QString getEditorPopupName() const;
	void setEditorPopup(EditorPopup);
	
private:
	/** A list of previously loaded projects. */
	QStringList m_slProjects;

	/** Defines the list of all configurable parameters in KScope.
		The use of a structure helps define default values (@see s_cpDef) */
	struct ConfParams {
		/** The full path of the Cscope executable. */
		QString sCscopePath;
		
		/** Whether the '-v' flag should be used with Cscope. */
		bool bVerboseCscope;
		
		/** The full path of the Ctags executable. */
		QString sCtagsPath;
		
		/** Whether the tag list should be visible. */
		bool bShowTagList;
		
		/** The widths of the tag list and editor panes inside an editor
			page. */
		SPLIT_SIZES siEditor;
		
		/** Background colour of the file list. */
		QColor clrFileListBack;
		
		/** Foreground colour of the file list. */
		QColor clrFileListFore;
		
		/** Background colour of the tag lists. */
		QColor clrTagListBack;
		
		/** Foreground colour of the tag lists. */
		QColor clrTagListFore;
		
		/** Background colour of the query result pages. */
		QColor clrQueryListBack;
		
		/** Foreground colour of the query result pages. */
		QColor clrQueryListFore;
		
		/** File list font. */
		QFont fontFileList;
		
		/** Tag lists font. */
		QFont fontTagList;
		
		/** Query pages font. */
		QFont fontQueryList;
		
		/** Maximal size of the position history list. */
		uint nMaxPosHistory;
		
		/** Sort order of the tag lists. */
		CtagSort ctagSortOrder;
		
		/** Whether KScope should operate in code read-only mode. */
		bool bReadOnlyMode;
		
		/** Whether the last open project should be reloaded on start-up. */
		bool bLoadLastProj;
		
		/** Whether tags should be highlighted based on the current cursor
			position. */
		bool bAutoTagHl;
		
		/** Whether query page captions should use mnemonics for query types,
			instead of the full description. */
		bool bBriefQueryCaptions;
		
		/** Whether the query toolbar should appear to the right of the query
			window. */
		bool bShowQueryButtons;
		
		/** Whether the warning should be displayed when file is modified on 
			disk by external process. */
		bool bWarnModifiedOnDisk;
		
		/** A command line pattern for an external editor (in read-only
			mode.)*/
		QString sExtEditor;
		
		/** How KScope should treat time-consuming operations. */
		SysProfile profile;
		
		/** The type of popup menu to use in the embedded editor. */
		EditorPopup popup;
	};

	/** The current configuration parameters */
	ConfParams m_cp;

	/** Holds default values for the configuration parameters */
	static ConfParams s_cpDef;
	
	/** Write font preferences only if modified by the user (keep default
		setting otherwise) */
	bool m_bFontsChanged;
};

extern KScopeConfig& Config();

#endif
