/***************************************************************************
                          filelist.cpp  -  description
                             -------------------
    begin                : Sun May 11 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qlineedit.h>
#include <qfileinfo.h>
#include <klocale.h>
#include "filelist.h"
#include "kscope.h"
#include "kscopeconfig.h"

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
FileList::FileList(QWidget* pParent, const char* szName) :
	SearchList(1, pParent, szName)
{
	// Set the list's columns
	m_pList->addColumn("");
	m_pList->addColumn(i18n("File"));
	m_pList->addColumn(i18n("Path"));
	// Sort only when asked by user
	m_pList->setSortColumn(m_pList->columns() + 1);
	m_pList->setAllColumnsShowFocus(true);
	
	// Set colours and font
	applyPrefs();
}

/**
 * Class destructor.
 */
FileList::~FileList()
{
}

/**
 * Adds a single entry to the file list.
 * Implements the addItem() virtual method of the FileListTarget base
 * class. When a FileList object is given as a parameter to
 * ProjectManager::fillList(), this method is called for each file included
 * in the project. A new list item is created, containing the file's name and
 * path, and is added to the list.
 * @param	sFilePath	The full path of a source file
 */
void FileList::addItem(const QString& sFilePath)
{
	QString sFileType, sFileName;
	int nTypePos;
		
	// Get the file's MIME-type (empty string for file names without an
	// extension)
	nTypePos = sFilePath.findRev('.');
	if (nTypePos > -1)
		sFileType = sFilePath.mid(nTypePos + 1);

	// Extract the file name and create the list item
	sFileName = sFilePath.mid(sFilePath.findRev('/') + 1);
	new QListViewItem(m_pList, sFileType, sFileName, sFilePath);
}

/**
 * Searches the list for the given file path.
 * @param	sPath	The full path of the file to find
 * @return	true if the file was found in the list, false otherwise
 */
bool FileList::findFile(const QString& sPath)
{
	return (m_pList->findItem(sPath, 2) != NULL);
}

/**
 * Removes all items from the file list.
 */
void FileList::clear()
{
	m_pList->clear();
	m_pEdit->setText("");
}

/**
 * Opens a file for editing when its entry is clicked in the file list.
 * @param	pItem	The clicked list item
 */
void FileList::processItemSelected(QListViewItem* pItem)
{
	QString sPath;

	// Get the file path and submit a request to open it for editing
	sPath = pItem->text(2);
	emit fileRequested(sPath, 0);
}

/**
 * Sets the list's colours and font, according the user's preferences.
 */
void FileList::applyPrefs()
{
	// Apply colour settings
	m_pList->setPaletteBackgroundColor(Config().getColor(
		KScopeConfig::FileList, KScopeConfig::Background));
	m_pList->setPaletteForegroundColor(Config().getColor(
		KScopeConfig::FileList, KScopeConfig::Foreground));
	m_pList->setFont(Config().getFont(KScopeConfig::FileList));
}

/**
 * Sets the keyboad focus to the search box.
 * This slot is connected to a keyboard shortcuts for quick access to the
 * file list.
 */
void FileList::slotSetFocus()
{
	m_pEdit->setFocus();
}
