/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "SwarmingNoise.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_colorspace_factory_registry.h>
#include <kis_meta_registry.h>
#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

typedef KGenericFactory<KritaSwarmingNoise> KritaSwarmingNoiseFactory;
K_EXPORT_COMPONENT_FACTORY( kritaSwarmingNoise, KritaSwarmingNoiseFactory( "krita" ) )

KritaSwarmingNoise::KritaSwarmingNoise(QObject *parent, const char *name, const QStringList &)
: KParts::Plugin(parent, name)
{
    setInstance(KritaSwarmingNoiseFactory::instance());

    kdDebug(41006) << "SwarmingNoise filter plugin. Class: "
    << className()
    << ", Parent: "
    << parent -> className()
    << "\n";

    if(parent->inherits("KisFilterRegistry"))
    {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisSwarmingNoiseFilter());
    }
}

KritaSwarmingNoise::~KritaSwarmingNoise()
{
}

KisSwarmingNoiseFilter::KisSwarmingNoiseFilter() 
    : KisFilter(id(), "SwarmingNoise", i18n("&Swarming noise..."))
{
}

std::list<KisFilterConfiguration*> KisSwarmingNoiseFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisSwarmingNoiseFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("threshold", 0.5);
    return config;
};

KisFilterConfigWidget * KisSwarmingNoiseFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    Q_UNUSED(dev);
    vKisDoubleWidgetParam param;
    param.push_back( KisDoubleWidgetParam( 0.0, 1.0, 0.5, i18n("Amount"), "amount" ) );
    KisFilterConfigWidget * w = new KisMultiDoubleFilterWidget(parent, id().id().ascii(), id().id().ascii(), param );
    Q_CHECK_PTR(w);
    return w;
}

KisFilterConfiguration* KisSwarmingNoiseFilter::configuration(QWidget* nwidget)
{
    KisMultiDoubleFilterWidget* widget = (KisMultiDoubleFilterWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
        config->setProperty("amount", widget->valueAt( 0 ));
        return config;
    }
}

void KisSwarmingNoiseFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);

    double amount;
    if(config)
    {
        amount = config->getDouble("amount", 0.5);
    } else {
        amount = 0.5;
    }

    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();
    
    KisColorSpace* grayCS = KisMetaRegistry::instance()->csRegistry()->getColorSpace(KisID("GRAYA"),"");
    Q_UINT8 gray[2];
    gray[1] = 0xFF;
    
    Q_UINT8* pixel = new Q_UINT8[ grayCS->pixelSize() ];
    const Q_UINT8 *colors[2];
    colors[0] = pixel;
    Q_UINT8 weights[2];
    weights[0] = amount * 255;
    weights[1] = (1.0 - amount) * 255;
    while( ! srcIt.isDone() )
    {
        if(srcIt.isSelected())
        {
            colors[1] = srcIt.oldRawData();
            gray[0] = rand() / (double)RAND_MAX * 0xFF;
            grayCS->convertPixelsTo(gray, pixel, cs, 1 );
            char alpha = cs->getAlpha(dstIt.rawData());
            cs->mixColors(colors, weights, 2, dstIt.rawData() );
            cs->setAlpha(dstIt.rawData(), alpha, 1);
        }
        setProgress(++pixelsProcessed);
        ++srcIt;
        ++dstIt;
    }

    delete[]pixel;
    setProgressDone(); // Must be called even if you don't really support progression
}
