/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef RADIOSTATION_H
#define RADIOSTATION_H

#include "get_stream_info.h"
#include "ripping.h"
#include "settings_stream_dialog.h"

/** \brief This class rips radio streams and uses the options available from a config file.
*
* It uses via #config_skeleton a config file (see property #configFileName) to
* determinate the URI of the stream and further options for the record (pregrap,
* postgrab, user agent string...).
*
* It provides the public slot <tt>void showSettingsDialog()</tt>, who shows
* a dialog that is a graphical interface to the config file. */
class radioStation : public ripping
{

     Q_OBJECT

     /** This property holds the path to the configuration file used by this object.
     *   You can determinate the path to the configuration file in the constructor (see there for
     *   further details). Once set there, it can't be changed anymore.
     *
     *   \li \e read: <tt> QString configFileName() const </tt>
     *
     *   Use #config_skeleton for comfortable access to the settings that are stored in the
     *   configuration file.
     *   \sa <tt> QString internal_configFileName </tt> */
     Q_PROPERTY(QString configFileName READ configFileName)

     /** This property holds the URI from which the stream comes.
     *
     * This property doesn't exist on it's own. It uses #serverUri() to get the
     * URI. The notify signal is emitted \e always when the config file changes.
     *
     * <b> Initialization value: </b> <EM> an empty string </EM>
     *
     * This property guards it's last value also when \e %streamripper terminates.
     *
     * \li \e read: \code virtual PropertyValue uri() const \endcode
     * \li \e notify:
     *     \code void uriChanged(qlonglong index, PropertyValue newUri) \endcode
     *
     * \sa \code static PropertyValue formatedServerName(const QString theServerName) \endcode */
     Q_PROPERTY(PropertyValue uri READ uri)

  public:
     //contructor(s) and destructor(s)
     /** Constructor of the class.
     * @param parent     Sets the parent of this object.
     * @param mainWidget When an options dialog gets displayed, this widget
     *                   is used as parent and the dialog
     *                   gets centerd to this Widget.
     * @param configFileName Here you can give a (local!) path to a configuration file. See
     *                   class \e QFile::setName for
     *                   details about which format is valid. If the file doesn't
     *                   exist, the object tries to create
     *                   it. If this isn't possible (for example if the path is invalid
     *                   or you don't have the rights
     *                   or you gave the name of a directory or if you didn't give any path),
     *                   then a new configuration file is created in the appdata directory
     *                   set in KDE, with
     *                   the name <em>stream_??????_rc</em> with the six <em>?</em> replaced by
     *                   random letters. You can obtain the path
     *                   of the really used file later trough the property #configFileName. */
     explicit radioStation(QObject *parent = 0,
                           QWidget *mainWidget = 0,
                           const QString & configFileName = QString());
     /** Conveniance constructor of the class.
     * @param parent     Sets the parent of this object.
     * @param mainWidget When an options dialog gets displayed, this widget is
     *                   used as parent and the dialog
     *                   gets centerd to this Widget.
     * @param configFileName Here you can give a (local!) path to a configuration file. See
     *                   class \e QFile::setName for
     *                   details about which format is valid. If the file doesn't
     *                   exist, the object tries to create
     *                   it. If this isn't possible (for example if the path is invalid
     *                   or you don't have the rights
     *                   or you gave the name of a directory or if you didn't give any path),
     *                   then a new configuration file is created in the appdata directory
     *                   set in KDE, with
     *                   the name <em>stream_??????_rc</em> with the six <em>?</em> replaced by
     *                   random letters. You can obtain the path
     *                   of the really used file later trough the property #configFileName.
     * @param index      Initializes the property #index. */
     explicit radioStation(QObject *parent,
                           QWidget *mainWidget,
                           const QString & configFileName,
                           qlonglong index);
     /** The destructor. */
     virtual ~radioStation();
     /** See property #configFileName. */
     QString configFileName() const;
     /** See property #uri. */
     virtual PropertyValue uri() const;
     /** Sets the URI of the stream and actualizes the meta data.
     * \param uri The new URI */
     virtual void setServerUri(const QUrl & uri);
     /** \returns A full-featured formateded version for the property #uri. */
     static PropertyValue formatedUri(const QString & theUri);

  signals:
     void uriChanged(qlonglong index, PropertyValue newUri);

  public slots:
     /** Execs a configuration dialog.
     *   \sa #helper_displaySettingsDialog() */
     int execSettingsDialog();
     /** Shows a configuration dialog.
     *   \sa #helper_displaySettingsDialog() */
     void showSettingsDialog();

  protected:
     /** Provides comfortable access to configuration file (for stream settings).
     *   \sa #configFileName
     *   \sa #shared_config (used internally to access #configFileName) */
     QPointer<settings_stream> config_skeleton;
     /** Is set by the constructor. The settings dialog will be centered
     *   to this widget.
     *   \sa showSettingsDialog() */
     QPointer<QWidget> m_mainWidget;
     /** This function provides the necessary parameters for calling \e streamripper.
     *   Uses ripping::parameterList() and adds the parameters readed from #configFileName. */
     virtual QStringList parameterList() const;
     /** \returns The stream URI readed from #configFileName. */
     virtual QString serverUri() const;
     /** \name Reimplemented properties
     *   These functions for property access are reimplemented. They write both to the property
     *   system of #ripping \e and to the file <tt>configFileName</tt>. */
     // doxygen doesn't accept "#configFileName" in this special section.
     //@{
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setBitrate(const qint64 bitrate);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setMetaInterval(const qint64 metaInterval);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setServerName(const QString & newServerName);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setStreamName(const QString & newStreamName);
     //@}
     /** A pointer to the settings dialog (if actually exists one).
     *   \sa showSettingsDialog() */
     QPointer<settings_stream_dialog> settingsDialog;
     virtual QString workingDirectory() const;

  protected slots:
     /** Provided for conveniance. It relies on the regular \e set method of
     * the property. It is important to guard a clear concept of responsabilities
     * and to make sure that all necessary updates to internal data is done.
     * \param ignored This argument hasn't any function.
     * \param newBitrate The new value. Only the value of its member \e internalValue
     * is used. */
     virtual void setBitrate(const qlonglong ignored, const PropertyValue & newBitrate);
     /** Provided for conveniance. It relies on the regular \e set method of
     * the property. It is important to guard a clear concept of responsabilities
     * and to make sure that all necessary updates to internal data is done.
     * \param ignored This argument hasn't any function.
     * \param newMetaInterval The new value. Only the value of its member \e internalValue
     * is used. */
     virtual void setMetaInterval(const qlonglong ignored, const PropertyValue & newMetaInterval);
     /** Provided for conveniance. It relies on the regular \e set method of
     * the property. It is important to guard a clear concept of responsabilities
     * and to make sure that all necessary updates to internal data is done.
     * \param ignored This argument hasn't any function.
     * \param newServerName The new value. Only the value of its member \e internalValue
     * is used. */
     virtual void setServerName(const qlonglong ignored, const PropertyValue & newServerName);
     /** Provided for conveniance. It relies on the regular \e set method of
     * the property. It is important to guard a clear concept of responsabilities
     * and to make sure that all necessary updates to internal data is done.
     * \param ignored This argument hasn't any function.
     * \param newStreamName The new value. Only the value of its member \e internalValue
     * is used. */
     virtual void setStreamName(const qlonglong ignored, const PropertyValue & newStreamName);

  private:
     /** A helper function used in the constructors.
     *   \li Tests if #configFileName is working fine
     *       as config file. (If it doesn't exist, than
     *       a new file is created and #configFileName
     *       is set to this new file.)
     *   \li Sets up #shared_config.
     *   \li Sets up #config_skeleton. */
     void helper_setupConfigSystem(const QString & configFile);
     /** Displays a configuration dialog for the settings in the config file.
     *   The dialog is centered to #m_mainWidget. When the dialog becomes closed,
     *   it will be deleted automatically.
     *   \warning The parent of the dialog isn't this object (because it isn't a
     *            widget), but #m_mainWidget. So the destructor must delete the
     *            dialog manually (if it still exists because it hasn't been closed).
     *   \param returnImmediately If \e true, we show() the dialog. If \e false, we
     *                            exec() the dialog.
     *   \returns If using exec(), it returns the exit code (see QDialog::DialogCode).
     *            Else the return value is undefined.
     *   \sa #settingsDialog */
     int helper_displaySettingsDialog(const bool returnImmediately);
     /** Used internally to store the property #configFileName. */
     QString internal_configFileName;
     /** This points to a KSharedConfig object that uses #configFileName. Use #config_skeleton
     *   to get comfortable access all these things.
     *
     *   <em>This is a <tt>KSharedConfig::Ptr</tt> (a <tt>KSharedPtr</tt> to a
     *   <tt>KSharedConfig</tt> object). The constructor will create a <tt>KSharedConfig</tt>
     *   object that uses #configFileName - and this pointer will point on it.
     *   When the pointer is deleted (=when the <tt>radiostation</tt>
     *   object it belongs to becomes deleted), it will automatically
     *   delete also the <tt>KSharedConfig</tt> object it points to.</em> */
     KSharedConfig::Ptr shared_config;
     /** \sa #updateMetaData() */
     QPointer<get_stream_info> infoCatcher;

  private slots:
    /** This function loads the values of the properties that are stored in #configFileName from
    * this file and writes them to the property system of the inherited class #ripping, causing
    * the notify signals to be emitted if necessary. It also emits always the notify signal for
    * the property #uri.
    *
    * This slot is necessary to catch changes to the config that that are not done by this class
    * itself (for example by the settings dialog, see helper_displaySettingsDialog()).
    *
    * <i>As we <b>must</b> catch this, in consequenze we don't need to worry about writing to the
    * properties of the class #ripping when implementing set functions - this is done
    * automatically by this function here after that the values are written to the config
    * file.</i> */
    void helper_write_properties_from_file_to_class_ripping();
    /** Releads parts of the meta data that has typically changed after the URI has changed.
    * \sa #setServerUri() */
    void updateMetaData();
 };

#endif
