 /**************************************************************************
 *   Copyright (C) 2006 by Danny Kukawka                                   *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License     *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*! 
 * \file 	hardware.cpp
 * \brief 	In this file can be found the hardware information related code. 
 * \author 	Danny Kukawka, <dkukawka@suse.de>, <danny.kukawka@web.de>
 * \date    	2006
 */

// include global header
#include <fcntl.h>

// include QT header
#include <qtimer.h>
#include <qdir.h>

// include own header
#include "hardware.h"

/*! The default constructor of the class HardwareInfo */
HardwareInfo::HardwareInfo() {
	myDebug ("HardwareInfo::HardwareInfo");

	// init members
	acadapter = true;
	lidclose = false;
	dbus_terminated = true;
	hal_terminated = true;
	laptop = false;
	brightness = false;

	// update everything the first time
	update_info_ac_changed = true;
	update_info_cpufreq_policy_changed = true;
	update_info_primBattery_changed = true;
	
	currentCPUFreqPolicy = -1;
	primaryBatteriesWarnLevel = 12;
	primaryBatteriesLowLevel = 7;
	primaryBatteriesCriticalLevel = 2;

	allUDIs = QStringList();
	BatteryList.setAutoDelete( true ); // the list owns the objects
	
	primaryBatteries = new BatteryCollection(BAT_PRIMARY);
	setPrimaryBatteriesWarningLevel(); // force default settings

	// connect to D-Bus and HAL
	dbus_HAL = new dbusHAL();
	if (dbus_HAL->isConnectedToDBUS()) {
		dbus_terminated = false;
		if (dbus_HAL->isConnectedToHAL()) {
			hal_terminated = false;
		} else {
			myDebug("Error: Could not connect to HAL");
		}
	} else { 
		myDebug("Error: Could not connect to D-Bus & HAL");
	}

	checkPowermanagement();
	checkIsLaptop();
	checkBrightness();
	checkCPUFreq();
	checkSuspend();
	intialiseHWInfo();

	updatePrimaryBatteries();

	// connect to signals
	connect(dbus_HAL, SIGNAL(msgReceived_withStringString( msg_type, QString, QString )),
		this, SLOT(processMessage( msg_type, QString, QString )));
	connect(dbus_HAL, SIGNAL(backFromSuspend(int)), this, SLOT(handleResumeSignal(int)));

}

/*! The default desctuctor of the class HardwareInfo */
HardwareInfo::~HardwareInfo() {
	myDebug ("HardwareInfo::~HardwareInfo");

	delete dbus_HAL;
	dbus_HAL = NULL;
}

/*!
 * This funtion is used to handle the reconnect to the D-Bus daemon
 */
void HardwareInfo::reconnectDBUS() {
	myDebug ("HardwareInfo::reconnectDBUS");

	if (!dbus_HAL->isConnectedToDBUS()) {
		bool _reconnect = dbus_HAL->reconnect();

		if (!_reconnect && !dbus_HAL->isConnectedToDBUS()) {
			//reconnect failed
			emit dbusRunning(DBUS_NOT_RUNNING);
			QTimer::singleShot(4000, this, SLOT(reconnectDBUS()));
		} else if (!_reconnect && dbus_HAL->isConnectedToDBUS()) {
			// reset everything, we are reconnected
			dbus_terminated = false;
			hal_terminated = true;
			emit dbusRunning(DBUS_RUNNING);
		} else if (_reconnect) {
			// reset everything, we are reconnected
			dbus_terminated = false;
			hal_terminated = false;
			reinitHardwareInfos();
			emit dbusRunning(hal_terminated);
			emit halRunning(DBUS_RUNNING);
		}  
	} 

}

/*!
 * This funtion is used to reinit all hardware information.
 * \return 		Boolean with result of the call
 * \retval true		if reinit HW infos correct
 * \retval false	if not
 */
bool HardwareInfo::reinitHardwareInfos () {
	myDebug("HardwareInfo::reinitHardwareInfos");
	
	if (dbus_HAL->isConnectedToDBUS() && dbus_HAL->isConnectedToHAL()) {
		/* first cleanup */
		acadapter = true;
		lidclose = false;
		laptop = false;
		brightness = false;
		has_APM = false;
		has_ACPI = false;
	
		update_info_ac_changed = true;
		update_info_cpufreq_policy_changed = true;
		update_info_primBattery_changed = true;
		
		allUDIs = QStringList();

		// TODO: check if we maybe need to unconnect the releated connects to signals !!!
		BatteryList.clear();
		primaryBatteries = new BatteryCollection(BAT_PRIMARY);
		
		/* reinit hardware data */
		checkPowermanagement();
		checkIsLaptop();
		checkBrightness();
		checkCPUFreq();
		checkSuspend();
		intialiseHWInfo();
		updatePrimaryBatteries();
		
		return true;
	} else {
		return false;
	}
}

/*!
 * This funtion is used to parse a message from D-Bus for the different 
 * messagetypes and events.
 * \param type 		a \ref msg_type which should be parse/processed
 * \param message 	the message
 * \param value 	an optional message value as e.g. message string
 */
void HardwareInfo::processMessage (msg_type type, QString message, QString value) {
	myDebug ("HardwareInfo::processMessage");

	switch(type) {
		case ACPI_EVENT:
			// we don't handle acpi events here atm
			break;
		case DBUS_EVENT:
			myDebug("D-Bus Event: '%s'", value.ascii());
			if ( message.startsWith("dbus.terminate")){
				// TODO: addapt from pdaemon
				dbus_terminated = true;
				QTimer::singleShot(4000, this, SLOT(reconnectDBUS()));
			}
			else if ( message.startsWith("hal.")) {
				if ( message.startsWith("hal.terminate")) {
					hal_terminated = true;
					emit halRunning(false);
					// TODO: update data
					emit generalDataChanged();
				} else if ( message.startsWith("hal.started")) {
					hal_terminated = false;
					reinitHardwareInfos();
					emit halRunning( true );
					// TODO: update data
					emit generalDataChanged();
				}
				// TODO: addapt from pdaemon
			}
			break;
		case HAL_DEVICE:
			// --> we can maybe ignore these events except for batteries, not shure atm
			int _type;

			if (message.startsWith("DeviceAdded")) {
				if (checkIfHandleDevice(value, &_type)) {
					switch (_type) {
						case BATTERY:
							// TODO: handle code
							break;
						default:
							myDebug("New device added, but this should never happen.");
							myDebug("Device udi: %s, type: %i", value.latin1(), _type);
							break;
					}
				}
			} else if (message.startsWith("DeviceRemoved")) {
				if (allUDIs.contains(value)) {
					if (checkIfHandleDevice(value, &_type)) {
						switch (_type) {
							case BATTERY:
								// TODO: handle code
								break;
							default:
								myDebug("Couldn't handle unknown device");
								break;
						}
					}
				} else {
					myDebug ("Not monitored device removed: %s", value.latin1());
				}
			} else {
				myDebug("Unknown HAL_DEVICE message: %s", message.latin1());
			}
			break;
		case HAL_PROPERTY_CHANGED:
			if (!message.isEmpty() && allUDIs.contains( message )) {
				if (value.startsWith( "ac_adapter.present" )) {
					checkACAdapterState();
				} else if (value.startsWith( "battery." )) {
					// this is a battery event
					updateBatteryValues(message, value);
				} else if (value.startsWith( "button.state.value" )) {
					if (message.startsWith( *udis["lidclose"] )) {
						checkLidcloseState();
					}
				}
				// TODO: add needed code
			} else {
				myDebug("HAL_PROPERTY_CHANGED event for unmonitored device: %s", value.latin1());
			}
			break;
		case HAL_CONDITION:
			// TODO: Check if we really need to monitor this events. We get maybe also 
			//	 HAL_PROPERTY_CHANGED event for the key
			if (message.startsWith("ButtonPressed")) {
				if (value.startsWith("lid")) {
					checkLidcloseState();
				} else if (value.startsWith("power")) {
					emit powerButtonPressed();
				} else if (value.startsWith("sleep") || value.startsWith("suspend")) {
					emit sleepButtonPressed();
				} else if (value.startsWith("hibernate")) {
					emit s2diskButtonPressed();
				}
			} else {
				myDebug("Unmonitored HAL_CONDITION: %s %s", message.latin1(), value.latin1());
			}
			// TODO: add needed code
			break;
		default:
			myDebug("Recieved unknown package type '%d'", type);
			break;
	};

}

/*!
 * This SLOT is used to fetch the resume signal and multiplex. If needed some
 * actions after resume, do this here.
 * \param result 	integer with the result of the resume/suspend
 */
void HardwareInfo::handleResumeSignal (int result) {
	myDebug("HardwareInfo::handleResumeSignal (int result: '%d')", result);

	emit resumed(result);
}

/*! 
 * This function check for a given UDI, if we should handle a device
 * \param _udi		QString with the UDI of the device
 * \param *type		pointer to a integer to return the type of the device, see \ref device_type
 * \return 		Boolean with info if we should handle the device.
 * \retval true		if we should handle
 * \retval false	if not
 */
bool HardwareInfo::checkIfHandleDevice ( QString _udi, int *type) {
	myDebug ("HardwareInfo::checkIfHandleDevice");
	
	QStringList _cap;

	if (dbus_HAL->halGetPropertyStringList( _udi, "info.capabilities", &_cap)) {
		if (!_cap.isEmpty()) {
			if (_cap.contains("ac_adapter")) {
				*type = BATTERY;
				return true;
			} else if (_cap.contains("button")) {
				QString _val;
				if (dbus_HAL->halGetPropertyString( _udi, "button.type", &_val)) {
					if (_val.startsWith("lid")) {
						*type = LID;
						return true;
					} else if ( _val.startsWith("power")) {
						*type = BUTTON_POWER;
					} else if ( _val.startsWith("sleep")) {
						*type = BUTTON_SLEEP;
						return true;
					}
				}
				*type = UNKNOWN_DEVICE;
				return false;
			} else if (_cap.contains("battery")) {
				*type = BATTERY;
				return true;
			} else {
				myDebug("Device with capability '%s' unhandled", _cap.join(", ").latin1());
			}
		}
	}
	*type = UNKNOWN_DEVICE;
	return false;
}

// --> set some values for devices
/*!
 * This function set the warning level for the primary battery collection
 * If all give param are -1 or not set this function force the current 
 * settings to the primary battery collection.
 * \param _warn		value for the state BAT_WARN or -1
 * \param _low		value for the state BAT_LOW or -1
 * \param _crit		value for the state BAT_CRIT or -1
 */
void HardwareInfo::setPrimaryBatteriesWarningLevel (int _warn, int _low, int _crit ) {
	myDebug ("HardwareInfo::setPrimaryBatteriesWarningLevel(_warn '%d', _low '%d', _crit '%d')", 
		 _warn, _low, _crit);

	if (_warn > -1 && _low > -1 && _crit > -1 ){
		primaryBatteriesWarnLevel = _warn;
		primaryBatteriesLowLevel = _low;
		primaryBatteriesCriticalLevel = _crit;
	}

	if (primaryBatteries) {
		primaryBatteries->setWarnLevel( primaryBatteriesWarnLevel );
		primaryBatteries->setLowLevel( primaryBatteriesLowLevel );
		primaryBatteries->setCritLevel( primaryBatteriesCriticalLevel );
	}
}

// --> init HW information section -- START <---

/*!
 * The function checks if the machine is a laptop.
 */
void HardwareInfo::checkIsLaptop () {
	myDebug ("HardwareInfo::isLaptop");

	QString ret;
	
	if (dbus_HAL->halGetPropertyString(HAL_COMPUTER_UDI, "system.formfactor", &ret)) {

		if (!ret.isEmpty() && ret.startsWith("laptop"))
			laptop = true;
		else 
			laptop = false;
	} else {
		// error case
		laptop = false;
	}
}

/*!
 * The function checks wether the machine support ACPI/APM/PMU or not.
 */
void HardwareInfo::checkPowermanagement() {
	myDebug ("HardwareInfo::checkPowermanagement");
	
	QString ret;

	has_APM = false;
	has_ACPI = false;
	has_PMU = false;
	
	if (dbus_HAL->halGetPropertyString( HAL_COMPUTER_UDI, "power_management.type", &ret)) {
	
		if (ret.isEmpty()) {
			return;
		} else if (ret.startsWith("acpi")) {
			has_ACPI = true;
		} else if (ret.startsWith("apm")) {
			has_APM = true;
		} else if (ret.startsWith("pmu")) {
			has_PMU = true;
		}
	}	
}


/*!
 * The function checks wether the machine can suspend/standby.
 */
void HardwareInfo::checkSuspend() {
	myDebug ("HardwareInfo::checkSuspend()");
	
	QStringList ret;
	bool _ret_b = false;

	suspend_states = SuspendStates();

	if (dbus_HAL->halGetPropertyStringList( HAL_COMPUTER_UDI, HAL_PM_IFACE ".method_names",
						&ret )) {
		// check for suspend2ram
		if (dbus_HAL->halGetPropertyBool( HAL_COMPUTER_UDI, "power_management.can_suspend", &_ret_b ) ||
		    dbus_HAL->halGetPropertyBool( HAL_COMPUTER_UDI, "power_management.can_suspend_to_ram", 
						  &_ret_b )) {
			suspend_states.suspend2ram_can = _ret_b;
			if (_ret_b) {
				if (ret.contains( "Suspend" )) {
					suspend_states.suspend2ram = true;
					suspend_states.suspend2ram_allowed = dbus_HAL->isUserPrivileged("hal-power-suspend");
			}
			} else {
				suspend_states.suspend2ram = false;
				suspend_states.suspend2ram_allowed = -1;
			}
		} else {
			suspend_states.suspend2ram_can = false;
			suspend_states.suspend2ram = false;
			suspend_states.suspend2ram_allowed = -1;
		}

		// check for suspend2disk
		if (dbus_HAL->halGetPropertyBool( HAL_COMPUTER_UDI, "power_management.can_hibernate", &_ret_b ) ||
		    dbus_HAL->halGetPropertyBool( HAL_COMPUTER_UDI, "power_management.can_suspend_to_disk", 
						  &_ret_b )) {
			suspend_states.suspend2disk_can = _ret_b;
			if (_ret_b) {
				if (ret.contains( "Hibernate" )) {
					suspend_states.suspend2disk = true;
					suspend_states.suspend2disk_allowed = dbus_HAL->isUserPrivileged("hal-power-hibernate");
			}
			} else {
				suspend_states.suspend2disk = false;
				suspend_states.suspend2disk_allowed = -1;
			}
		} else {
			suspend_states.suspend2disk_can = false;
			suspend_states.suspend2disk = false;
			suspend_states.suspend2disk_allowed = -1;
		}

		// check for StandBy
		if (dbus_HAL->halGetPropertyBool( HAL_COMPUTER_UDI, "power_management.can_standby", &_ret_b )) {
			suspend_states.standby_can = _ret_b;
			if (_ret_b) {
				if (ret.contains( "Standby" )) {
					suspend_states.standby = true;
					suspend_states.standby_allowed = dbus_HAL->isUserPrivileged("hal-power-hibernate");
			}
			} else {
				suspend_states.standby = false;
				suspend_states.standby_allowed = -1;
			}
		} else {
			suspend_states.standby_can = false;
			suspend_states.standby = false;
			suspend_states.standby_allowed = -1;
		}
	}
}

/*!
 * The function checks wether the machine support CPU frequency changes
 * via HAL.
 */
void HardwareInfo::checkCPUFreq() {
	myDebug ("HardwareInfo::checkCPUFreq()");
	
	bool ret = false;

	if (dbus_HAL->halQueryCapability( HAL_COMPUTER_UDI, "cpufreq_control", &ret )) {
		cpuFreq = ret;
		cpuFreqAllowed = dbus_HAL->isUserPrivileged( "hal-power-cpufreq" );

		checkCurrentCPUFreqPolicy();
	} else {
		cpuFreq = false;
	}
}

/*!
 * The function check the currently selected CPU Frequency policy
 * \return the current policy
 */
int HardwareInfo::checkCurrentCPUFreqPolicy() {
	myDebug ("HardwareInfo::checkCurrentCPUFreqPolicy()");
	
	char *gov;

	int _current = -1;

	if (cpuFreq) {

		if (dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI, 
						    HAL_CPUFREQ_IFACE, "GetCPUFreqGovernor", 
						    &gov, DBUS_TYPE_STRING, DBUS_TYPE_INVALID)) {
			if (gov != NULL) {
				myDebug ("checkCurrentCPUFreqPolicy() - got CPU Freq gov: '%s'", gov);
				if (!strcmp(gov, "ondemand") || !strcmp(gov, "userspace") ||
				    !strcmp(gov, "conservative")) {
					_current = DYNAMIC;
				} else if (!strcmp(gov, "powersave")) {
					_current = POWERSAVE;
				} else if (!strcmp(gov, "performance")) {
					_current = PERFORMANCE;
				} else {
                                        myDebug("Got unknown CPUFreq Policy ('%s') back.", gov);
                                }
			}
			else {
				myDebug ("Could not get information about current governor");
			}
	
		} else {
			myDebug ("Could not get information about current governor");
		}
	} else {
		myDebug ("CPU Frequency interface not supported by machine or HAL");
	}

	cpuFreqGovernor = gov;

	if (_current != currentCPUFreqPolicy) {
		currentCPUFreqPolicy = _current;
		update_info_cpufreq_policy_changed = true;
		emit currentCPUFreqPolicyChanged();
	} else {
		update_info_cpufreq_policy_changed = false;
	}
	
	return currentCPUFreqPolicy;
}


/*!
 * The function checks wether the machine provide a brightness interface and init 
 * (if needed) brightness information.
 */
void HardwareInfo::checkBrightness() {
	myDebug ("HardwareInfo::checkBrightness");

	QStringList devices;

	brightness = false;
	currentBrightnessLevel = -1;
	availableBrightnessLevels = -1;

	if (dbus_HAL->halFindDeviceByString( "info.category", "laptop_panel", &devices)) {
		if (devices.isEmpty()) {
			myDebug("no devie with category laptop_panel found");
			return;
		} else {
			
			int retval;
			brightness = true;
	
			myDebug("HardwareInfo::checkBrightness: %s", devices.first().latin1());
			// we should asume there is only one laptop panel device in the system
			if (dbus_HAL->halGetPropertyInt( devices.first(), "laptop_panel.num_levels", &retval )) {
				udis.insert("laptop_panel", new QString( devices.first() ));
				if (!allUDIs.contains( devices.first() ))
					allUDIs.append( devices.first() );
				availableBrightnessLevels = retval;
			}
	
			// get the current level via GetBrightness
			checkCurrentBrightness();
		}
	}
}


/*!
 * The function check the current brigthness
 */
void HardwareInfo::checkCurrentBrightness() {
	myDebug ("HardwareInfo::checkCurrentBrightness");

	if (brightness) {
		int retval;
		// get the current level via GetBrightness
		if (dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, *udis["laptop_panel"], HAL_LPANEL_IFACE, 	
						    "GetBrightness", &retval, DBUS_TYPE_UINT32,
						    DBUS_TYPE_INVALID ) ) {
			currentBrightnessLevel = retval;
		}
	}
}


/*!
 * The function initialise the hardware information and collect all
 * initial information from HAL.
 * \return boolean with result of the operation
 * \retval true  if successful
 * \retval false else, if a error occurs
 */
bool HardwareInfo::intialiseHWInfo() {
	myDebug ("HardwareInfo::intialiseHWInfo");

	// TODO: - add code to get udis for all related devies, 

	QStringList ret;

	if (!dbus_HAL->isConnectedToDBUS() || !dbus_HAL->isConnectedToHAL()) 
		return false;

	if( dbus_HAL->halFindDeviceByCapability("ac_adapter", &ret)) {
		// there should be normaly only one device, but let be sure
		for ( QStringList::iterator it = ret.begin(); it != ret.end(); ++it ) {
			// we need a deep copy
			udis.insert("acadapter", new QString( *it ));
			if (!allUDIs.contains( *it ))
				allUDIs.append( *it );
			checkACAdapterState();
		}
	}
	
	ret.clear();

	if( dbus_HAL->halFindDeviceByString("button.type", "lid", &ret)) {
		// there should be normaly only one device, but let be sure
		for ( QStringList::iterator it = ret.begin(); it != ret.end(); ++it ) {
			// we need a deep copy
			udis.insert("lidclose", new QString( *it ));
			if (!allUDIs.contains( *it ))
				allUDIs.append( *it );
			checkLidcloseState();
		}
	}

	ret.clear();

	// find batteries and fill battery information
	if( dbus_HAL->halFindDeviceByCapability("battery", &ret)) {
		if (!ret.isEmpty()) {
			// there should be normaly only one device, but let be sure
			for ( QStringList::iterator it = ret.begin(); it != ret.end(); ++it ) {
				if (!allUDIs.contains( *it ))
					allUDIs.append( *it );
				BatteryList.append( new Battery(dbus_HAL, *it) );
			}
		
			// connect to signals for primary batteries:
			Battery *bat;
			for (bat = BatteryList.first(); bat; bat = BatteryList.next() ) {
				if (bat->getType() == BAT_PRIMARY) {
					connect(bat, SIGNAL(changedBattery()),this, SLOT(updatePrimaryBatteries()));
				}
			}
		}
	}

	return true;
}

/*!
 * The function checks the state of the AC adapter.
 */
void HardwareInfo::checkACAdapterState() {
	myDebug ("HardwareInfo::checkACAdapterState");

	if ( udis["acadapter"] ) {
		bool _state;

		if (dbus_HAL->halGetPropertyBool(*udis["acadapter"] , "ac_adapter.present", &_state )) {
			if (_state != acadapter) {
				acadapter = _state;
				update_info_ac_changed = true;
				emit ACStatus( acadapter );
			} else {
				update_info_ac_changed = false;
			}
		} else {
			// we use true as default e.g. for workstations
			acadapter = true;
		}
	}
}

/*!
 * The function checks the state of the Lidclose button.
 */
void HardwareInfo::checkLidcloseState() {
	myDebug ("HardwareInfo::checkLidcloseState");

	if ( udis["lidclose"] ) {
		bool _state;

		if (dbus_HAL->halGetPropertyBool(*udis["lidclose"] , "button.state.value", &_state )) {
			if (_state != lidclose) {
				lidclose = _state;
				emit lidcloseStatus( lidclose );
			}
		} else {
			lidclose = false;
		}
	}
}

/*!
 * This funtion is used to call a update of a battery value for a given 
 * UDI and the given changed property
 * \param udi 		QString with the UDI of the battery
 * \param property	QString with the changed property
 */
void HardwareInfo::updateBatteryValues (QString udi, QString property) {
	myDebug ("HardwareInfo::updateBatteryValues");

	if (!udi.isEmpty() && allUDIs.contains( udi )) {
		// find effected battery object
		Battery *bat;
		for (bat = BatteryList.first(); bat; bat = BatteryList.next() ) {
			if (udi.startsWith( bat->getUdi())) {
				// found a battery with udi
				bat->updateProperty(udi, property);
			}
		}
	} else {
		myDebug("UDI is empty or not in the list of monitored devices: %s", udi.latin1());
	}
	return;
}

/*!
 * This function refresh the information for the primary battery collection.
 */
void HardwareInfo::updatePrimaryBatteries () {
	myDebug ("HardwareInfo::updateOverallBattery");

	if (!BatteryList.isEmpty()) {
		if (primaryBatteries->getNumBatteries() < 1) {
			setPrimaryBatteriesWarningLevel();
			primaryBatteries->refreshInfo( BatteryList );
			connect(primaryBatteries, SIGNAL(batteryChanged()), this, 
				SLOT(setPrimaryBatteriesChanges()));
			connect(primaryBatteries, SIGNAL(batteryWarnState(int,int)), this,
				SLOT(emitBatteryWARNState(int,int)));
		} else {
			setPrimaryBatteriesWarningLevel();
			primaryBatteries->refreshInfo( BatteryList );
		}
	} else {
		primaryBatteries = new BatteryCollection(BAT_PRIMARY);
	}
}

/*!
 * This function set the change status for the primary battery collection
 */
void HardwareInfo::setPrimaryBatteriesChanges () {
	myDebug ("HardwareInfo::setPrimaryBatteriesChanges");

	update_info_primBattery_changed = true;
	emit primaryBatteryChanged();
}

/*!
 * This slot emit a signal if a warning state of a battery reached
 */
void HardwareInfo::emitBatteryWARNState (int type, int state) {
	myDebug ("HardwareInfo::emitBatteryWARNState");

	if (type == BAT_PRIMARY) 
		emit primaryBatteryChanged();
	else
		emit generalDataChanged();

	emit batteryWARNState(type, state);
}

// --> init HW information section -- END <---
// --> HAL method call (trigger actions) section -- START <---

/*!
 * Function to trigger a suspend via HAL 
 * \param suspend 	enum of suspend_type with the requested suspend
 * \return boolean with result of the operation
 * \retval true  	if successful
 * \retval false 	else, if a error occurs
 */
bool HardwareInfo::suspend( suspend_type suspend ) {
	myDebug ("HardwareInfo::suspend");

	if (dbus_HAL->isConnectedToDBUS() && dbus_HAL->isConnectedToHAL()) {
		switch (suspend) {
			case SUSPEND2DISK:
				if (suspend_states.suspend2disk && (suspend_states.suspend2disk_allowed != 0)) {
					if (dbus_HAL->dbusMethodCallSuspend("Hibernate")) {
						return true;
					} else {
						return false;
					}
				} else {
					if ( !suspend_states.suspend2disk ) 
						myDebug("The machine does not support suspend to disk.");
					else
						myDebug("Policy forbid user to trigger suspend to disk");

					return false;
				}
				break;
			case SUSPEND2RAM:
				if (suspend_states.suspend2ram && (suspend_states.suspend2ram_allowed != 0)) {
					if (dbus_HAL->dbusMethodCallSuspend("Suspend")) {
						return true;
					} else {
						return false;
					}
				} else {
					if ( !suspend_states.suspend2ram ) 
						myDebug("The machine does not support suspend to ram.");
					else
						myDebug("Policy forbid user to trigger suspend to ram");
					
					return false;
				}
				break;
			case STANDBY:
				if (suspend_states.standby && (suspend_states.standby_allowed != 0)) {
					if (dbus_HAL->dbusMethodCallSuspend("Standby")) {
						return true;
					} else {
						return false;
					}
				} else {
					if ( !suspend_states.standby ) 
						myDebug("The machine does not support standby.");
					else
						myDebug("Policy forbid user to trigger standby");

					return false;
				}
				break;
			default:
				return false;
		}
	}
	return false;
}

/*!
 * Function to set brightness via HAL (if supported by hardware)
 * \param level		Integer with the level to set, (range: 0 - \ref availableBrightnessLevels )
 * \param percent	Integer with the brightness percentage to set
 * \return boolean with result of the operation
 * \retval true  	if successful
 * \retval false 	else, if a error occurs
 */
bool HardwareInfo::setBrightness ( int level, int percent ){
	myDebug ("HardwareInfo::setBrightness(level: '%d', percent: '%d')", level, percent);

	if ((level == -1) && (percent >= 0)) {
		if (percent == 0) {
			level = 0;
		} else if (level >= 98) {
			level = (availableBrightnessLevels - 1);
		} else {
			level = (int)((float)availableBrightnessLevels * ((float)percent/100.0));
			if (level > (availableBrightnessLevels -1))
				level = availableBrightnessLevels -1;
			myDebug ("HardwareInfo::setBrightness - percentage mapped to new level: '%d'", level);
		}
	}

	if (dbus_HAL->isConnectedToDBUS() && dbus_HAL->isConnectedToHAL()) {
		checkBrightness();

		if (!brightness || (level < 0 ) || (level >= availableBrightnessLevels)) {
			myDebug("Change brightness or requested level not supported ");
			return false;
		}
		if (currentBrightnessLevel == level) {
			myDebug ("Brightness level not changed, requested level == current level");
			return true;
		}

		if (dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, *udis["laptop_panel"], 
						    HAL_LPANEL_IFACE, "SetBrightness", 
						    DBUS_TYPE_INT32, &level,
						    DBUS_TYPE_INVALID )) {
			currentBrightnessLevel = level;
			return true;
		} else {
			return false;
		}
	} else {
		return false;
	}
} 

/*!
 * Function to set the CPU frequency policy via HAL. 
 * \param cpufreq 	enum of cpufreq_type with the policy to set
 * \param limit 	integer with range 0 - 100 (only if cpufreq == DYNAMIC)
 * \return boolean with result of the operation
 * \retval true  	if successful
 * \retval false 	else, if a error occurs
 */
bool HardwareInfo::setCPUFreq ( cpufreq_type cpufreq, int limit ) {
	myDebug ("HardwareInfo::suspend(cpufreq_type: '%d')",cpufreq);

	if (!cpuFreq) {
		myDebug ("This machine does not support change the CPU Freq via HAL");
		return false;
	}

	if (dbus_HAL->isConnectedToDBUS() && dbus_HAL->isConnectedToHAL()) {
		if (cpuFreqAllowed  == 0) {
			myDebug("Could not set CPU Freq, this is has not the needed privileges.");
			return false;
		}
		
		if (checkCurrentCPUFreqPolicy() == cpufreq) {
			if (cpufreq == DYNAMIC && !cpuFreqGovernor.startsWith("ondemand")) {
				myDebug("CPU Freq Policy is already DYNAMIC, but not governor "
					"is currently not 'ondemand'. Try to set ondemand governor.");
			} else {
				myDebug("Didn't change Policy, was already set.");
				return true;
			}
		}
		
		char *governor;
		int consider = getAcAdapter();

		switch (cpufreq) {
			case PERFORMANCE:
				governor = "performance";
				if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI, 
								     HAL_CPUFREQ_IFACE, "SetCPUFreqGovernor", 
								     DBUS_TYPE_STRING, &governor, 
								     DBUS_TYPE_INVALID)) {
					myDebug("Could not set CPU Freq to performance policy");
					return false;
				}
				break;
			case DYNAMIC:
				governor = "ondemand";
				if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI, 
								     HAL_CPUFREQ_IFACE, "SetCPUFreqGovernor", 
								     DBUS_TYPE_STRING, &governor,
								     DBUS_TYPE_INVALID) || 
				    (checkCurrentCPUFreqPolicy() != cpufreq)){
					myDebug("Couldn't set CPUFreq to ondemand, try now userspace");
					governor = "userspace";
					if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI,
									     HAL_CPUFREQ_IFACE,
									     "SetCPUFreqGovernor", 
									     DBUS_TYPE_STRING, &governor,
									     DBUS_TYPE_INVALID) ||
					    (checkCurrentCPUFreqPolicy() != cpufreq)) {
						myDebug("Couldn't set CPUFreq to userspace, try conservative");
						governor = "conservative";
						if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE,
										     HAL_COMPUTER_UDI,
										     HAL_CPUFREQ_IFACE,
										     "SetCPUFreqGovernor", 
										     DBUS_TYPE_STRING, &governor,
										     DBUS_TYPE_INVALID) ||
						    (checkCurrentCPUFreqPolicy() != cpufreq)) {
							myDebug("Couldn't set CPUFreq to DYNAMIC, not to "
								"ondemand, userspace nor to conservative");
							return false;
						}
					}
				}
			
				// set dynamic performance limit
				if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI,
								     HAL_CPUFREQ_IFACE,
								     "SetCPUFreqPerformance", 
								     DBUS_TYPE_INT32, &limit,
								     DBUS_TYPE_INVALID)) {
					myDebug("Could not call/set SetCPUFreqPerformance with value: "
					        "%d on HAL", limit);
				}

				// correct set ondemand
				if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI,
								     HAL_CPUFREQ_IFACE,
								     "SetCPUFreqConsiderNice",
								     DBUS_TYPE_BOOLEAN, &consider,
								     DBUS_TYPE_INVALID)) {
					myDebug ("Couldn't set SetCPUFreqConsiderNice for DYNAMIC");
				}

				break;
			case POWERSAVE:
				governor = "powersave";
				if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI, 
								     HAL_CPUFREQ_IFACE, "SetCPUFreqGovernor", 
								     DBUS_TYPE_STRING, &governor, 
								     DBUS_TYPE_INVALID)) {
					myDebug("Could not set CPU Freq to performance policy");
					return false;
				}
				break;
			default:
				myDebug("Unknown cpufreq_type: %d", cpufreq);
				return false;
				break;
		}
		
		// check if the policy was really set (and emit signal)
		if (checkCurrentCPUFreqPolicy() == cpufreq) {
//			update_info_cpufreq_policy_changed = true;
//			emit currentCPUFreqPolicyChanged();
			return true;
		} else {
			return false;
		}
	} else {	
		return false;
	}
}

/*!
 * Function to set the powersave mode (incl. e.g. disk settings) via HAL. 
 * \param  on		boolean which tell if enable/disable powersave mode
 * \return boolean with result of the operation
 * \retval true  	if successful
 * \retval false 	else, if a error occurs
 */
bool HardwareInfo::setPowerSave( bool on ) {
	myDebug ("HardwareInfo::setPowerSave");

	if (dbus_HAL->isConnectedToDBUS() && dbus_HAL->isConnectedToHAL()) {
		int _tmp = (int) on;

		if (!dbus_HAL->dbusSystemMethodCall( HAL_SERVICE, HAL_COMPUTER_UDI, HAL_PM_IFACE,
						     "SetPowerSave", DBUS_TYPE_BOOLEAN, &_tmp,
						     DBUS_TYPE_INVALID)) {
			myDebug("Could not call/set SetPowerSave on HAL");
			return false;
		} else {
			return true;
		}
	} else {
		return false;
	}
}

// --> HAL method call (trigger actions) section -- END <---
// --> get private members section -- START <---

/*!
 * The function return the current state of the ac adapter.
 * \return boolean with the current state 
 * \retval true 	if adapter is present/connected or unknown
 * \retval false 	if not 
 */
bool HardwareInfo::getAcAdapter() const {
	return acadapter;
}

/*!
 * The function return the current state of the lidclose button.
 * \return boolean with the current state 
 * \retval true 	if the lid is closed
 * \retval false 	if the lid is opend
 */
bool HardwareInfo::getLidclose() const {
	return lidclose;
}

/*!
 * The function return the maximal available brightness level
 * \return Integer with max level or -1 if not supported
 */
int HardwareInfo::getMaxBrightnessLevel() const {
	if (brightness)
		return availableBrightnessLevels;
	else
		return -1;
}

/*!
 * The function return the current brightness level
 * \return Integer with max level or -1 if not supported or unkown
 */
int HardwareInfo::getCurrentBrightnessLevel() const {
	if (brightness)
		return currentBrightnessLevel;
	else
		return -1;
}

/*!
 * The function return the current set CPU Frequency Policy
 * \return Integer with currently set Policy or -1 if not supported or unkown
 */
int HardwareInfo::getCurrentCPUFreqPolicy() const {
	return currentCPUFreqPolicy;
}

/*!
 * The function return information if the system support the different
 * suspend/standby methodes and if the user can call them.
 * \return struct with information from \ref suspend_states
 * TODO: check if we maybe should replace this by more different functions
 */
SuspendStates HardwareInfo::getSuspendSupport() const {
	return suspend_states;
}

/*!
 * The function return a pointer to the battery collection of primary batteries.
 * \return BatteryCollection with type == PRIMARY
 */
BatteryCollection* HardwareInfo::getPrimaryBatteries() const {
	return primaryBatteries;
}

/*!
 * The function return all batteries
 * \return QPtrList<Battery>
 */
QPtrList<Battery> HardwareInfo::getAllBatteries() const {
	return BatteryList;
}


/*!
 * The function return the status of \ref laptop.
 * \return boolean with info if machine is a laptop 
 * \retval true 	if a laptop
 * \retval false 	else/if not a laptop
 */
bool HardwareInfo::isLaptop() const {
	return laptop;
}

/*!
 * The function return info if there is a working connection to D-Bus and HAL.
 * This mean if we get hardwareinformation
 * \return boolean with info if D-Bus and HAL work
 * \retval true 	if connected
 * \retval false 	if not connected
 */
bool HardwareInfo::isOnline() const {
	return (!dbus_terminated && !hal_terminated);
}

/*!
 * The function return the status of \ref has_ACPI.
 * \return boolean with info if machine support ACPI 
 * \retval true 	if support ACPI
 * \retval false 	else
 */
bool HardwareInfo::hasACPI() const {
	return has_ACPI;
}

/*!
 * The function return the status of \ref has_APM.
 * \return boolean with info if machine support APM
 * \retval true 	if support APM
 * \retval false 	else
 */
bool HardwareInfo::hasAPM() const {
	return has_APM;
}

/*!
 * The function return the status of \ref has_PMU.
 * \return boolean with info if machine support PMU 
 * \retval true 	if support PMU
 * \retval false 	else
 */
bool HardwareInfo::hasPMU() const {
	return has_PMU;
}

/*!
 * The function return the status of \ref brightness.
 * \return boolean with info if machine support brightness changes via HAL
 * \retval true 	if support brightness changes
 * \retval false 	else
 */
bool HardwareInfo::supportBrightness() const {
	return brightness;
}

/*!
 * The function return the status of \ref cpuFreq.
 * \return boolean with info if machine support change the CPU frequency via HAL
 * \retval true 	if support brightness changes
 * \retval false 	else
 */
bool HardwareInfo::supportCPUFreq() const {
	return cpuFreq;
}

/*!
 * The function return \ref cpuFreqAllowed and tell by this if the user is allowed
 * to change the CPU Freq.
 * \return \ref cpuFreqAllowed
 * \retval	0 allowed
 * \retval	0 not allowed
 * \retval	-1 unknown
 */
int HardwareInfo::isCpuFreqAllowed () const {
	return cpuFreqAllowed;
}

// --> get private members section -- END <---
