/***************************************************************************
                          directconnectionbase.h -  description
                             -------------------
    begin                : Tue 12 27 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef DIRECTCONNECTIONBASE_H
#define DIRECTCONNECTIONBASE_H

#include <qdatetime.h>
#include <qobject.h>
#include <qtimer.h>

#include "../kmessbuffer.h"

class KExtendedSocket;

/**
 * The class for receiving and sending files.
 *
 * @author Diederik van der Boor
 * @ingroup NetworkExtra
 */
class DirectConnectionBase : public QObject
{
  Q_OBJECT

  public:
    // The constructor
                           DirectConnectionBase(QObject *parent = 0, const char *name = 0);
    // The destructor
    virtual               ~DirectConnectionBase();

    // Close the connection
    virtual void           closeConnection();
    // Register a slot which will be called each time the write buffer is empty
    void                   connectWriteHandler(QObject *receiver, const char *slot);
    // Unregister the slot which will be called each time the write buffer is empty
    void                   disconnectWriteHandler(QObject *receiver, const char *slot);
    // Return true if the last write action failed.
    bool                   hasLastWriteFailed() const;
    // Return true when the write buffer is full at the moment
    bool                   hasTemporaryWriteError() const;
    // Find out if the connection has been inactive since 15 minutes
    bool                   hasTimedOut() const;
    // Initialize the connection, return true when this was succesful
    virtual bool           initialize();
    // Return whether the connection login was successful.
    virtual bool           isAuthorized() const;
    // Return true if a connection is active
    bool                   isConnected() const;
    // Return true if this class acts as server, false if it acts as client.
    bool                   isServer() const;
    // Return true when a write handler is connnected.
    bool                   isWriteHandlerConnected() const;
    // Get the server port that will be used
    int                    getLocalServerPort();
    // Get the remote ip the socket is connected with.
    QString                getRemoteIp() const;
    // Get the remote port the socket is connected with.
    QString                getRemotePort() const;
    // Return the error description
    QString                getSocketError() const;
    // Connect to a host
    bool                   openConnection(const QString &ipAddress, int port, bool async = false);
    // Wait for an incoming connection
    bool                   openServerPort();

  protected slots:
    // This is called when the connection is established
    virtual void           slotConnectionEstablished();
    // This is called when the connection could not be made.
    virtual void           slotConnectionFailed();
    // This is called when data is received from the socket.
    virtual void           slotDataReceived() = 0;

  private slots:
    // Accept incoming connections on the socket.
    void                   slotAcceptConnection();
    // A timeout occured to connect a socket
    void                   slotConnectionTimeout();
    // Signal that the connection was closed
    void                   slotSocketClosed( int state );
    // Signal that the connection was established
    void                   slotSocketConnected();
    // Signal that the connection could not be made.
    virtual void           slotSocketFailed();
    // Slot called when the socket is ready to write data.
    void                   slotSocketReadyWrite();

  protected: // Protected methods
    // Close and delete the server socket
    bool                   closeServerSocket();
    // Return the number of bytes which are already received in the local buffer
    int                    getAvailableBytes() const;
    // Return the name of the locally opened port.
    QString                getListeningServiceName() const;
    // Verify how many bytes the read buffer has. Note this actually reads the data to test it.
    int                    peekBlock( const uint size );
    // Read data from the socket
    int                    readBlock( char *buffer, const uint size );
    // Read data from the socket (uses the QByteArray size() as block size)
    int                    readBlock( QByteArray &buffer, const uint maxSize = 0, const uint offset = 0 );
    // Mark the remote host as authorized (usually after the handshake was successful)
    void                   setAuthorized(bool authorized);
    // Write data to the socket
    bool                   writeBlock( const char *block, const uint size );
    // Write data to the socket
    bool                   writeBlock( const QByteArray &block );

  private:  // Private methods
    // Close and delete the client socket
    bool                   closeClientSocket();

  protected: // Protected attributes
    // Time of last activity
    QTime                  lastActivity_;

  private: // Private attributes
    // Additional write buffer when KExtendedSocket reports it can't write all data.
    KMessBuffer            additionalWriteBuffer_;
    // Whether the connection handshake was successful
    bool                   authorized_;
    // The location openConnection() is connecting to, for debugging
    QString                connectingTo_;
    // The timeout handling for openConnection()
    QTimer                 connectionTimer_;
    // Whether the class acts as server or client
    bool                   isServer_;
    // Whether the last write action failed.
    bool                   lastWriteFailed_;
    // The server socket which listens for incoming connections.
    KExtendedSocket       *server_;
    // The server port this class will use
    int                    serverPort_;
    // The next server port used with a openServerPort() call.
    static int             nextServerPort_;
    // The socket over which data is sent or received.
    KExtendedSocket       *socket_;
    // Whether an timeout occured
    bool                   timeout_;
    // True if the user cancelled the session
    bool                   userCancelled_;
    // The number of write handlers.
    int                    writeHandlerCount_;

  signals:
    // Signal that the connection was autorized
    void                   connectionAuthorized();
    // Signal that the connection was closed
    void                   connectionClosed();
    // Signal that the connection was established
    void                   connectionEstablished();
    // Signal that the connection was failed
    void                   connectionFailed();
    // Signal that the socket is ready to write data (use connectWriteHandler() instead)
    void                   writeHandlerReady();
};

#endif
