/***************************************************************************
                          transferentry.cpp  -  description
                             -------------------
    begin                : Wed Nov 3 2004
    copyright            : (C) 2004 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "transferentryinterface.h"
#include "transferentry.h"

#include "../kmessdebug.h"

#include <qlabel.h>
#include <qimage.h>
#include <qpixmap.h>

#include <klocale.h>
#include <kdebug.h>

#include <kurllabel.h>
#include <kprogress.h>
#include <krun.h>

#include <kurl.h>
#include <kmimetype.h>
#include <kiconloader.h>
#include <ksqueezedtextlabel.h>


// Constructor
TransferEntry::TransferEntry( QWidget *parent, const QString filename, uint filesize,
                              bool incoming, const QImage preview)
: TransferEntryInterface(parent, "TransferEntry"),
 isDone_(false),
 incoming_(incoming),
 filename_(filename),
 filesize_(filesize),
 preview_(preview),
 previousPercentage_(-1),
 transferID_(-1)
{
#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::TransferEntry() - Initializing, file='" << filename_ 
            << "', preview=" << ( preview_.isNull() ? "null" : "set" ) << "." << endl;
#endif

  // Set pixmap and filename
  if( preview_.isNull() )
  {
    // Get a default icon pixmap
    KIconLoader *loader    = KGlobal::iconLoader();
    QString      iconTitle = KMimeType::iconForURL( KURL( filename_) );
    preview_               = loader->loadIcon( iconTitle, KIcon::NoGroup, iconLabel_->width() );

    if( ! preview_.isNull() )
    {
      iconLabel_->setPixmap( preview_ );
    }
  }
  else
  {
    // Scale down if needed. Don't scale large images up
    if( preview_.width()  > iconLabel_->width()
    ||  preview_.height() > iconLabel_->height() )
    {
      preview_ = preview_.smoothScale( iconLabel_->width(), iconLabel_->height(), QImage::ScaleMin );
    }

    iconLabel_->setPixmap( preview_ );
  }

  filenameLabel_->setText("<b>" + filename_ + "</b>");

  // Enable urls?
  if(incoming_)
  {
    // Can't open it yet.
    openLabel_->hide();
    openLabel_->setEnabled(false);
  }

  cancelLabel_->setEnabled(true);
}



// Destructor
TransferEntry::~TransferEntry()
{
#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "DESTROYED TransferEntry [file=" << filename_ << "]" << endl;
#endif
}



// Called when cancel is clicked.
void TransferEntry::cancelClicked()
{
#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::cancelClicked() - Notifying listeners." << endl;
#endif

  emit cancelTransfer( transferID_ );

  if(! isDone_)
  {
    // If a slot didn't call failTransfer(),
    // we do this with the correct (!) status message.
    failTransfer( i18n("Cancelled") );
  }
}



// Indicate the transfer is complete or cancelled.
bool TransferEntry::isDone() const
{
  return isDone_;
}



// Return the file name
QString TransferEntry::getFileName() const
{
  return filename_;
}



// Mark the transfer as failed
void TransferEntry::failTransfer(const QString &message)
{
  if( isDone_ )
  {
#ifdef KMESSDEBUG_TRANSFERENTRY
    kdDebug() << "TransferEntry::failTransfer() - already done." << endl;
#endif
    return;
  }

#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::failTransfer() - Displaying error message." << endl;
#endif

  // Update the widgets
  if(message == 0 || message.isEmpty())
  {
    statusLabel_->setText( i18n("Failed!") );
  }
  else
  {
    statusLabel_->setText(message);
  }

  // Make sure the widget does not resize when the progressbar hides
  setMinimumHeight( height() );

  progressBar_->hide();
  cancelLabel_->setEnabled(false);
  cancelLabel_->hide();

  isDone_ = true;
}



// Mark the transfer as complete
void TransferEntry::finishTransfer()
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif

  if( isDone_ )
  {
#ifdef KMESSDEBUG_TRANSFERENTRY
    kdDebug() << "TransferEntry::finishTransfer() - already done." << endl;
#endif
    return;
  }

#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::finishTransfer() - Displaying success message." << endl;
#endif

  // Make sure the widget does not resize when the progressbar hides
  setMinimumHeight( height() );

  // Update the widgets
  statusLabel_->setText( i18n("Completed") );
  progressBar_->hide();
  cancelLabel_->setEnabled(false);
  cancelLabel_->hide();
  openLabel_->setEnabled(true);
  openLabel_->show();

  isDone_ = true;
}



// Called when open is clicked
void TransferEntry::openClicked()
{
  if( filename_.startsWith("/") )
  {
    // Using "file://" causes problems.
    // It makes KDE think we want a smb //hostname/path file. A bit weird though.
    new KRun("file:"  + filename_); // This class has "autodelete".
  }
  else
  {
    new KRun("file:/" + filename_); // This class has "autodelete".
  }
}



// Set the transfer ID to identify ourselves to the Transfer Window
void TransferEntry::setTransferID( int transferID )
{
  transferID_ = transferID;
}


// Convert a string to some more readable form
QString TransferEntry::toReadableBytes(uint bytes)
{
  QString format;
  if(bytes > 1048576)
  {
    // Using '%.2f' instead of '%.1f' removes the ".0" part, but it's less pretty.
    format.sprintf("%.1f", (double) bytes / 1048576.0);
    return i18n("%1 MB").arg(format);
  }
  else if(bytes > 1024)
  {
    format.sprintf("%.1f", (double) bytes / 1024.0);
    return i18n("%1 kB").arg(format);
  }
  else
  {
    return i18n("%1 bytes").arg(bytes);
  }
}


// Set a status message
void TransferEntry::setStatusMessage(const QString &message)
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif

  if( isDone_ )
  {
#ifdef KMESSDEBUG_TRANSFERENTRY
    kdDebug() << "TransferEntry::setStatusMessage() - already done." << endl;
#endif
    return;
  }

#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::setStatusMessage() - displaying '" << message << "'." << endl;
#endif

  statusLabel_->setText(message);
}



// Update the progress bar
void TransferEntry::updateProgress( uint bytesTransferred )
{
#ifdef KMESSTEST
  ASSERT( ! isDone_ );
#endif

  if( isDone_ )
  {
#ifdef KMESSDEBUG_TRANSFERENTRY
    kdDebug() << "TransferEntry::updateProgress() - already done." << endl;
#endif
    return;
  }


   int percent = (int) ( ( (double) bytesTransferred * 1000.0 ) / (double) filesize_ );

  // Avoid useless repaints
  if( percent == previousPercentage_ )
  {
    return;
  }

#ifdef KMESSDEBUG_TRANSFERENTRY
  kdDebug() << "TransferEntry::updateProgress() - transfer is at " << ((float)percent/10) << " percent." << endl;
#endif

  progressBar_->setProgress( percent );

  QString statusLabel;
  QString transfered = toReadableBytes(bytesTransferred);
  QString complete   = toReadableBytes(filesize_);

  previousPercentage_ = percent;

  if( incoming_ )
  {
    statusLabel = i18n("%1 of %2 received");
  }
  else
  {
    statusLabel = i18n("%1 of %2 sent");
  }

  statusLabel_->setText( statusLabel.arg(transfered).arg(complete) );
}



#include "transferentry.moc"
