/***************************************************************************
                                qscoord.h
                             -------------------                                         
    begin                : 01-January-2000
    copyright            : (C) 2000 by Kamil Dobkowski                         
    email                : kamildobk@poczta.onet.pl                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef QSCOORD_H
#define QSCOORD_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include<qwindowdefs.h>
#include<qrect.h>
#include<qpoint.h>


/**
  * \brief  Base class for point objects.
  *
  * @author Kamil Dobkowski
  */
class QSCoord {
  public:
  static inline double mmToPixels( double mm, double dpi=72.0 ) { return mm*dpi/25.4; }
  static inline double pixelsToMM( double pixels, double dpi=72.0 ) { return pixels*25.4/dpi; }
  static inline double pointsToPixels( double points, double dpi=72.0 ) { return points*dpi/72.0; }
  static inline double pixelsToPoints( double pixels, double dpi=72.0 ) { return pixels*72.0/dpi; }
 };


//-------------------------------------------------------------//

/**
 * \brief The point on the screen.
 *
 * Used internally.
 * @author Kamil Dobkowski
 */
class QSPt2 : public QSCoord {
      public:
       // exactly as defined in QPoint
       #if defined(_OS_MAC_)
        QCOORD y;
        QCOORD x;
       #else
        QCOORD x;
        QCOORD y;
       #endif

       explicit QSPt2( const QPoint& p ) {
		set( p.x(), p.y() );
		}	
       QSPt2( int newX=0, int newY=0 ) {
		set( newX, newY );
		}
      ~QSPt2() {
		}
       QSPt2 &set( int newX, int newY ) {
		x=newX;
		y=newY;
		return *this;
		}
       QPoint point() const {
		return QPoint( x, y );
		}
       //friend ostream& operator<<( ostream& s, const QSPt2& p );
       friend QSPt2 operator+ ( const QSPt2& p1, const QSPt2& p2 );
       friend QSPt2 operator- ( const QSPt2& p1, const QSPt2& p2 );
       friend bool operator==( const QSPt2& p1, const QSPt2& p2 );
       friend bool operator!=( const QSPt2& p1, const QSPt2& p2 );
      };

//-------------------------------------------------------------//

/**
  * \brief The point in the 3D space.
  *
  * Used internally.
  * @author Kamil Dobkowski
  */
class QSPt3 : public QSCoord {
        public:

         int x;
         int y;
         int z;

         QSPt3( int newX=0, int newY=0, int newZ=0 ) {
		set( newX, newY, newZ );
		}
        ~QSPt3() {
		}
         QSPt3 &set( int newX, int newY, int newZ ) {
                        x=newX;
                        y=newY;
                        z=newZ;
                        return *this;
                       }
         //friend ostream& operator<<( ostream& s, const QSPt3& p );
         friend QSPt3 operator+ ( const QSPt3& p1, const QSPt3& p2 );
         friend QSPt3 operator- ( const QSPt3& p1, const QSPt3& p2 );
         friend  bool operator==( const QSPt3& p1, const QSPt3& p2 );
         friend  bool operator!=( const QSPt3& p1, const QSPt3& p2 );
       };

//-------------------------------------------------------------//

/**
  * \brief The point on the screen with the floating-point coordinates.
  *
  * Used internally.
  * @author Kamil Dobkowski
  */
class QSPt2f : public QSCoord {
      public:
       double x;
       double y;

       explicit QSPt2f( const QPoint& p ) {
		set( p.x(), p.y() );
		}
       QSPt2f( double newX=0.0, double newY=0.0 ) {
		set( newX, newY );
		}
      ~QSPt2f() {
		}
       QSPt2f &set( double newX, double newY ) {
                        x=newX;
                        y=newY;
                        return *this;
                       }
       QPoint point() const {
		return QPoint( int(x+0.5), int(y+0.5) );
		}
       //friend ostream& operator<<( ostream& s, const QSPt2f& p );
       friend QSPt2f operator+ ( const QSPt2f& p1, const QSPt2f& p2 );
       friend QSPt2f operator- ( const QSPt2f& p1, const QSPt2f& p2 );
       friend   bool operator==( const QSPt2f& p1, const QSPt2f& p2 );
       friend   bool operator!=( const QSPt2f& p1, const QSPt2f& p2 );
      };

//-------------------------------------------------------------//

/**
  * \brief The point in th 3D space with the floating-point coordinates.
  *
  * Used internally.
  * @author Kamil Dobkowski
  */
class QSPt3f : public QSCoord {
        public:

         double x;
         double y;
         double z;

         QSPt3f( double newX=0.0, double newY=0.0, double newZ=0.0 ) {
		set( newX, newY, newZ );
		}
        ~QSPt3f() {
		}
         QSPt3f &set( double newX, double newY, double newZ ) {
                        x=newX;
                        y=newY;
                        z=newZ;
                        return *this;
                       }
         //friend ostream& operator<<( ostream& s, const QSPt3f& p );
         friend QSPt3f operator+ ( const QSPt3f& p1, const QSPt3f& p2 );
         friend QSPt3f operator- ( const QSPt3f& p1, const QSPt3f& p2 );
         friend   bool operator==( const QSPt3f& p1, const QSPt3f& p2 );
         friend   bool operator!=( const QSPt3f& p1, const QSPt3f& p2 );
       };

//-------------------------------------------------------------//


/**
  * \brief Rectangle with floating point coordinates.
  */
class QSRectf : public QSCoord
 {
  public:
	 /**
	   * Position of this rectangle.
	   */
	 QSPt2f pos;
	 /**
	   * Size of this rectangle.
	   */
	 QSPt2f size;
         /**
	   * Constructor
	   */
	 QSRectf() {
		}	
         /**
	   * Constructor
	   */
	 explicit QSRectf( const QRect& r ) {
		pos.x = r.left(); pos.y = r.top(); size.x = r.width(); size.y = r.height();
		}
         /**
	   * Constructor
	   */
	 QSRectf( const QSPt2f& p2, const QSPt2f& p1, bool norm ) {
		pos = p1;
		size = p2-p1;
		if ( norm ) { QSRectf r = normalize(); pos = r.pos; size = r.size; }
		}
         /**
	   * Constructor
	   */
	 QSRectf( const QSPt2f& initPos, const QSPt2f& initSize ) {
	 	pos = initPos;
	 	size = initSize;
	 	}
         /**
	   * Constructor
	   */
	 QSRectf( double x, double y, double w, double h ) {
	 	pos.x = x;
	 	pos.y = y;
	 	size.x = w;
	 	size.y = h;
	 	}
         /**
	   * Destructor
	   */
	~QSRectf() {
		}
	 /**
           * P1 is point at 'pos'. This function changes 'pos' and 'size' of this rectangle
	   * but never changes P2
           */	
	 void setP1( const QSPt2f& p ) {
		size = size - (p-pos);
		pos = p;
		}
	 /**
           * P2 is the point at 'pos'+'size'. This function changes 'size' of this rectangle,
	   * but never changes 'size'.
           */
	 void setP2( const QSPt2f& p ) {
		size = p-pos;
		}
	 /**
           * Returns 'pos'.
           */
	 QSPt2f p1() const {
		return pos;
		}
	 /**
           * Returns 'pos'+'size'
           */
	 QSPt2f p2() const {
		return pos+size;
		}
	 /**
           * Sets top-left ( with minimal coordinates ) corner of this rectange. Doesnt change bottom-right corner.
           */
	 void setTopLeft( const QSPt2f& p );
	 /**
           * Sets bottom-right ( with maximum coordinates ) corner of this rectangle to 'p'.
	   * Never changes topLeft corner..
           */
	 void setBottomRight( const QSPt2f& p );
	/**
	  * Returns coordinates of the top-left corner ( the corner closest to (0,0) ).
	  */
	 QSPt2f topLeft() const;
	 /**
	   * Returns coordinates of bottom-left corner ( the corner furthermost from (0,0) )
	   */
	 QSPt2f bottomRight() const;
	 /**
	   * Returns if point is inside the rectangle
	   */
	 bool contains( const QSPt2f& pos ) const;
	 /**
	   * Unites the rectangle with other rectangle.
	   */
	 void unite( const QSRectf& rect );
	 /**
	   * Normalizes rectangle ( makes size > 0 )
	   */
	 QSRectf normalize() const;
	 /**
	   * Normalizes width ( makes it > 0 )
	   */
	 QSRectf normalizeWidth() const;
	 /**
	   * Normalizes height ( makes it > 0 )
	   */
	 QSRectf normalizeHeight() const;
	 /**
	   * Rounds coordinates and converst rectangle to QRect.
	   */
	 QRect rect() const;
         friend bool operator==( const QSRectf& r1, const QSRectf& r2 );
         friend bool operator!=( const QSRectf& r1, const QSRectf& r2 );	
	};

#endif











