#include <string.h>
#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kpworkoutmodel.h"
#include "kpsplitworkout.h"
#include "kptraininglog.h"
#include "kppresetdata.h"
#include "kipina-i18n.h"
#include "kpworkout.h"
#include "kpsplit.h"
#include "kputil.h"



/* GObject stuff */
static void       kp_split_workout_class_init           (GObjectClass *klass,
                                                         gpointer data);
static void       kp_workout_model_init                 (KPWorkoutModelIface *iface);
static void       kp_split_workout_instance_init        (GObject *object,
                                                         gpointer data);
static void       kp_split_workout_instance_finalize    (GObject *object);

/* Splits */  
static gint       split_cmp                             (gconstpointer a,
                                                         gconstpointer b);
static void       kp_split_workout_split_to_xml         (KPSplit *split,
                                                         KPSplitWorkout *wo,
                                                         xmlNodePtr parent);
/* KPCalendarEntry virtual method implementations */
static gchar     *kp_split_workout_get_human_name       (KPCalendarEntry *entry);
static gchar     *kp_split_workout_get_icon_name        (KPCalendarEntry *entry);
static xmlNodePtr kp_split_workout_to_xml               (KPCalendarEntry *entry);
static G_CONST_RETURN gchar *
                  kp_split_workout_to_calendar_string   (KPCalendarEntry *entry);
static gboolean   kp_split_workout_parse                (KPCalendarEntry *entry,
                                                         xmlNodePtr node);

/* KPWorkout things */
static void       kp_split_workout_add_sport            (KPSplitWorkout *wo,
                                                         const gchar *sport);
static void       kp_split_workout_remove_sport         (KPSplitWorkout *wo, 
                                                         const gchar *sport);
static GSList    *kp_split_workout_get_sport_list       (KPWorkout *wo,
                                                         guint *n);

/* KPWorkoutModel */
static guint      kp_split_workout_get_duration         (KPWorkoutModel *wo);
static gdouble    kp_split_workout_get_distance         (KPWorkoutModel *wo);
static gdouble    kp_split_workout_get_effective_speed  (KPWorkoutModel *wo);
static guint      kp_split_workout_get_effective_pace   (KPWorkoutModel *wo);

  
GType
kp_split_workout_get_type ()
{
  static GType kp_split_workout_type = 0;

  if (!kp_split_workout_type) {
    static const GTypeInfo kp_split_workout_info = {
      sizeof (KPSplitWorkoutClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_split_workout_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPSplitWorkout),
      0,
      (GInstanceInitFunc) kp_split_workout_instance_init,
      NULL
    };
    static const GInterfaceInfo workout_model_info = {
      (GInterfaceInitFunc) kp_workout_model_init,
       NULL,
       NULL
    };
    
    kp_split_workout_type = g_type_register_static (KP_TYPE_WORKOUT,
                                                   "KPSplitWorkout",
                                                   &kp_split_workout_info,
                                                    0);
    g_type_add_interface_static (kp_split_workout_type,
                                 KP_TYPE_WORKOUT_MODEL,
                                &workout_model_info);
  }
  return kp_split_workout_type;
}


static void
kp_split_workout_class_init (GObjectClass *klass, gpointer data)
{
  KPCalendarEntryClass *entry_class;
  KPWorkoutClass *workout_class;
  GObjectClass *object_class;

  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_icon_name = kp_split_workout_get_icon_name;
  entry_class->get_human_name = kp_split_workout_get_human_name;
  entry_class->to_string = kp_split_workout_to_calendar_string;
  entry_class->to_xml = kp_split_workout_to_xml;
  entry_class->parse = kp_split_workout_parse;
  
  workout_class = KP_WORKOUT_CLASS (klass);
  workout_class->get_sport_list = kp_split_workout_get_sport_list;
  
  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_split_workout_instance_finalize;
}

static void
kp_workout_model_init (KPWorkoutModelIface *iface)
{
  iface->get_duration = kp_split_workout_get_duration;
  iface->get_distance = kp_split_workout_get_distance;
  iface->get_pace = kp_split_workout_get_effective_pace;
  iface->get_speed = kp_split_workout_get_effective_speed;
}


static void
kp_split_workout_instance_init (GObject *object, gpointer data)
{
  KPSplitWorkout *wo;

  wo = KP_SPLIT_WORKOUT (object);
}

static void
kp_split_workout_instance_finalize (GObject *object)
{
  /*GObjectClass *parent_class;*/

  /*parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);*/
  /* TODO: Why does the above cause crash? */
}

/**
 * kp_split_workout_new:
 * 
 * Create a new instance of #KPSplitWorkout.
 * 
 * Returns: A new #KPSplitWorkout.
 */
KPSplitWorkout *
kp_split_workout_new ()
{
  return g_object_new (kp_split_workout_get_type (), NULL);
}

/**
 * kp_split_workout_add_split:
 * @wo: A #KPSplitWorkout
 * @type: Split type
 * @n: number of split
 * @distance: distance
 * @duration: Duration in milliseconds
 * @comment: Some comments or NULL
 *
 * Add a split to @wo.
 * 
 */
void
kp_split_workout_add_split (KPSplitWorkout *wo, const gchar *type,
                            guint n, gdouble distance, guint32 duration,
                            const gchar *comment)
{
  KPSplit *split;

  split = kp_split_new_full (type, n, distance, duration, NULL, comment);
  
  if (g_slist_find_custom (wo->splits, split, (GCompareFunc) split_cmp)) {
    g_warning ("Split with the same number is already in the list!");
    return;
  }
  wo->splits = g_slist_insert_sorted (wo->splits, (gpointer) split, split_cmp);
}


void
kp_split_workout_add_split_struct (KPSplitWorkout *wo, KPSplit *split)
{
  g_return_if_fail (KP_IS_SPLIT_WORKOUT (wo));
  g_return_if_fail (split != NULL);
  
  if (g_slist_find_custom (wo->splits, split, (GCompareFunc) split_cmp)) {
    g_warning ("Split with the same number is already in the list!");
    return;
  }
  wo->splits = g_slist_insert_sorted (wo->splits, (gpointer) split, split_cmp);
  if (split->sport)
    kp_split_workout_add_sport (wo, split->sport);
}


void
kp_split_workout_remove_split (KPSplitWorkout *wo, guint split_n)
{
  GSList *list, *old;
  GSList *node;
  gboolean still_exists = FALSE;
  gchar *sport;
  
  g_return_if_fail (KP_IS_SPLIT_WORKOUT (wo));
  
  for (list = wo->splits; list; list = list->next) {
    if (KP_SPLIT (list->data)->n == split_n) {
      old = list;

      sport = g_strdup (KP_SPLIT (old->data)->sport);
      kp_split_free (KP_SPLIT (old->data));
      old->data = NULL;
      wo->splits = g_slist_remove_link (wo->splits, list);
   
      for (node = wo->splits; node; node = node->next) {
        g_return_if_fail (KP_SPLIT (node->data)->sport != NULL);
        
        if (strcmp (KP_SPLIT (node->data)->sport, sport) == 0) {
          still_exists = TRUE;
          break;
        }
      }
      if (still_exists == FALSE) 
        kp_split_workout_remove_sport (wo, sport);
      
      g_free (sport);
      
      return;
    }
  }
  
  g_return_if_reached ();
}

/**
 * kp_split_workout_get_n_splits:
 * @wo: A #KPSplitWorkout
 *
 * Just returns the number of splits of this workout.
 * 
 * Returns: Number of splits.
 */
guint
kp_split_workout_get_n_splits (KPSplitWorkout *wo)
{
  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (wo), 0);
  if (!wo->splits)
    return 0;
  return (g_slist_length (wo->splits));
}


/**
 * kp_split_workout_get_splits:
 * @wo: A #KPSplitWorkout
 *
 * Retrieve list of splits of this workout.
 * 
 * Returns: List of all splits of the workout or NULL if
 *          there are not any splits.
 */
GSList *
kp_split_workout_get_splits (KPSplitWorkout *wo)
{
  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (wo), NULL);
  return wo->splits;
}


static gint
split_cmp (gconstpointer a, gconstpointer b)
{
  const KPSplit *a_;
  const KPSplit *b_;

  a_ = a;
  b_ = b;

  if (a_->n > b_->n)
    return 1;

  if (a_->n < b_->n)
    return -1;

  return 0;
}


/**
 * kp_split_workout_get_distance:
 * @wo: a KPSplitWorkout
 *
 * Returns distance as gdouble.
 **/
static gdouble
kp_split_workout_get_distance (KPWorkoutModel *wo)
{
  gdouble dist = 0.0;
  GSList *item;

  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (wo), 0.0);

  for (item = KP_SPLIT_WORKOUT (wo)->splits; item; item = item->next) 
    dist += KP_SPLIT (item->data)->distance;

  return dist;
}

/**
 * kp_split_workout_get_duration:
 * @wo: A #KPSplitWorkout
 * 
 * Retrieve duration of the kp_split_workout in milliseconds.
 *
 * Returns: Number of milliseconds
 */
static guint
kp_split_workout_get_duration (KPWorkoutModel *wo)
{
  GSList *list;
  guint32 msec;
 
  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (wo), 0); 

  /* Split workout */
  for (list = KP_SPLIT_WORKOUT (wo)->splits, msec = 0; list; list = list->next) 
    msec += KP_SPLIT (list->data)->duration;

  return msec;
}


void
kp_split_workout_get_effective_values (KPSplitWorkout *wo, guint *duration, 
                                       gdouble *distance)
{
  GSList *list;
  guint du;
  gdouble di;
  
  g_return_if_fail (KP_IS_SPLIT_WORKOUT (wo));
  
  for (list = wo->splits, du = 0, di = 0; list; list = list->next)
    if (strcasecmp (KP_SPLIT_TYPE_NORMAL, KP_SPLIT (list->data)->type) == 0) {
      du += KP_SPLIT (list->data)->duration;
      di += KP_SPLIT (list->data)->distance;
    }

  if (duration)
    *duration = du;
  if (distance)
    *distance = di;
}


gdouble
kp_split_workout_get_effective_speed (KPWorkoutModel *wo)
{
  gdouble distance;
  guint duration;
  gdouble dduration;
    
  kp_split_workout_get_effective_values (KP_SPLIT_WORKOUT (wo), &duration, 
                                        &distance);

  dduration = (gdouble) duration;
  
  if (dduration > 0.0 && distance > 0.0) {
    dduration /= (gdouble)(60 * 60 * 1000);
    return (gdouble) (distance / dduration);
  }  
  return 0.0;
}

  
gdouble
kp_split_workout_get_whole_speed (KPSplitWorkout *wo)
{
  gdouble distance;
  gdouble duration;
    
  duration = (gdouble) kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo)); 
  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo)); 

  if (duration > 0.0 && distance > 0.0) {
    duration /= (gdouble)(60 * 60 * 1000);
    return (gdouble) (distance / duration);
  }  
  return 0.0;
}


guint
kp_split_workout_get_effective_pace (KPWorkoutModel *wo)
{
  guint duration;
  gdouble distance;
    
  kp_split_workout_get_effective_values (KP_SPLIT_WORKOUT (wo), &duration, 
                                        &distance);
  
  if (duration > 0 && distance > 0.0) {
    duration = (guint) ((gdouble) duration / (distance));
    duration /= 1000;
    return duration * 1000;
  }

  return 0;
}

  
guint
kp_split_workout_get_whole_pace (KPSplitWorkout *wo)
{
  guint duration;
  gdouble distance;

  duration = kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo));
  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));

  if (duration > 0 && distance > 0.0) {
    duration = (guint) ((gdouble) duration / (distance));
    duration /= 1000;
    return duration * 1000;
  }
  return 0;
}


static void
kp_split_workout_add_sport (KPSplitWorkout *wo, const gchar *sport)
{
  GSList *node;

  g_return_if_fail (KP_IS_SPLIT_WORKOUT (wo));
  g_return_if_fail (sport != NULL);
  
  for (node = KP_WORKOUT (wo)->sports; node; node = node->next) 
    if (strcmp ((const gchar *) node->data, sport) == 0)
      return;
  
  KP_WORKOUT (wo)->sports = g_slist_prepend (KP_WORKOUT (wo)->sports, 
                                             g_strdup (sport));
}

  
static void
kp_split_workout_remove_sport (KPSplitWorkout *wo, const gchar *sport)
{
  GSList *node;

  g_return_if_fail (KP_IS_SPLIT_WORKOUT (wo));
  g_return_if_fail (sport != NULL);
  
  for (node = KP_WORKOUT (wo)->sports; node; node = node->next) 
    if (strcmp ((const gchar *) node->data, sport) == 0) {
      KP_WORKOUT (wo)->sports = g_slist_remove (KP_WORKOUT (wo)->sports, node->data);
      return;
    }
}


static GSList *
kp_split_workout_get_sport_list (KPWorkout *wo, guint *n)
{
  g_print ("Getting sport from KPSplitWorkout!");

  return KP_WORKOUT (wo)->sports;
}


static gchar *
kp_split_workout_get_icon_name (KPCalendarEntry *entry)
{
  return g_strdup ("spworkout.png");
}

/**
 * Get kp_split_workout as a string that can be passed directly to CalendarView or 
 * something like that.
 *
 * Returns a newly-allocated string that must be freed by the caller.
 */
static G_CONST_RETURN gchar *
kp_split_workout_to_calendar_string (KPCalendarEntry *entry)
{
  KPPresetDataItem *sport;
  KPParam *param;
  gdouble distance;
  GString *string;
  gchar *duration;
  gchar *comment;
  gchar *color;
  KPSplitWorkout *wo;
  const gchar *sport_str;
  gchar *str, *tmp;
  gchar *prefix;
  gchar *esc;
  
  wo = KP_SPLIT_WORKOUT (entry);
  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (wo), NULL);

  distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));

  param = kp_param_list_get_param (KP_WORKOUT (wo)->param_list, "duration");
  duration = (param) ? kp_param_get_as_string (param) : NULL;

  string = g_string_new (NULL);

  sport_str = kp_workout_get_sport (KP_WORKOUT (wo));

  if ((sport = kp_preset_data_get_item (KP_PRESET_DATA_SPORT, sport_str))) {
    color = sport->data;
    sport_str = sport->abbreviation;
  } else 
    color = NULL;
  
  if (distance && duration)
    g_string_printf (string, "<b>%s</b> %.1fkm\n%s\n", sport_str, distance,
                     duration);
  else if (duration) 
    g_string_printf (string, "<b>%s</b>: %s\n", sport_str, duration);
  else 
    g_string_printf (string, "<b>%s</b>: %.1fkm\n", sport_str, distance);

  comment = kp_workout_get_comment (KP_WORKOUT (wo));
  
  if (comment) {
    /* Don't have a problem with < and > */
    esc = g_markup_escape_text (comment, -1);
    string = g_string_append (string, esc);
    g_free (esc);
    g_free (comment);
  }
 
  if (string->str[string->len-1] == '\n')
    g_string_erase (string, string->len-1, 1);
    
  if (duration)
    g_free (duration);

  /* Set the time */
  str = kp_calendar_time_to_time_string (entry->datetime);
  tmp = g_strdup_printf ("<b>%s</b>: ", str);
  g_string_prepend (string, tmp);
  g_free (tmp);
  g_free (str);
 
  /* Set the color */
  if (color) {
    prefix = g_strdup_printf ("<span color=\"%s\">", color);
    g_string_append (string, "</span>");
    g_string_prepend (string, prefix);
    g_free (prefix);
  }
  
  str = string->str;
  g_string_free (string, FALSE);
 
  return str;
}


static void
parse_split (xmlNodePtr node, KPSplitWorkout *wo, gchar *old_type_attr)
{
  KPSplit *split;
  xmlNodePtr child;
  xmlChar *time_str;
  gboolean valid = FALSE;
  gchar *split_type;
  gchar *distance_str;
  gchar *type;
  gchar *attr;
  
  split = kp_split_new ();
  
  attr = (gchar *) xmlGetProp (node, BAD_CAST ("n"));
  if (attr != NULL) {
    split->n = (guint) g_ascii_strtod (attr, NULL);
    xmlFree (attr);
  } else
    return;
  
  for (child = node->children; child != NULL; child = child->next) {
    /** <duration>, <time> is obsolete **/
    if (KP_TAG_MATCH (child, "time")
     || KP_TAG_MATCH (child, "duration")) {

      time_str = xmlNodeGetContent (child);
      split->duration = kp_duration_str_to_ms ((const gchar *)time_str);
      g_free (time_str);
      valid = TRUE;
    }
    /** distance **/
    if (KP_TAG_MATCH (child, "distance")) {
      distance_str = (gchar *) xmlNodeGetContent (child);
      split->distance = (gfloat) g_ascii_strtod (distance_str, NULL);
    }

    /** <comment> **/
    if (KP_TAG_MATCH (child, "comment"))
      split->comment = (gchar *) xmlNodeGetContent (child);
   
    /** <sport> **/
    if (KP_TAG_MATCH (child, "sport"))
      split->sport = (gchar *) xmlNodeGetContent (child);
    
    /** <type> **/
    if (KP_TAG_MATCH (child, "type")) {
      type = (gchar *) xmlNodeGetContent (child);

      if (strcasecmp (type, "warmup") == 0
       || strcasecmp (type, "warm up") == 0)
        split_type = "Warm Up";
      else if (strcasecmp (type, "cooldown") == 0
            || strcasecmp (type, "cool down") == 0)
        split_type = "Cool Down";
      else if (strcasecmp (type, "recovery") == 0)
        split_type = "Recovery";
      else if (strcasecmp (type, "normal") == 0)
        split_type = "Normal";
      else
        g_return_if_reached ();

      split->type = g_strdup (split_type);
      g_free (type);
    }
  }

  if (valid) {
    kp_debug ("Adding a split!");
   
    /* This is for 0.1.x logs */
    if (split->sport == NULL && old_type_attr != NULL)
      split->sport = g_strdup (old_type_attr);
    
    kp_split_workout_add_split_struct (wo, split);
  }
}


/**
 * kp_split_workout_parse:
 * @entry: A #KPCalendarEntry
 * @node: A #xmlNodePtr.
 *
 * Parses a kp_split_workout and returns it. If kp_split_workout's type
 * is interval, interval-params are stored to params-hashtable.
 * Interval-things' keys use "__interval__"-prefix to prevent
 * name-collisions which should be impossible this way.
 *
 * Returns: TRUE if successful or FALSE otherwise.
 */
static gboolean
kp_split_workout_parse (KPCalendarEntry *entry, xmlNodePtr node)
{
  xmlNodePtr child;
  xmlChar *type_str;
  xmlChar *val_str;
  xmlChar *date;
  gchar *old_type_attr;
  xmlChar *name;
  guint msec;
  KPParam *param;

  /* Read the old style log too! */
  old_type_attr = (gchar *) xmlGetProp (node, BAD_CAST ("type"));  
  child = node->children;

  kp_debug ("Parsing a split workout!");
  
  while (child) {
    
    /** <datetime> **/
    if (KP_TAG_MATCH (child, "datetime")) {
      date = xmlNodeGetContent (child);
      if (!kp_calendar_time_set_datetime (KP_CALENDAR_ENTRY (entry)->datetime,
                                          (const gchar *) date))
        goto error;
      g_free (date);
    }

    /** <split> **/
    else if (KP_TAG_MATCH (child, "split")) {
      parse_split (child, KP_SPLIT_WORKOUT (entry), old_type_attr);
    }
    
    /** <param> **/
    else if (KP_TAG_MATCH (child, "param")) {
      val_str = xmlNodeGetContent (child);
      type_str = xmlGetProp (child, BAD_CAST ("type"));

      if (type_str == NULL) {
        g_warning ("Parse error: param must have a type-attribute set!");
        goto error;
      }
      if (val_str == NULL) {
        g_warning ("Parse error: param must have a value!");
        goto error;
      }
      name = xmlGetProp (child, BAD_CAST ("name"));
      kp_param_list_set_string (KP_WORKOUT (entry)->param_list,
                               "default", (const gchar *) name, (const gchar *) val_str);
      g_free (val_str);
      g_free (name);
    }
    else if (KP_TAG_MATCH (child, "text")) {
      /* doesn't need to care about these */    
    } else {
      g_warning ("<workout> has an unknown child: %s", child->name);
    }
    child = child->next;
  }

  /* If the workout is a split workout, duration must be set explicitly */
  if (KP_IS_SPLIT_WORKOUT (entry) && KP_SPLIT_WORKOUT (entry)->splits) {
    msec = kp_workout_model_get_duration (KP_WORKOUT_MODEL (entry));
    
    param = kp_param_new ("duration");
    kp_param_set_time (param, msec);
    kp_param_list_insert (KP_WORKOUT (entry)->param_list, "default", param);
  }

  if (old_type_attr)
    g_free (old_type_attr);
  
  return TRUE;
  
error:
  g_object_unref (entry);
  return FALSE;
}


static gchar *
kp_split_workout_get_human_name (KPCalendarEntry *entry)
{
  return g_strdup (_("Split Workout"));
}


static xmlNodePtr
kp_split_workout_to_xml (KPCalendarEntry *entry)
{
  xmlNodePtr date_child;
  xmlNodePtr node;
  gchar *date;
  GSList *list;

  g_return_val_if_fail (KP_IS_SPLIT_WORKOUT (entry), NULL);

  node = xmlNewNode (NULL, BAD_CAST ("workout"));

/* type-attribute is depreceated in 0.2.x */
#if 0
  if (kp_workout_get_sport (KP_WORKOUT (entry)) != NULL) {
    attr = xmlNewProp (node, "type", kp_workout_get_sport (KP_WORKOUT (entry)));
    g_return_val_if_fail (attr != NULL, NULL);
  }
#endif

  date = kp_workout_get_formatted_date (KP_WORKOUT (entry));
  date_child = xmlNewChild (node, NULL, BAD_CAST ("datetime"), BAD_CAST (date));

  for (list = KP_SPLIT_WORKOUT (entry)->splits; list; list = list->next) {
    kp_split_workout_split_to_xml (KP_SPLIT (list->data), KP_SPLIT_WORKOUT (entry),
                                   node);
  }
  kp_param_list_export_as_xml (KP_WORKOUT (entry)->param_list, node);

  return node;
}

static void
kp_split_workout_split_to_xml (KPSplit *split, KPSplitWorkout *wo,
                               xmlNodePtr parent)
{
  xmlNodePtr node;
  xmlNodePtr child;
  gchar *distance_str;
  gchar *type_str;
  gchar *time_str;
  gchar *n_str;

  time_str = kp_date_mseconds_to_std_string (split->duration);

  distance_str = g_strdup_printf ("%.3f", split->distance);
  n_str = g_strdup_printf ("%u", split->n);

  type_str = g_strstrip (g_ascii_strdown (split->type, -1));
  
  node = xmlNewChild (parent, NULL, BAD_CAST ("split"), NULL);

  child = xmlNewChild (node, NULL, BAD_CAST ("time"), BAD_CAST (time_str));
  child = xmlNewChild (node, NULL, BAD_CAST ("comment"), BAD_CAST (split->comment));
  child = xmlNewChild (node, NULL, BAD_CAST ("type"), BAD_CAST (type_str));
  child = xmlNewChild (node, NULL, BAD_CAST ("distance"), BAD_CAST (distance_str));
  child = xmlNewChild (node, NULL, BAD_CAST ("sport"), BAD_CAST (split->sport));
  
  (void) xmlSetProp (node, BAD_CAST ("n"), BAD_CAST (n_str));

  g_free (type_str);
  g_free (distance_str);
  g_free (time_str);
  g_free (n_str);
}


