#include <string.h>
#include <errno.h>

#include "kphtmldiagram.h"
#include "kphtmlwriter.h"
#include "kphtmlutil.h"
#include "page-year.h"

#include "../kplogstatitem.h"
#include "../kplogstat.h"

#include "../kpworkoutmodel.h"
#include "../kipina-i18n.h"
#include "../kpconv.h"


static KPHtmlNaviData navi_data = 
{
  0, 0, 0,
  { N_("Stats by Sports"), NULL, NULL, NULL, NULL },
  { "index-sport",         NULL, NULL, NULL, NULL }
};


static void
diagram_years (KPHtmlWriter *writer, KPTrainingLog *log, FILE *fp)
{
  KPHtmlDiagram *diag;
  KPLogStatItem *item;
  gdouble *dur_data;
  gdouble *dis_data;
  gchar buf[16];
  TYear *ty;
  GList *node;
  guint n_years;
  guint i;

  if (!(n_years = g_list_length (log->year_list)))
    return;
  
  diag = kp_html_diagram_new (1, n_years);

  dur_data = g_new0 (gdouble, n_years * 2);
  dis_data = g_new0 (gdouble, n_years * 2);

  /* Collect data */ 
  for (i=0, node = log->year_list; node; node = node->next, i++) {
    ty = node->data;
 
    item = kp_log_stat_get_year_stats (writer->stat, ty->year, NULL, NULL);
    g_return_if_fail (item != NULL);
   
    dis_data[i] = item->tot_di;
    dur_data[i] = (gdouble) item->tot_du / 3600000;
    
    g_snprintf (buf, sizeof (buf)-1, "%u", ty->year);
    kp_html_diagram_set_x_hint (diag, i, buf);
    
    kp_log_stat_item_free (item);
  }
  
  /* Distance diagram */
  kp_html_diagram_set_title (diag, _("Distance Statistics by Year"));
  kp_html_diagram_set_x_title (diag, _("Year"));
  kp_html_diagram_set_y_title (diag, _("Distance (km)"));
  kp_html_diagram_set_group_title (diag, 0, _("Distance (km)"));
  kp_html_diagram_set_data (diag, dis_data); 
  kp_html_diagram_write (diag, fp);

  fputs ("<br/>", fp);

  /* Duration diagram */
  kp_html_diagram_set_title (diag, _("Duration Statistics by Year"));
  kp_html_diagram_set_y_title (diag, _("Duration (h)"));
  kp_html_diagram_set_group_title (diag, 0, _("Duration (h)"));
  kp_html_diagram_set_data (diag, dur_data); 
  kp_html_diagram_write (diag, fp);
  kp_html_diagram_free (diag);
  
  return;
}


static void
stats_table_main (KPHtmlWriter *writer, FILE *fp)
{
  KPLogStatItem *item;
  GList *node;
  TYear *ty;

  kp_html_table_head (fp, _("Year"), _("Total Distance"), _("Total Duration"),
                          _("N Workouts"), _("Other Entries"), NULL);
  
  for (node = writer->log->year_list; node != NULL; node = node->next) {
    ty = node->data;
    item = kp_log_stat_get_year_stats (writer->stat, ty->year, NULL, NULL);
    if (item == NULL) {
      g_warning ("Can't get stats for year: %u!\n", ty->year);
      return; 
    }
    
    kp_html_table_row_start (fp);
    kp_html_table_cell (fp, "<a href=\"year-%u.html\"><b>%u</b></a>\n", ty->year, ty->year);
    kp_html_table_cell (fp, "%.2f", item->tot_di);
    kp_html_table_cell (fp, "%.1f h", item->tot_du / 3600000.0f);
    kp_html_table_cell (fp, "%u", item->n_workouts);
    kp_html_table_cell (fp, "%u", item->n_entries - item->n_workouts);
    kp_html_table_row_end (fp);
  
    kp_log_stat_item_free (item);
  }

  /* Total */
  item = kp_log_stat_get_stats (writer->stat);
  kp_html_table_row_start (fp);
  kp_html_table_cell (fp, "<strong>%s</strong>", _("Total [~]"));
  kp_html_table_cell (fp, "<strong>%u</strong>", (guint) item->tot_di);
  kp_html_table_cell (fp, "<strong>%u h</strong>", 
                      item->tot_du / (3600 * 1000));
  kp_html_table_cell (fp, "<strong>%u</strong>", item->n_workouts);
  kp_html_table_cell (fp, "<strong>%u</strong>", 
                      item->n_entries - item->n_workouts);
  kp_html_table_row_end (fp);
  kp_log_stat_item_free (item);

  kp_html_table_foot (fp);
}
  

void
page_index (KPHtmlWriter *writer, GError **err)
{
  GError *tmp_error = NULL;
  const gchar *str;
  FILE *fp;
  
  fp = kp_html_writer_open (writer, "index.html", &tmp_error);

  if (fp == NULL) {
    if (tmp_error) {
      g_warning ("Can't open index.html: %s\n", tmp_error->message);
      g_propagate_error (err, tmp_error);
    } 
    else
      g_warning ("Can't open index.html for writing: %s\n", strerror (errno));
    return;
  }

  /* Navi bar */ 
  kp_html_navi (fp, &navi_data, 0, 0, 0);

  /* Log title */
  str = kp_training_log_get_info_entry (writer->log, "title");
  kp_html_h1 (fp, (str == NULL || strlen (str) == 0) ? _("Untitled Log") : str);

  /* Log description */
  if ((str = kp_training_log_get_info_entry (writer->log, "description")))
    fprintf (fp, "<p>%s</p>", str);
  
  /* Main page table */
  stats_table_main (writer, fp);
  fputs ("<br/>", fp);
 
  /* Diagram */ 
  diagram_years (writer, writer->log, fp);
  fputs ("<br/>", fp);

  kp_html_writer_close (fp);
}

