#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kppresetdata.h"
#include "../kpworkoutmodel.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"

#include "kpnewsplitworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpworkouteditor.h"
#include "kpguiutils.h"
#include "kplistview.h"
#include "kpviewmodel.h"
#include "kpcontexts.h"
#include "kpentryviewmodel.h"
#include "kpentries.h"

/* Callbacks */
static void       kp_list_view_class_init       (KPListViewClass *klass);
static void       kp_list_view_init             (KPListView *lv);
static void       kp_list_view_model_init       (KPViewModelIface *iface);
static void       kp_list_view_finalize         (GObject *object);

/* Callback functions */
static void       view_popup_menu               (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
static gboolean   on_popup_menu                 (GtkWidget *treeview,
                                                 gpointer data);
static gboolean   on_button_press_event         (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 KPListView *lv);
/* Follow log changes.. */
static void       log_connect_signals           (KPTrainingLog *log,
                                                 KPListView *lv);
static void       log_disconnect_signals        (KPTrainingLog *log,
                                                 KPListView *lv);
static void       log_changed                   (KPTrainingLog *log,
                                                 KPListView *lv);
static void       kp_list_view_set_dmy          (KPViewModel *view,
                                                 guint d,
                                                 guint m,
                                                 guint y);
static void       kp_list_view_get_dmy          (KPViewModel *view,
                                                 guint *d,
                                                 guint *m,
                                                 guint *y);
static void       kp_list_view_activate         (KPViewModel *model);
static void       kp_list_view_deactivate       (KPViewModel *model);
static void       kp_list_view_set_log          (KPViewModel *view,
                                                 KPTrainingLog *log);
static void       kp_list_view_unset_log        (KPViewModel *view);
static void       kp_list_view_set_view_type    (KPViewModel *view,
                                                 KPViewModelType type);
static gchar     *kp_list_view_get_icon_name    (KPViewModel *view);
static
KPViewModelType   kp_list_view_get_view_type    (KPViewModel *view);
static void       kp_list_view_update_dates     (KPListView *view);
static void       kp_list_view_update_entries   (KPListView *lv);
static void       kp_list_view_update           (KPListView *lv);

/* KPViewPopupModel */
KPCalendarEntry  *kp_list_view_get_active_entry (KPViewPopupModel *popup_model);
static void       kp_list_view_popup_model_init (KPViewPopupModelIface *iface);


typedef struct KPListViewPrivateData_
{
  GtkWidget        *popup_menu;

  GtkWidget        *popup_mi_properties;
  GtkWidget        *popup_mi_delete;
  GtkWidget        *popup_mi_edit;

  KPEntryPopup     *popup_data;
  
  GList            *entries;
  GtkListStore     *store;

  KPViewModelType   type;
  KPTrainingLog    *log;

  GDate            *date_s;
  GDate            *date_e;
  GDate            *date;
} KPListViewPrivateData;

#define KP_LIST_VIEW_PRIVATE_DATA(widget) (((KPListViewPrivateData*) \
      (KP_LIST_VIEW (widget)->private_data)))

enum {
  COLUMN_DATE,
  COLUMN_SPORT,
  COLUMN_DISTANCE,
  COLUMN_TIME,
  COLUMN_PACE,
  COLUMN_COMMENT,
  COLUMN_OTHER,
  COLUMN_TYPE,
  COLUMN_DESCRIPTION,
  COLUMN_N
};

enum {
  SORT_ID_DATE,
  SORT_ID_SPORT,
  SORT_ID_DISTANCE,
  SORT_ID_TIME,
  SORT_ID_PACE,
  SORT_ID_COMMENT,
  SORT_ID_OTHER,
  SORT_ID_TYPE,
  SORT_ID_DESCRIPTION
};

GType
kp_list_view_get_type (void)
{
  static GType        kp_list_view_type = 0;

  if (!kp_list_view_type) {
    static const GTypeInfo kp_list_view_info = {
      sizeof (KPListViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_list_view_class_init,
      NULL,
      NULL,
      sizeof (KPListView),
      0,
      (GInstanceInitFunc) kp_list_view_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_list_view_model_init,
      NULL,
      NULL
    };
    static const GInterfaceInfo view_popup_model_info = {
      (GInterfaceInitFunc) kp_list_view_popup_model_init,
      NULL,
      NULL
    };
    kp_list_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                               "KPListView",
                                               &kp_list_view_info, 0);
    g_type_add_interface_static (kp_list_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
    g_type_add_interface_static (kp_list_view_type,
                                 KP_TYPE_VIEW_POPUP_MODEL,
                                &view_popup_model_info);
  }
  return kp_list_view_type;
}

static void
kp_list_view_class_init (KPListViewClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_list_view_finalize;
}

static void
str_cell_func (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
               GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
  KPPresetDataItem *sport;
  gchar *sport_name;
  const gchar *color;
  gchar *str;
  gint id;

  id = gtk_tree_view_column_get_sort_column_id (column);
  
  gtk_tree_model_get (model, iter,
                      id, &str,
                      COLUMN_SPORT, &sport_name,
                      -1);

  /* If the entry is not workout, COLUMN_SPORT is empty */
  if (sport_name != NULL) {
    sport = kp_preset_data_get_item (KP_PRESET_DATA_SPORT, sport_name);
    color = (sport) ? sport->data : "black";
  
    g_object_set (renderer, "markup", str, "foreground", color, 
                 "foreground-set", TRUE, NULL);
  }
  g_free (sport_name);
  g_free (str);
}

static void
distance_cell_func (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
                    GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
  KPPresetDataItem *sport;
  const gchar *color;
  gdouble distance;
  gchar buf[16];
  gchar *sport_name;

  gtk_tree_model_get (model, iter,
                      COLUMN_DISTANCE, &distance,
                      COLUMN_SPORT, &sport_name,
                     -1);

  /* If the entry is not workout, COLUMN_SPORT is empty */
  if (sport_name != NULL) {
    sport = kp_preset_data_get_item (KP_PRESET_DATA_SPORT, sport_name);
    color = (sport) ? sport->data : "black";
  
    g_free (sport_name);

    if (distance == (guint) distance)
      g_snprintf (buf, sizeof (buf), "%.0f", distance);
    else
      g_snprintf (buf, sizeof (buf), "%.3f", distance);

    g_object_set (renderer, 
                 "text", (distance) ? buf : "",
                 "foreground", color,
                 "foreground-set", TRUE,
                  NULL);
  }
}

  
static GtkTreeViewColumn *
new_column (KPListView *lv, const gchar *title, guint column,
            GtkTreeCellDataFunc func)
{
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;

  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes (title, ren, "markup", column,
                                                  NULL);
  gtk_tree_view_column_set_sort_column_id (col, column);
  gtk_tree_view_append_column (GTK_TREE_VIEW (lv), col);

  if (func)
    gtk_tree_view_column_set_cell_data_func (col, ren, func, NULL, NULL);

  if (column == COLUMN_DESCRIPTION)
    gtk_tree_view_column_set_visible (col, FALSE);

  gtk_tree_view_column_set_resizable (col, TRUE);

  return col;
}

gint
time_compare_func (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
                   KPListView *lv)
{
  gint32 ms1;
  gint32 ms2;
  gchar *str1;
  gchar *str2;

  g_return_val_if_fail (a != NULL, -1);
  g_return_val_if_fail (b != NULL, -1);

  gtk_tree_model_get (model, a, COLUMN_TIME, &str1, -1);
  gtk_tree_model_get (model, b, COLUMN_TIME, &str2, -1);

  /* This is possible if the column is a comment for example */
  if (str1 == NULL || str2 == NULL) {
    if (str1)
      g_free (str1);
    if (str2)
      g_free (str2);
    return -1;
  }
  
  ms1 = kp_duration_str_to_ms (str1);
  ms2 = kp_duration_str_to_ms (str2);

  if (ms1 > 0 || ms2 > 0)
    kp_debug ("str1: %s, str2: %s, ms1: %u, ms2: %u\n", str1, str2, ms1, ms2);

  g_free (str1);
  g_free (str2);
  
  if (ms1 == ms2)
    return 0;
  
  return (ms1 > ms2) ? 1 : -1;
}

gint
date_compare_func (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, 
                   gpointer data)
{
  static GDate *date1 = NULL;
  static GDate *date2 = NULL;
  gchar *str1;
  gchar *str2;
  guint ret;

  if (!date1)
    date1 = g_date_new ();
  if (!date2)
    date2 = g_date_new ();

  g_return_val_if_fail (a != NULL, -1);
  g_return_val_if_fail (b != NULL, -1);

  gtk_tree_model_get (model, a, COLUMN_DATE, &str1, -1);
  gtk_tree_model_get (model, b, COLUMN_DATE, &str2, -1);

  if (str1 == NULL || str2 == NULL) {
    if (str1)
      g_free (str1);
    if (str2)
      g_free (str2);
    return -1;
  }
  g_date_set_parse (date1, str1);
  g_date_set_parse (date2, str2);

  if (g_date_valid (date1) == FALSE || g_date_valid (date2) == FALSE)
    return 0;

  g_free (str1);
  g_free (str2);

  ret = g_date_compare (date1, date2);

  g_date_clear (date1, 1);
  g_date_clear (date2, 1);

  return ret;
}


static void
kp_list_view_init (KPListView *lv)
{
  KPListViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *tcol;
  GtkTreeSortable *sortable;
  
  lv->private_data = g_new (KPListViewPrivateData, 1);
  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);
  p_data->date_s = g_date_new ();
  p_data->date_e = g_date_new ();
  p_data->date = g_date_new ();
  p_data->log = NULL;
  p_data->type = KP_VIEW_MODEL_TYPE_MONTH;
  p_data->entries = NULL;
  p_data->popup_data = kp_entries_get_popup (KP_VIEW_POPUP_MODEL (lv));
  
  (void) new_column (lv, _("Date"), COLUMN_DATE, str_cell_func);
  (void) new_column (lv, _("Sport"), COLUMN_SPORT, str_cell_func);
  (void) new_column (lv, _("Type"), COLUMN_TYPE, str_cell_func);
  (void) new_column (lv, _("Distance"), COLUMN_DISTANCE, distance_cell_func);
  tcol = new_column (lv, _("Duration"), COLUMN_TIME, str_cell_func);
  (void) new_column (lv, _("Pace"), COLUMN_PACE, str_cell_func);
  (void) new_column (lv, _("Details"), COLUMN_OTHER, str_cell_func);
  (void) new_column (lv, _("Comment"), COLUMN_COMMENT, str_cell_func);
  (void) new_column (lv, _("Description"), COLUMN_DESCRIPTION, str_cell_func);

  
  p_data->store = gtk_list_store_new (COLUMN_N,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_DOUBLE,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (lv), GTK_TREE_MODEL (p_data->store));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (lv), TRUE);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (lv), TRUE);

  /* Time column needs custom sort function */
  sortable = GTK_TREE_SORTABLE (p_data->store);
  gtk_tree_sortable_set_sort_func (sortable,
                                   SORT_ID_TIME,
                                  (GtkTreeIterCompareFunc) time_compare_func,
                                  (gpointer) lv,
                                   NULL);

  /* Date column needs custom sort function */
  gtk_tree_sortable_set_sort_func (sortable,
                                   SORT_ID_DATE,
                                  (GtkTreeIterCompareFunc) date_compare_func,
                                  (gpointer) lv,
                                   NULL);
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (lv));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  
  g_signal_connect (G_OBJECT (lv), "button-press-event",
                    G_CALLBACK (on_button_press_event), lv);
  g_signal_connect (G_OBJECT (lv), "popup-menu",
                    G_CALLBACK (on_popup_menu), NULL);
}

static void
kp_list_view_model_init (KPViewModelIface *iface)
{
  iface->get_dmy = kp_list_view_get_dmy;
  iface->set_dmy = kp_list_view_set_dmy;
  iface->set_log = kp_list_view_set_log;
  iface->unset_log = kp_list_view_unset_log;
  iface->set_view_type = kp_list_view_set_view_type;
  iface->get_view_type = kp_list_view_get_view_type;
  iface->get_icon_name = kp_list_view_get_icon_name;
  iface->activate = kp_list_view_activate;
  iface->deactivate = kp_list_view_deactivate;
}

static void
kp_list_view_popup_model_init (KPViewPopupModelIface *iface)
{
  iface->get_active_entry = kp_list_view_get_active_entry;
}


static void
kp_list_view_finalize (GObject *object)
{
  KPListViewPrivateData *p_data;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (object);

  g_print ("Running finalize!");
    
  g_return_if_fail (p_data != NULL);
  g_date_free (p_data->date);

  g_return_if_fail (KP_IS_LIST_VIEW (object));
  
  g_free (KP_LIST_VIEW (object)->private_data);  
  KP_LIST_VIEW (object)->private_data = NULL;
}


GtkWidget *
kp_list_view_new (void)
{
  KPListViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_list_view_get_type (), NULL);

  p_data = KP_LIST_VIEW_PRIVATE_DATA (widget);
  
  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

static void
kp_list_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPListViewPrivateData *p_data;
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_date_set_dmy (p_data->date, d, m, y);

  kp_list_view_update (KP_LIST_VIEW (view));
  
  kp_debug ("Date set: %u.%u.%u", d, m, y);
}

static void
kp_list_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (KP_IS_LIST_VIEW (view));
  g_return_if_fail (g_date_valid (p_data->date));
  
  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}

static void
kp_list_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);
  p_data->type = type;

  if (type == KP_VIEW_MODEL_TYPE_ALL_TIME)
    kp_debug ("Set all the time on!");
  
  kp_list_view_update (KP_LIST_VIEW (view));
}


static KPViewModelType
kp_list_view_get_view_type (KPViewModel *view)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);
  
  return p_data->type;
}

static gchar *
kp_list_view_get_icon_name (KPViewModel *view)
{
  return g_strdup ("list.png");
}


static void
kp_list_view_set_log (KPViewModel *model, KPTrainingLog *log)
{
  KPListViewPrivateData *p_data;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);
  p_data->log = log;

  kp_list_view_update (KP_LIST_VIEW (model));

  log_connect_signals (log, KP_LIST_VIEW (model));
}

static void
kp_list_view_unset_log (KPViewModel *model)
{
  KPListViewPrivateData *p_data;
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);

  gtk_list_store_clear (p_data->store);
  
  log_disconnect_signals (p_data->log, KP_LIST_VIEW (model));
  p_data->log = NULL;
}

static void
kp_list_view_activate (KPViewModel *model)
{
  KPListViewPrivateData *p_data;
  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);
  
  log_connect_signals (p_data->log, KP_LIST_VIEW (model));
  kp_list_view_update (KP_LIST_VIEW (model));
}

static void
kp_list_view_deactivate (KPViewModel *model)
{
  KPListViewPrivateData *p_data;
  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  p_data = KP_LIST_VIEW_PRIVATE_DATA (model);
  
  log_disconnect_signals (p_data->log, KP_LIST_VIEW (model));
}

static void
kp_list_view_update_dates (KPListView *view)
{
  KPListViewPrivateData *p_data;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (view);

  if (!p_data->date) 
    return;

  /* For all time, the log is needed to figure out which are the first
   * and the last date of the log */
  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME
   && !KP_IS_TRAINING_LOG (p_data->log))
    g_return_if_reached ();
    
  kp_gui_get_dates_for_view_type (p_data->date, p_data->type, &p_data->date_s,
                                 &p_data->date_e, p_data->log);
}


KPCalendarEntry *
kp_list_view_get_active_entry (KPViewPopupModel *popup_model)
{
  KPListViewPrivateData *p_data;
  KPCalendarEntry *entry = NULL;
  GtkTreeView *treeview = GTK_TREE_VIEW (popup_model);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GDate *date;
  gchar *date_str;
  gchar *mark;
  guint d,m,y;

  p_data = KP_LIST_VIEW_PRIVATE_DATA (popup_model);

  g_return_val_if_fail (GTK_IS_TREE_VIEW (treeview), NULL);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  date = g_date_new ();

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    gtk_tree_model_get (model, &iter,
                        COLUMN_DATE, &date_str,
                        COLUMN_DESCRIPTION, &mark,
                        -1);

    g_date_set_parse (date, date_str);

    if (!g_date_valid (date)) {
      kp_debug ("Couldn't parse date string: %s\n", date);
      return NULL;
    }

    d = g_date_get_day (date);
    m = g_date_get_month (date);
    y = g_date_get_year (date);

    g_return_val_if_fail (KP_IS_TRAINING_LOG (p_data->log), NULL);
    
    entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
  }
  return entry;
}

  
static void
kp_list_view_update (KPListView *lv)
{
  kp_list_view_update_dates (lv);
  kp_list_view_update_entries (lv);
}


static void
kp_list_view_update_entries (KPListView *lv)
{
  KPListViewPrivateData *p_data;
  KPParamListCategory *cat;
  KPWorkout *wo;
  GtkTreeIter iter;
  gchar date_str[32];
  gchar *markup_escape;
  gchar *type_name;
  gchar *param_str;
  gchar *duration;
  gchar *comment;
  gchar *pace_str;
  GString *p_str;
  guint32 pace;
  const gchar *entry;
  GDate *date;
  GList *list;
  GList *pn;

  g_return_if_fail (KP_IS_LIST_VIEW (lv));
  
  kp_debug ("Updating entries.");
  
  p_data = KP_LIST_VIEW_PRIVATE_DATA (lv);
  if (!p_data->log)
    return;
  
  list = kp_training_log_get_all_entries_between (p_data->log, p_data->date_s,
                                                  p_data->date_e, NULL);
  p_data->entries = list;
  gtk_list_store_clear (p_data->store);
 
  date = g_date_new ();
    
  pace_str = NULL;
  
  p_str = g_string_new_len (NULL, 128);
  
  while (list) {
    entry = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));
    date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (list->data)->datetime);
    g_return_if_fail (g_date_valid (date));
   
    (void) g_date_strftime (date_str, sizeof (date_str)-1, "%x", date);

    type_name = kp_calendar_entry_get_human_name (list->data);

    gtk_list_store_append (p_data->store, &iter);
    gtk_list_store_set (p_data->store, &iter,
                        COLUMN_DATE, date_str,
                        COLUMN_TYPE, type_name,
                        COLUMN_DESCRIPTION, entry,
                        COLUMN_TIME, "",
                        -1);
    
    if (KP_IS_WORKOUT (list->data)) {
      g_string_assign (p_str, "");
      wo = KP_WORKOUT (list->data);
      
      if ((pace = kp_workout_model_get_pace (KP_WORKOUT_MODEL (wo)))) {
        /* Let's get rid of milliseconds*/
        pace /= 1000;
        pace *= 1000;
        
        pace_str = kp_date_mseconds_to_std_string (pace);
      } 
      param_str = NULL;
      
      /* Categories */
      cat = kp_param_list_get_category (KP_WORKOUT (wo)->param_list, "detail");
      /* Params */
      for (pn = cat->list; pn; pn = pn->next) {
        if (KP_IS_PARAM (pn->data)) {
          param_str = kp_param_get_as_string (KP_PARAM (pn->data));
          g_string_append_printf (p_str, "%s: %s\n",
                                  kp_param_get_name (KP_PARAM (pn->data)),
                                  param_str);
          g_free (param_str); 
        }
      }
      comment = kp_workout_get_comment (wo);
      duration = kp_param_list_get_as_string (KP_WORKOUT (wo)->param_list, "duration");

      markup_escape = g_markup_escape_text (comment, -1);
     
      g_return_if_fail (gtk_list_store_iter_is_valid (p_data->store, &iter));
      gtk_list_store_set (p_data->store, &iter,
                          COLUMN_TIME, (duration) ? duration : "",
                          COLUMN_SPORT, kp_workout_get_sport (wo),
                          COLUMN_OTHER, (p_str->str) ? p_str->str : "",
                          COLUMN_PACE, (pace) ? pace_str : "",
                          COLUMN_DISTANCE, kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo)),
                          COLUMN_COMMENT, (markup_escape) ? markup_escape : "",
                         -1);

      g_free (duration);
      g_free (comment);
    }
    
    g_free (type_name);

    list = list->next;
  }

  g_string_free (p_str, TRUE);
}


static void
log_changed (KPTrainingLog *log, KPListView *lv)
{
  kp_list_view_update (lv);
}
  
static void
log_connect_signals (KPTrainingLog *log, KPListView *lv)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), lv);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPListView *lv)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  g_signal_handlers_disconnect_by_func (log, log_changed, lv);
}

  
static gboolean
on_button_press_event (GtkWidget *treeview, GdkEventButton *event,
                       KPListView *lv)
{
  if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
    view_popup_menu (treeview, event, NULL);
  }
  
  return FALSE;
}


static gboolean
on_popup_menu (GtkWidget *treeview, gpointer data)
{
  view_popup_menu (treeview, NULL, data);
  return TRUE; 
}


static void
view_popup_menu (GtkWidget *treeview, GdkEventButton *event, gpointer data)
{
  KPListViewPrivateData *p_data = KP_LIST_VIEW_PRIVATE_DATA (treeview);

  kp_entries_popup_prepare_dynamic (p_data->popup_data, 
                                    KP_VIEW_POPUP_MODEL (treeview));
  
  gtk_menu_popup (GTK_MENU (p_data->popup_data->menu), NULL, NULL, NULL, NULL,
                  (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));
}

