#include <string.h>
#include <glib.h>
#include <glib-object.h>

#include "kpcalendarentry.h"
#include "kptraininglog.h"
#include "kipina-i18n.h"
#include "kpworkout.h"
#include "kputil.h"

/* GObject stuff */
static void       kp_workout_class_init           (GObjectClass *klass,
                                                   gpointer data);
static void       kp_workout_instance_init        (GObject *object,
                                                   gpointer data);
static void       kp_workout_instance_finalize    (GObject *object);


/* Parameter handling */
static void       wo_params_to_xml                (gpointer key,
                                                   gpointer val,
                                                   gpointer data);
static void       wo_param_to_string              (gpointer key,
                                                   gpointer val,
                                                   GString *string);
/* Splits */
static gint       split_cmp                       (gconstpointer a,
                                                   gconstpointer b);
static void       wo_splits_to_xml                (gpointer data,
                                                   gpointer u_data);

/* KPCalendarEntry virtual method implementations */
static xmlNodePtr kp_workout_to_xml               (KPCalendarEntry *entry);
static gchar     *kp_workout_to_calendar_string   (KPCalendarEntry *entry);
static gboolean   kp_workout_parse                (KPCalendarEntry *entry,
                                                   xmlNodePtr node);
static gchar     *kp_workout_get_human_name       (KPCalendarEntry *entry);

GType
kp_workout_get_type ()
{
  static GType kp_workout_type = 0;

  if (!kp_workout_type) {
    static const GTypeInfo kp_workout_info = {
      sizeof (KPWorkoutClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_workout_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPWorkout),
      0,
      (GInstanceInitFunc) kp_workout_instance_init,
      NULL
    };

    kp_workout_type = g_type_register_static (KP_TYPE_CALENDAR_ENTRY,
                                             "KPWorkout",
                                             &kp_workout_info,
                                              0);
  }
  return kp_workout_type;
}

static void
kp_workout_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;
  KPCalendarEntryClass *entry_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_workout_instance_finalize;

  entry_class = KP_CALENDAR_ENTRY_CLASS (klass);
  entry_class->get_human_name = kp_workout_get_human_name;
  entry_class->to_string = kp_workout_to_calendar_string;
  entry_class->to_xml = kp_workout_to_xml;
  entry_class->parse = kp_workout_parse;
}

static void
kp_workout_instance_init (GObject *object, gpointer data)
{
  KPWorkout *wo;

  wo = KP_WORKOUT (object);
  
  wo->params = g_hash_table_new_full (g_str_hash,
                                      g_str_equal,
                                      g_free, 
                                      (GDestroyNotify) g_value_unset);
  wo->splits = NULL;
  wo->sport = g_string_new (NULL);
}

static void
kp_workout_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  KPWorkout *wo = (KPWorkout *) object;

  if (wo->params)
    g_hash_table_destroy (wo->params);

  g_string_free (wo->sport, TRUE);
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_workout_new:
 * 
 * Create a new instance of #KPWorkout.
 * 
 * Returns: A new #KPWorkout.
 */
KPWorkout *
kp_workout_new ()
{
  return g_object_new (kp_workout_get_type (), NULL);
}

/**
 * kp_workout_add_split:
 * @wo: A #KPWorkout
 * @type: Type of split
 * @n: number of split
 * @distance: distance
 * @ct: Duration of the split
 * @comment: Some comments or NULL
 *
 * Dum.
 * 
 */
void
kp_workout_add_split (KPWorkout * wo,
                      KPWorkoutSplitType type,
                      guint n,
                      gfloat distance,
                      KPCalendarTime * ct,
                      const gchar * comment)
{
  KPWorkoutSplit *split;

  g_return_if_fail (KP_IS_WORKOUT (wo));
  g_return_if_fail (KP_IS_CALENDAR_TIME (ct));
  g_return_if_fail (n > 0);
  g_assert (type == KP_WORKOUT_SPLIT_TYPE_WARM_UP
         || type == KP_WORKOUT_SPLIT_TYPE_COOL_DOWN
         || type == KP_WORKOUT_SPLIT_TYPE_NORMAL
         || type == KP_WORKOUT_SPLIT_TYPE_RECOVERY);

  /* If the lap number n is already in the list, don't
   * add it (Lap numbering starts from 1 and list numbering starts from 0,
   * so decrease n by one when finding lap from list). */
  if (g_slist_nth (wo->splits, n - 1) != NULL)
    return;

  split = g_new0 (KPWorkoutSplit, 1);
  split->n = n;
  split->time = ct;
  split->type = type;
  split->distance = distance;
  split->comment = (comment) ?  g_strdup (comment) : g_strdup ("");

  kp_debug ("Adding split.\n");
  wo->splits = g_slist_insert_sorted (wo->splits, (gpointer) split, split_cmp);
}

/**
 * kp_workout_is_valid:
 * @wo: A #KPWorkout
 *
 * Check the validity of the #KPWorkout.
 * 
 * Returns: TRUE if workout is valid and FALSE otherwise.
 */
gboolean
kp_workout_is_valid (KPWorkout * wo)
{
  if (wo == NULL)
    return FALSE;

  if (wo->sport == NULL || wo->sport->len == 0)
    return FALSE;

  if (KP_CALENDAR_ENTRY (wo)->datetime == NULL)
    return FALSE;

  return TRUE;
}

/**
 * kp_workout_get_n_splits:
 * @wo: A #KPWorkout
 *
 * Just returns the number of splits of this workout.
 * 
 * Returns: Number of splits.
 */
guint
kp_workout_get_n_splits (KPWorkout * wo)
{
  g_return_val_if_fail (KP_IS_WORKOUT (wo), 0);
  if (!wo->splits)
    return 0;
  return (g_slist_length (wo->splits));
}


/**
 * kp_workout_get_splits:
 * @wo: A #KPWorkout
 *
 * Retrieve list of splits of this workout.
 * 
 * Returns: List of all splits of the workout or NULL if
 *          there are not any splits.
 */
GSList *
kp_workout_get_splits (KPWorkout * wo)
{
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);
  return wo->splits;
}

static gint
split_cmp (gconstpointer a,
           gconstpointer b)
{
  const KPWorkoutSplit *a_;
  const KPWorkoutSplit *b_;

  a_ = a;
  b_ = b;

  if (a_->n > b_->n)
    return 1;

  if (a_->n < b_->n)
    return -1;

  return 0;
}


/**
 * kp_workout_get_distance:
 * @wo: a KPWorkout
 *
 * Returns distance as gdouble.
 **/
gdouble
kp_workout_get_distance (KPWorkout * wo)
{
  KPWorkoutSplit *split;
  gdouble dist = 0.0;
  GValue *distance;
  GSList *item;

  g_return_val_if_fail (KP_IS_WORKOUT (wo), 0);
  distance = g_hash_table_lookup (wo->params, "distance");

  /* No splits */
  if (distance != NULL) {
    if (G_VALUE_HOLDS (distance, G_TYPE_FLOAT))
      return (gdouble) g_value_get_float (distance);

    if (G_VALUE_HOLDS (distance, G_TYPE_DOUBLE))
      return g_value_get_double (distance);
  }

  if (wo->splits == NULL)
    return 0.0;
  
  /* Splits */
  for (item = wo->splits; item != NULL; item = item->next) {
    split = item->data;

    if (split)
      dist += split->distance;
  }

  return dist;
}


gdouble
kp_workout_get_param_double (KPWorkout *wo, const gchar *param)
{
  GValue *val;

  g_return_val_if_fail (wo != NULL, 0);
  g_return_val_if_fail (param != NULL, 0);

  if (strcmp (param, "distance") == 0)
    return kp_workout_get_distance (wo);

  if (strcmp (param, "pace") == 0)
    return kp_workout_get_pace (wo);

  if (strcmp (param, "duration") == 0)
    return kp_workout_get_duration_in_milliseconds (wo);
 
  if (wo->params == NULL)
    return 0;

  val = g_hash_table_lookup (wo->params, param);
  if (!val)
    return 0;

  if (G_VALUE_HOLDS_DOUBLE (val))
    return g_value_get_double (val);

  if (G_VALUE_HOLDS_UINT (val))
    return (gdouble) g_value_get_uint (val);

  if (G_VALUE_HOLDS_INT (val))
    return (gdouble) g_value_get_int (val);

  if (G_VALUE_HOLDS_FLOAT (val)) {
    kp_debug ("float (%.2f) shouldn't be used in params!"
            "Use double instead.\n", g_value_get_float (val));
    return g_value_get_float (val);
  }
  return 0.0;
}

/**
 * kp_workout_set_sport:
 * @wo: A #KPWorkot
 * @sport: The name of the sport, can be NULL.
 *
 * Set the sport of the workout.
 */
void
kp_workout_set_sport (KPWorkout *wo, const gchar *sport)
{
  g_return_if_fail (KP_IS_WORKOUT (wo));
  g_return_if_fail (wo->sport != NULL);

  g_string_assign (wo->sport, (sport != NULL) ? sport : "");
}

/**
 * kp_workout_get_sport:
 * @wo: A #KPWorkout
 * 
 * Get the sport of the workout.
 * 
 * Returns: String that must NOT be freed.
 */
gchar *
kp_workout_get_sport (KPWorkout *wo)
{
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);
  g_return_val_if_fail (wo->sport != NULL, NULL);
 
  return wo->sport->str;
}

/**
 * kp_workout_get_pace:
 * @wo: A #KPWorkout
 * * 
 * Get pace as milliseconds.
 *
 * Returns: Pace as milliseconds.
 */
guint32
kp_workout_get_pace (KPWorkout *wo)
{
  gint32 distance;
  gint32 duration;
  gchar *d_str;
  
  d_str = kp_workout_get_param (wo, "duration");
  if (!d_str)
    return 0;

  duration = kp_duration_str_to_ms ((const gchar *) d_str);
  distance = (gint32) kp_workout_get_distance (wo) * 1000;

  g_free (d_str);

  if (duration > 0 && distance > 0) 
    return (guint32)(duration / (distance / 1000));

  return 0;
}


/**
 * kp_workout_set_param:
 * @wo: a KPWorkout
 * @replace: if TRUE and param exists, it will be replaced
 * @param: name of the param
 * @value: value of the param
 *
 * Just sets kp_workouts' @param to @value and returns TRUE if
 * successful and FALSE otherwise.
 */
gboolean
kp_workout_set_param (KPWorkout * wo, gboolean replace, const gchar * param,
                      const GValue * value)
{
  gpointer val = NULL;
  GValue *hash_val;

  g_return_val_if_fail (wo != NULL, FALSE);
  g_return_val_if_fail (param != NULL, FALSE);
  g_return_val_if_fail (value != NULL, FALSE);
  g_return_val_if_fail (wo->params != NULL, FALSE);

  if (!replace)
    val = g_hash_table_lookup (wo->params, param);

  if (!replace && val)
    return FALSE;
  else {
    hash_val = g_new0 (GValue, 1);

    g_value_init (hash_val, G_VALUE_TYPE (value));
    g_value_copy (value, hash_val);

    g_hash_table_insert (wo->params, g_strdup (param), hash_val);
  }
  return TRUE;
}


/**
 * kp_workout_get_param:
 * @wo: A #KPWorkout
 * @param: The name of the param whose value to get
 *
 * Get value of the param.
 * 
 * Returns: A newly-allocated string that must be freed by
 *          the caller or NULL if there is no such param in 
 *          his #KPWorkout.
 */
gchar *
kp_workout_get_param (KPWorkout *wo, const gchar *param)
{
  GValue *val;

  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);
  g_return_val_if_fail (param != NULL, NULL);

  if (wo->params == NULL)
    return NULL;

  val = g_hash_table_lookup (wo->params, param);
  if (!val)
    return NULL;

  return kp_g_value_to_string (val);
}


/**
 * kp_workout_get_duration_in_milliseconds:
 * @wo: A #KPWorkout
 * 
 * Retrieve duration of the kp_workout in milliseconds.
 *
 * Returns: Number of milliseconds
 */
guint32
kp_workout_get_duration_in_milliseconds (KPWorkout *wo)
{
  gchar *duration;
  guint32 msec;
  
  g_return_val_if_fail (KP_IS_WORKOUT (wo), 0); 

  duration = kp_workout_get_param (wo, "duration");
  if (!duration)
    return 0;

  msec = kp_duration_str_to_ms ((const gchar *)duration);

  g_free (duration);
  return msec;
}

void
kp_workout_set_duration_in_milliseconds (KPWorkout *wo, guint32 msecs)
{
  GValue *value;
  gchar *str;
  guint h, m, s, t;
  guint seconds;

  seconds = msecs / 1000;

  h = seconds / (60 * 60);
  m = (seconds - h * 60 * 60) / 60;
  s = seconds - h * 60 * 60 - m * 60;
  t = msecs % 1000;

  str = g_strdup_printf ("%.2u:%.2u:%.2u.%.3u", h, m, s, t);

  value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_STRING);
  g_value_set_string (value, str);

  kp_workout_set_param (wo, TRUE, "duration", value);

  g_value_unset (value);
  g_free (str);
}


/**
 * kp_workout_get_duration_in_seconds:
 * @wo: A #KPWorkout
 * 
 * Retrieve duration of the kp_workout in seconds.
 * It can have up to 3 decimals because it can include
 * milliseconds too.
 *
 * Returns: Number of seconds (and milliseconds).
 */
gdouble
kp_workout_get_duration_in_seconds (KPWorkout *wo)
{
  gchar *duration;
  gdouble sec;
  
  g_return_val_if_fail (KP_IS_WORKOUT (wo), 0.0); 

  duration = kp_workout_get_param (wo, "duration");
  if (!duration)
    return 0.0;

  sec = kp_is_valid_duration_entry_str ((const gchar *)duration);

  g_free (duration);
  return sec;
}

gchar *
kp_workout_get_duration_str (KPWorkout *wo)
{
  gdouble sec;

  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);
  
  sec = kp_workout_get_duration_in_seconds (wo);
  if (sec <= 0.0)
    return NULL;

  return kp_date_seconds_to_string (sec);
}
 

/**
 * kp_workout_get_params_string:
 * @wo: A #KPWorkout
 * 
 * Get string that lists all the params of
 * the workout not including distance, duration and comment,
 * because they are available via get_distance () and so on.
 *
 * Returns: A newly-allocated string that must be freed.
 */
gchar *
kp_workout_get_params_string (KPWorkout *wo)
{
  GString *string;
  gchar *val;

  string = g_string_new (NULL);

  g_hash_table_foreach (wo->params, (GHFunc) wo_param_to_string, string);

  val = string->str;
  g_string_free (string, FALSE);

  kp_debug ("PARAMS: \"%s\"", val);
  
  return val;
}

/**
 * kp_workout_set_duration_in_seconds:
 * @wo: a KPWorkout
 * @seconds: duration in seconds
 *
 * Sets 'duration' key in params-hashtable to string which
 * presents the same amount of time as @seconds.
 **/
void
kp_workout_set_duration_in_seconds (KPWorkout *wo, gdouble seconds)
{
  GValue *value;
  gchar *str;
  guint h, m, s, t;

  h = (guint) ((guint) seconds / (60 * 60));
  m = (guint) (((guint) seconds - h * 60 * 60) / 60);
  s = (guint) ((guint) seconds - h * 60 * 60 - m * 60);
  t = (guint) ((seconds - h * 60 * 60 - m * 60 - s) * 1000);

  str = g_strdup_printf ("%.2u:%.2u:%.2u.%.3u", h, m, s, t);

  value = g_new0 (GValue, 1);
  g_value_init (value, G_TYPE_STRING);
  g_value_set_string (value, str);

  kp_workout_set_param (wo, TRUE, "duration", value);

  g_value_unset (value);
  g_free (str);
}

/**
 * get_output_format_date (KPWorkout *wo)
 * @wo: a KPWorkout.
 *
 * Returns date in format CCYY-MM-DDThh:mm:ss. Result-string
 * is malloc'd, so it must be freed.
 **/
gchar *
kp_workout_get_formatted_date (KPWorkout *wo)
{
  GDate *date;
  gchar *format;

  g_return_val_if_fail (wo != NULL, NULL);
  g_return_val_if_fail (KP_CALENDAR_ENTRY (wo)->datetime != NULL, NULL);

  date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (wo)->datetime);
  g_return_val_if_fail (date != NULL, NULL);

  format = g_strdup_printf ("%.4d-%.2d-%.2dT%.2d:%.2d:%.2d",
                            g_date_get_year (date),
                            g_date_get_month (date),
                            g_date_get_day (date),
                            KP_CALENDAR_ENTRY (wo)->datetime->h,
                            KP_CALENDAR_ENTRY (wo)->datetime->m,
                            KP_CALENDAR_ENTRY (wo)->datetime->s);
  return format;
}


/**
 * Get kp_workout as a string that can be passed directly to CalendarView or 
 * something like that.
 *
 * Returns a newly-allocated string that must be freed by the caller.
 */
static gchar *
kp_workout_to_calendar_string (KPCalendarEntry *entry)
{
  gdouble distance;
  GString *string;
  gchar *duration;
  gchar *comment;
  KPWorkout *wo;
  gchar *sport;
  gchar *str;
  
  wo = KP_WORKOUT (entry);
  g_return_val_if_fail (KP_IS_WORKOUT (wo), NULL);

  distance = kp_workout_get_distance (KP_WORKOUT (wo));
  duration = kp_workout_get_duration_str (KP_WORKOUT (wo));

  string = g_string_new (NULL);

  sport = kp_workout_get_sport (wo);
  
  if (distance > 0.0 && duration)
    g_string_printf (string, "<b>%s</b> %.1fkm\n%s\n", sport, distance,
                     duration);
  else if (duration) 
    g_string_printf (string, "<b>%s</b>: %s\n", sport, duration);
  else 
    g_string_printf (string, "<b>%s</b>: %.1fkm\n", sport, distance);

  g_hash_table_foreach (wo->params, (GHFunc) wo_param_to_string, string);

  comment = kp_workout_get_param (wo, "comment");
  if (comment) {
    g_string_append (string, comment);
    g_free (comment);
  }
 
  if (string->str[string->len-1] == '\n')
    string->str[string->len-1] = 0;
  
  if (duration)
    g_free (duration);

  str = string->str;
  g_string_free (string, FALSE);
 
  return str;
}

static void
wo_param_to_string (gpointer key,
                    gpointer val,
                    GString *string)
{
  gchar *str;

  if (strcmp ((gchar *) key, "duration") == 0)
    return;
  if (strcmp ((gchar *) key, "distance") == 0)
    return;
  if (strcmp ((gchar *) key, "comment") == 0)
    return;
  
  str = kp_g_value_to_string (val);
 
  g_return_if_fail (str != NULL);
  
  if (strlen (str) == 0) {
    g_free (str);
    return;
  }
  
  g_string_append_printf (string, "%s: %s\n", (gchar *) key, str);

  g_free (str);
}

static void
parse_split (xmlNodePtr node, KPWorkout *wo)
{
  KPCalendarTime *time = NULL;
  KPWorkoutSplitType split_type;
  xmlNodePtr child;
  xmlChar *time_str;
  gboolean valid = FALSE;
  gboolean val;
  gchar *comment = NULL;
  gchar *distance_str;
  gchar *type;
  gchar *attr;

  gfloat distance = 0.0;
  guint n;

  attr = xmlGetProp (node, "n");
  if (attr != NULL) {
    n = (guint) strtod (attr, NULL);
    xmlFree (attr);
  } else {
    g_warning (_("split-tag must have attribute 'n'!"));
    return;
  }
  /* default type */
  split_type = KP_WORKOUT_SPLIT_TYPE_NORMAL;

  for (child = node->children; child != NULL; child = child->next) {
    /** <duration>, <time> is obsolete **/
    if (KP_TAG_MATCH (child, "time")
     || KP_TAG_MATCH (child, "duration")) {

      time_str = xmlNodeGetContent (child);

      time = kp_calendar_time_new ();
      val = kp_calendar_time_set_hmst_str (time, time_str);

      if (val == FALSE) {
        g_warning (_("Invalid time format in split: %s!"), time_str);
        g_object_unref (time);
        return;
      }
      valid = TRUE;
    }
    /** distance **/
    if (KP_TAG_MATCH (child, "distance")) {
      distance_str = xmlNodeGetContent (child);
      distance = (gfloat) strtod (distance_str, NULL);
    }

    /** <comment> **/
    if (KP_TAG_MATCH (child, "comment"))
      comment = xmlNodeGetContent (child);
    
    /** <type> **/
    if (KP_TAG_MATCH (child, "type")) {
      type = xmlNodeGetContent (child);

      if (xmlStrcasecmp (type, "warmup") == 0)
        split_type = KP_WORKOUT_SPLIT_TYPE_WARM_UP;
      if (xmlStrcasecmp (type, "cooldown") == 0)
        split_type = KP_WORKOUT_SPLIT_TYPE_COOL_DOWN;
      if (xmlStrcasecmp (type, "recovery") == 0)
        split_type = KP_WORKOUT_SPLIT_TYPE_RECOVERY;

      /* Otherwise default, KP_WORKOUT_SPLIT_TYPE_NORMAL will 
         be used. */
    }
  }

  if (valid)
    kp_workout_add_split (wo, split_type, n, distance, time, comment);
}


/**
 * kp_workout_parse:
 * @entry: A #KPCalendarEntry
 * @node: A #xmlNodePtr.
 *
 * Parses a kp_workout and returns it. If kp_workout's type
 * is interval, interval-params are stored to params-hashtable.
 * Interval-things' keys use "__interval__"-prefix to prevent
 * name-collisions which should be impossible this way.
 *
 * Returns: TRUE if successful or FALSE otherwise.
 */
static gboolean
kp_workout_parse (KPCalendarEntry *entry, xmlNodePtr node)
{
  GValue *param_val;
  xmlNodePtr child;
  xmlChar *type_str;
  xmlChar *val_str;
  xmlChar *date;

  xmlChar *prop = xmlGetProp (node, "type");

  kp_workout_set_sport (KP_WORKOUT (entry), prop);
  g_free (prop);
  child = node->children;

  while (child) {
    
    /** <datetime> **/
    if (KP_TAG_MATCH (child, "datetime")) {
      date = xmlNodeGetContent (child);
      if (!kp_calendar_time_set_datetime (KP_CALENDAR_ENTRY (entry)->datetime,
                                          date))
        return FALSE;
      g_free (date);
    }

    /** <split> **/
    else if (KP_TAG_MATCH (child, "split")) {
      parse_split (child, KP_WORKOUT (entry));
    }
    
    /** <param> **/
    else if (KP_TAG_MATCH (child, "param")) {
      g_return_val_if_fail (kp_workout_is_valid (KP_WORKOUT (entry)), FALSE);

      val_str = xmlNodeGetContent (child);
      type_str = xmlGetProp (child, "type");

      if (type_str == NULL) {
        g_warning ("Parse error: param must have a type-attribute set!");
        return FALSE;
      }
      if (val_str == NULL) {
        g_warning ("Parse error: param must have a value!");
        return FALSE;
      }
      param_val = g_new0 (GValue, 1);

      if (xmlStrEqual (type_str, "float")) {
        g_value_init (param_val, G_TYPE_DOUBLE);
        g_value_set_double (param_val, strtod (val_str, NULL));
      }
      else if (xmlStrEqual (type_str, "uint")) {
        g_value_init (param_val, G_TYPE_UINT);
        g_value_set_uint (param_val, (guint) strtod (val_str, NULL));
      }
      else if (xmlStrEqual (type_str, "bool")) {
        g_value_init (param_val, G_TYPE_BOOLEAN);
        g_value_set_boolean (param_val, (((guint) strtod (val_str, NULL)) > 0));
      }
      else {
        g_value_init (param_val, G_TYPE_STRING);
        g_value_set_string (param_val, val_str);
      }

      kp_workout_set_param (KP_WORKOUT (entry), TRUE,
                            xmlGetProp (child, "name"), param_val);
      g_value_unset (param_val);
    }
    else if (KP_TAG_MATCH (child, "text")) {
      /* doesn't need to care about these */    
    } else {
      g_warning ("<workout> has an unknown child: %s", child->name);
    }
    child = child->next;
  }

  if (!kp_workout_is_valid (KP_WORKOUT (entry)))
    g_warning ("Parse error: this isn't valid workout!\n");
  else
    return TRUE;

  g_object_unref (entry);

  return FALSE;
}

static gchar *
kp_workout_get_human_name (KPCalendarEntry *entry)
{
  return g_strdup (_("Workout"));
}

static xmlNodePtr
kp_workout_to_xml (KPCalendarEntry * entry)
{
  xmlNodePtr date_child;
  xmlNodePtr node;
  xmlAttrPtr attr = NULL;
  xmlChar *date;

  g_return_val_if_fail (KP_IS_WORKOUT (entry), NULL);

  node = xmlNewNode (NULL, "workout");

  if (kp_workout_get_sport (KP_WORKOUT (entry)) != NULL) {
    attr = xmlNewProp (node, "type", kp_workout_get_sport (KP_WORKOUT (entry)));
    g_return_val_if_fail (attr != NULL, NULL);
  }

  date = kp_workout_get_formatted_date (KP_WORKOUT (entry));
  date_child = xmlNewChild (node, NULL, "datetime", date);
  g_return_val_if_fail (attr != NULL, NULL);

  g_object_set_data (G_OBJECT (entry), "node", node);
  
  if (KP_WORKOUT (entry)->params)
    g_hash_table_foreach (KP_WORKOUT (entry)->params, wo_params_to_xml, entry);

  if (KP_WORKOUT (entry)->splits)
    g_slist_foreach (KP_WORKOUT (entry)->splits,
                    (GFunc) wo_splits_to_xml, entry);

  return node;
}

static void
wo_splits_to_xml (gpointer data,
                  gpointer u_data)
{
  xmlNodePtr node;
  xmlNodePtr child;
  gchar *distance_str;
  gchar *type_str;
  gchar *time_str;
  gchar *n_str;

  KPWorkout *wo = (KPWorkout *) u_data;
  KPWorkoutSplit *split = (KPWorkoutSplit *) data;

  g_return_if_fail (split->time != NULL);
  kp_debug ("Adding split %u to kp_workout.\n", split->n);

  time_str = kp_calendar_time_to_str_hmst (KP_CALENDAR_TIME (split->time));

  distance_str = g_strdup_printf ("%.3f", split->distance);
  n_str = g_strdup_printf ("%u", split->n);

  switch (split->type)
  {
  case KP_WORKOUT_SPLIT_TYPE_NORMAL:
    type_str = "normal";
    break;

  case KP_WORKOUT_SPLIT_TYPE_WARM_UP:
    type_str = "warmup";
    break;

  case KP_WORKOUT_SPLIT_TYPE_COOL_DOWN:
    type_str = "cooldown";
    break;
  
  case KP_WORKOUT_SPLIT_TYPE_RECOVERY:
    type_str = "recovery";
    break;
    
  default:
    type_str = "";
    kp_debug ("Bug occurred. (not serious but not nice either..)");
  }
  
  node = xmlNewChild (g_object_get_data (G_OBJECT (wo), "node"), NULL,
                      "split", NULL);

  child = xmlNewChild (node, NULL, "time", time_str);
  child = xmlNewChild (node, NULL, "comment", split->comment);
  child = xmlNewChild (node, NULL, "type", type_str);
  child = xmlNewChild (node, NULL, "distance", distance_str);

  (void) xmlSetProp (node, "n", n_str);

  g_free (distance_str);
  g_free (time_str);
  g_free (n_str);
}

static void
wo_params_to_xml (gpointer key,
                  gpointer val,
                  gpointer data)
{
  GValue *value;
  GType type;

  const gchar *type_str;
  gchar *val_str;

  xmlNodePtr child;
  KPWorkout *wo = (KPWorkout *) data;
  xmlNodePtr node;

  value = val;

  type = G_VALUE_TYPE (value);

  switch (type) {
  case G_TYPE_UINT:
    val_str = g_strdup_printf ("%u", g_value_get_uint (value));
    type_str = "uint";
    break;

  case G_TYPE_FLOAT:
    val_str = g_strdup_printf ("%.3f", g_value_get_float (value));
    type_str = "float";
    break;
    
  case G_TYPE_DOUBLE:
    val_str = g_strdup_printf ("%.3f", g_value_get_double (value));
    type_str = "float";
    break;

  case G_TYPE_BOOLEAN:
    if (g_value_get_boolean (value))
      val_str = g_strdup ("true");
    else
      val_str = g_strdup ("false");

    type_str = "bool";
    break;

  case G_TYPE_STRING:
    val_str = g_strdup (g_value_get_string (value));
    type_str = "str";
    break;

  default:
    /* Compiler is stupid and reports warnings.. :/ */
    val_str = "";
    type_str = "";
    g_assert_not_reached ();
  }

  node = g_object_get_data (G_OBJECT (wo), "node");
  
  child = xmlNewChild (node, NULL, "param", val_str);
  (void) xmlNewProp (child, "type", type_str);
  (void) xmlNewProp (child, "name", key);

  g_free (val_str);
}
