#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>

#include <libxslt/xsltutils.h>
#include <libxslt/transform.h>
#include <libxslt/xslt.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include <glib-object.h>
#include <glib.h>

#include "kpcalendarentry.h"
#include "kptraininglog.h"
#include "kpmarshalers.h"
#include "kipina-i18n.h"
#include "kpsettings.h"
#include "kpcomment.h"
#include "kpworkout.h"
#include "kparray2d.h"
#include "kputil.h"

enum
{
  SAVED_SIGNAL,
  CHANGED_SIGNAL,
  ENTRY_ADDED_SIGNAL,
  ENTRY_REMOVED_SIGNAL,
  SPORT_LIST_CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_training_log_signals[LAST_SIGNAL] = { 0 };

static void       copy_css_file                     (const gchar *destdir);
static
KPTrainingLog    *kp_training_log_parse_from_xml    (const gchar *filename,
                                                     KPTrainingLog *log_,
                                                     GError **error);
xmlDocPtr         kp_training_log_to_xml            (KPTrainingLog *log);
static void       kp_training_log_year_to_xml       (gpointer val,
                                                     gpointer data);
static void       list_add_item_to_passed_list      (gpointer val,
                                                     gpointer data);
static void       kp_training_log_class_init        (GObjectClass *klass,
                                                     gpointer data);
static void       kp_training_log_instance_init     (GObject *object,
                                                     gpointer data);
static void       kp_training_log_instance_finalize (GObject *object);

/* Handling sports */
GSList           *find_sport                        (KPTrainingLog *log,
                                                     const gchar *sport);

static GString *last_error = NULL;


GType
kp_training_log_get_type ()
{
  static GType kp_training_log_type = 0;

  if (!kp_training_log_type) {
    static const GTypeInfo kp_training_log_info = {
      sizeof (KPTrainingLogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_training_log_class_init,
      NULL,
      NULL,
      sizeof (KPTrainingLog),
      0,
      (GInstanceInitFunc) kp_training_log_instance_init,
      NULL
    };

    kp_training_log_type = g_type_register_static (G_TYPE_OBJECT, "KPTrainingLog",
                                                &kp_training_log_info, 0);
  }
  return kp_training_log_type;
}


void
kp_training_log_xml_structured_error_func (void *ctx, xmlErrorPtr err)
{
  kp_debug ("XML2 error (%d): %s", err->level, err->message);
  g_string_assign (last_error, err->message);
}

static void
kp_training_log_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_training_log_instance_finalize;

  kp_training_log_signals[SAVED_SIGNAL]
    = g_signal_new ("saved",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, saved),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    0);

  kp_training_log_signals[CHANGED_SIGNAL]
    = g_signal_new ("changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, changed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE,
                    0);
  
  kp_training_log_signals[ENTRY_ADDED_SIGNAL]
    = g_signal_new ("entry-added",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, entry_added),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__UINT_UINT_UINT_STRING,
                    G_TYPE_NONE,
                    4,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING);

  kp_training_log_signals[ENTRY_REMOVED_SIGNAL]
    = g_signal_new ("entry-removed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, entry_removed),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__UINT_UINT_UINT_STRING,
                    G_TYPE_NONE,
                    4,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING);
  
  kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL]
    = g_signal_new ("sport-list-changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, sport_list_changed),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__BOOLEAN_STRING,
                    G_TYPE_NONE,
                    2,
                    G_TYPE_BOOLEAN,
                    G_TYPE_STRING);

  if (last_error == NULL)
    last_error = g_string_new (NULL);

  xmlSetStructuredErrorFunc (NULL, kp_training_log_xml_structured_error_func);
}

static void
kp_training_log_instance_init (GObject *object, gpointer data)
{
  KPTrainingLog *log;

  log = KP_TRAINING_LOG (object);
  log->sports = NULL;
  log->year_list = NULL;
  log->n_workouts = 0;
  log->n_entries = 0;
}


static void
kp_training_log_instance_finalize (GObject *object)
{
  KPTrainingLog *log = KP_TRAINING_LOG (object);
  GObjectClass *parent_class;

  kp_debug ("Finalized KPTrainingLog(%p) with %u entries.\n", log,
             kp_training_log_get_size (log));
 
  kp_training_log_remove_all (log);
  
  log->title = NULL;
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}

/**
 * kp_training_log_new:
 * 
 * Create a new instance of #KPTrainingLog.
 *
 * Returns: A #KPTrainingLog
 */
KPTrainingLog *
kp_training_log_new (void)
{
  KPTrainingLog *log;

  log = g_object_new (kp_training_log_get_type (), NULL);

  return log;
}

KPTrainingLog *
kp_training_log_new_from_file (const gchar *file)
{
  KPTrainingLog *log;

  log = kp_training_log_parse_from_xml (file, NULL, NULL);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  
  return log;
}

/**
 * kp_training_log_add_from_file:
 * @log: A #KPTrainingLog
 * @file: The name of the log file
 *
 * Add all the entries from the log specified by the @file to the
 * existing log.
 *
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_training_log_add_from_file (KPTrainingLog *log, const gchar *file,
                               GError **error)
{
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), FALSE);
  g_return_val_if_fail (file != NULL, FALSE);
  
  if (!kp_training_log_parse_from_xml (file, log, error)) {
    return FALSE;
  }
 
  return TRUE;
}


GQuark
kp_training_log_error_quark (void)
{
  static GQuark quark = 0;

  if (!quark)
    quark = g_quark_from_static_string ("training-log-error-quark");
  return quark;
}

void
kp_training_log_destroy (KPTrainingLog * log)
{
  g_return_if_fail (log != NULL);

  /*g_hash_table_foreach (log->years, destroy_year, NULL);
     g_hash_table_destroy (log->years); */

  g_list_free (log->year_list);
}

/**
 * kp_training_log_save:
 * @log: A #KPTrainingLog
 * @file: File to save log into
 *
 * Just save the contents of the log to the file specified by
 * the @file.
 *
 * Returns: TRUE if successful or FALSE otherwise.
 */
gboolean
kp_training_log_save (KPTrainingLog *log, const gchar *file, GError **error)
{
  xmlDocPtr doc;
  gboolean ret_val;
  int val;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (file != NULL, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  doc = kp_training_log_to_xml (log);

  g_return_val_if_fail (doc != NULL, FALSE);

  val = xmlSaveFormatFile (file, doc, 1);

  if (val > 0) {  
    g_signal_emit (G_OBJECT (log), kp_training_log_signals[SAVED_SIGNAL], 0);
    ret_val = TRUE;
  }
  else {
    kp_debug ("Setting error!: %s", last_error->str);
   
    if (error)
      g_set_error (error,
                   KP_TRAINING_LOG_ERROR,
                   KP_TRAINING_LOG_SAVE_ERROR,
                 _("The log (%s) couldn't be saved: %s!"), 
                   file, last_error->str);

    ret_val = FALSE;
  }

  xmlFreeDoc (doc);

  return ret_val;
}

/**
 * kp_training_log_create_html_stats:
 * @log: A #KPTrainingLog
 * @output_dir: Directory to output the HTML files
 * @error: location to store possible error
 *
 * Generate HTML statistics from data.
 * 
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_training_log_create_html_stats (KPTrainingLog *log, const gchar *output_dir,
                                   GError **error)
{
  xsltStylesheetPtr cur = NULL;
  xmlDocPtr doc;
  xmlDocPtr res = NULL;

  gboolean retval = FALSE;

  gchar *output_file;
  gchar *stylesheet;
  gchar *params[6];
  gchar *language;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  doc = kp_training_log_to_xml (log);

  language = kp_settings_get_str ("html_language");
  
  params[0] = "output_dir";
  params[1] = g_strdup_printf ("'%s'", output_dir);
  params[2] = "lang";
  params[3] = (language) ? language : "en";
  params[4] = NULL;
  params[5] = NULL;

  stylesheet = g_build_filename (KIPINA_DATA_DIR, "xsl", "kipina-report.xsl",
                                 NULL);
  output_file = g_build_filename (output_dir, "kipina-report.html", NULL);
 
  if (!g_file_test (output_dir, G_FILE_TEST_IS_DIR)) {
    g_set_error (error,
                 G_FILE_ERROR,
                 g_file_error_from_errno (errno),
              _("There is no directory \"%s\"!\nCan't create HTML statistics."
                "\n\nCreate the directory and try again."),
                 output_dir);
    goto error;
  }
                 
  cur = xsltParseStylesheetFile (stylesheet);
  if (cur == NULL) {
    g_set_error (error,
                 KP_TRAINING_LOG_ERROR,
                 KP_TRAINING_LOG_INTERNAL_ERROR,
               _("Internal error happened! (Parsing stylesheet failed.)"
                 "\n\nTHIS SHOULD NOT HAPPEN EVER!"));
    goto error;
  }
  res = xsltApplyStylesheet (cur, doc, (const gchar **) params);
  if (res == NULL) {
    g_set_error (error,
                 KP_TRAINING_LOG_ERROR,
                 KP_TRAINING_LOG_INTERNAL_ERROR,
               _("Internal error happened! (Applying stylesheet to xmlDoc"
                 "failed.)\n\nTHIS SHOULD NOT HAPPEN EVER!"));
    goto error;
  }

  retval = (xsltSaveResultToFilename(output_file, res, cur, 0) != -1);
  if (retval) {
    kp_debug ("Statistics saved to '%s'", output_file);
    copy_css_file (output_dir);
  } else {
    g_set_error (error,
                 KP_TRAINING_LOG_ERROR,
                 KP_TRAINING_LOG_SAVE_HTML_ERROR,
               _("HTML Statistics couldn't be saved to file \"%s\":"
                 "Some error happened!"),
                 output_file);
    goto error;
  }

  retval = TRUE;
error:
  
  xsltFreeStylesheet (cur);
  xsltCleanupGlobals ();

  if (res)
    xmlFreeDoc (res);
  if (doc)
    xmlFreeDoc (doc);

  g_free (params[1]);
  g_free (stylesheet);
  g_free (output_file);

  return retval;
}


static void
copy_css_file (const gchar * destdir)
{
  gint ret;
  gchar *cmd;

  cmd = g_strdup_printf ("cp %s/%s/%s %s", KIPINA_DATA_DIR,
                         "xsl", "log.css", destdir);
  ret = system (cmd);
  if (ret != 0)
    g_warning ("Couldn't copy log.css to output directory!");
  g_free (cmd);

  cmd = g_strdup_printf ("cp %s/%s/%s %s", KIPINA_DATA_DIR,
                         "xsl", "horiz.png", destdir);
  ret = system (cmd);
  if (ret != 0)
    g_warning ("Couldn't copy horiz.png to output directory!");
  g_free (cmd);
}

/**
 * kp_training_log_add:
 * @log: A #KPTrainingLog
 * @entry: A #KPCalendarEntry to add to the log.
 *
 * Just add the calendar entry to the log.
 *
 * Returns: TRUE if the entry was added and FALSE if some error occurred.
 */
gboolean
kp_training_log_add (KPTrainingLog *log, KPCalendarEntry *entry)
{
  KPSportEntry *s_entry;
  GSList *list;
  GDate *date;
  TYear *ty;
  gchar *str;
  guint y;
  guint m;
  guint d;
  gint i;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (entry != NULL, FALSE);

  date = kp_calendar_time_get_date (entry->datetime);
  g_return_val_if_fail (date != NULL, FALSE);

  y = g_date_get_year (date);
  m = g_date_get_month (date);
  d = g_date_get_day (date);

  kp_debug ("Adding workout to the log, date: %u.%u.%u", d, m, y);
  g_return_val_if_fail (g_date_valid_dmy (d, m, y), FALSE);
  
  ty = kp_training_log_get_year (log, y);

  /* If there is no year in list, add it */
  if (ty == NULL) {
    ty = g_new (TYear, 1);
    ty->year = y;

    log->year_list = g_list_prepend (log->year_list, ty);

    for (i = 0; i < 12; i++)
      ty->workouts[i] = NULL;

    ty->n_workouts = 0;
    ty->n_entries = 0;
  }

  g_return_val_if_fail (m > 0, FALSE);
  g_return_val_if_fail (m <= 12, FALSE);

  /* Insert entry to the month list */
  ty->workouts[m - 1] = g_list_insert_sorted (ty->workouts[m - 1],
                                              entry,
                               (GCompareFunc) kp_calendar_entry_cmp);

  str = kp_calendar_entry_to_string (entry);
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[ENTRY_ADDED_SIGNAL],
                 0, d, m, y, str);

  g_free (str);
  
  if (KP_IS_WORKOUT (entry)) {
    ty->n_workouts++;
    log->n_workouts++;

    str = kp_workout_get_sport (KP_WORKOUT (entry));
     
    if (!(list = find_sport (log, str))) {
      s_entry = g_new (KPSportEntry, 1);
      s_entry->n = 1;
      s_entry->name = g_strdup (str);
        
      log->sports = g_slist_prepend (log->sports, s_entry);
      g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                     TRUE, str);
    } else {
      ((KPSportEntry *) list->data)->n++;
    }
  }
  ty->n_entries++;
  log->n_entries++;
  
  return TRUE;
}


GSList *
find_sport (KPTrainingLog *log, const gchar *sport)
{
  KPSportEntry *entry;
  GSList *list;
  gchar *tmp1;
  gchar *tmp2;

  list = log->sports;

  while (list) {
     
    entry = (KPSportEntry *) list->data;
      
    tmp1 = g_utf8_collate_key (entry->name, -1);
    tmp2 = g_utf8_collate_key (sport, -1);

    if (strcmp (tmp1, tmp2) == 0) {
      g_free (tmp1);
      g_free (tmp2);
      
      return list;
    }
    g_free (tmp1);
    g_free (tmp2);

    list = list->next;
  }

  return NULL;
}


/**
 * kp_training_log_remove:
 * @log: A #KPTrainingLog
 * @entry: A #KPCalendarEntry to remove from the log.
 *
 * Just remove the calendar entry from the log.
 *
 * Returns: TRUE if the entry was removed and FALSE if some error occurred.
 */
gboolean
kp_training_log_remove (KPTrainingLog *log, KPCalendarEntry *entry)
{
  GSList *list;
  gchar *sport;
  GDate *date;
  TYear *ty;
  gchar *str;
  guint y;
  guint m;
  guint d;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), FALSE);
  g_return_val_if_fail (entry->datetime != NULL, FALSE);

  date = kp_calendar_time_get_date (entry->datetime);
  y = g_date_get_year (date);
  m = g_date_get_month (date);
  d = g_date_get_day (date);
  ty = kp_training_log_get_year (log, y);
  str = kp_calendar_entry_to_string (entry);
  
  g_return_val_if_fail (ty, FALSE);

  ty->workouts[m-1] = g_list_remove (ty->workouts[m-1], entry);
  
  if (KP_IS_WORKOUT (entry)) {
    ty->n_workouts--;
    log->n_workouts--;

    sport = kp_workout_get_sport (KP_WORKOUT (entry));
    list = find_sport (log, sport);
   
    if (list && ((KPSportEntry *) list->data)->n-- == 1) {
      g_free (((KPSportEntry *) list->data)->name);
      log->sports = g_slist_remove (log->sports, list->data);
 
      g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                     FALSE, sport);
    }
  }
  ty->n_entries--;
  log->n_entries--;

  if (ty->n_workouts == 0 && ty->n_entries == 0) {
    kp_debug ("Deleted year from the log!");
    log->year_list = g_list_remove (log->year_list, ty);
    g_free (ty);
    ty = NULL;
  }

  kp_debug ("Removed: %d.%d.%d: %s\n", d, m, y, str);
  
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[ENTRY_REMOVED_SIGNAL],
                 0, d, m, y, str);

  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);

  g_object_unref (entry);

  return TRUE;
}

/**
 * kp_training_log_remove_all:
 * @log: A #KPTrainingLog
 *
 * This function removes all the workouts from the log.
 * After running this function, @log will be still ready
 * to use, it just doesn't contain any workouts after that.
 */
void
kp_training_log_remove_all (KPTrainingLog *log)
{
  TYear *ty;
  GSList *s; /* Sport */
  GList *y;  /* Year */
  GList *e;  /* Entry */
  guint m;   /* Month - 1*/
  gchar *sport;

  y = log->year_list;

  while (y)
  {
    ty = (TYear *) y->data;
      
    for (m=0; m < 12; m++) {
      e = ty->workouts[m];

      while (e) {
        g_return_if_fail (KP_IS_CALENDAR_ENTRY (e->data));
        
        if (KP_IS_WORKOUT (e->data)) {
          log->n_workouts--;
          ty->n_workouts--;
        }
        log->n_entries--;
        ty->n_entries--;
        
        g_object_unref (G_OBJECT (e->data));
          
        e = e->next;
      }
      g_list_free (ty->workouts[m]);
    }

    g_return_if_fail (ty->n_workouts == 0);
    g_return_if_fail (ty->n_entries == 0);
    
    g_free (ty);
    
    y = y->next;
  } 

  g_list_free (log->year_list);

  s = log->sports;

  while (s) {
    sport = ((KPSportEntry *) s->data)->name;
    g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                   FALSE, sport);

    g_free (sport);
    g_free (s->data);
    
    s = s->next;
  }
  g_slist_free (log->sports);
  log->sports = 0;
  
  g_return_if_fail (log->n_workouts == 0);
  g_return_if_fail (log->n_entries == 0);
}




/**
 * kp_training_log_get_sports:
 * @log: A #KPTrainingLog
 * 
 * Get list of all different sports that have entries 
 * in the log.
 * 
 * Returns: List that should NOT be freed. 
 */
GSList *
kp_training_log_get_sports (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  return log->sports;
}


KPDate *
kp_training_log_get_earliest_date (KPTrainingLog *log)
{
  GList *list;
  guint d, m, y;
  GDate *gdate;
  KPDate *date;
  TYear *ty;
  guint i;
  
  if (log->n_entries == 0)
    goto return_today;
  
  if (log->year_list == NULL)
    goto return_today;
  
  list = log->year_list;
  ty = list->data;
  
  while (list) {
    if (((TYear *) list->data)->year < ty->year)
      ty = list->data;

    list = list->next;
  }
  kp_debug ("Found earliest year: %u", ty->year);
  
  y = ty->year;

  for (i=0; i < 12; i++) {
    if (ty->workouts[i]) {

      gdate = kp_calendar_time_get_date (kp_calendar_entry_get_date 
        (KP_CALENDAR_ENTRY (ty->workouts[i]->data)));

      g_return_val_if_fail (g_date_valid (gdate), NULL);

      d = g_date_get_day (gdate);
      m = g_date_get_month (gdate);
      y = g_date_get_year (gdate);

      date = kp_date_new_dmy (d, m, y);
      g_return_val_if_fail (kp_date_valid (date), NULL);
    
      kp_debug ("First date: %u.%u.%u\n", d, m, y);
      
      return date;
    }
  }

return_today:

  gdate = g_date_new ();
  g_date_set_time (gdate, time (NULL));

  date = kp_date_new_dmy (g_date_get_day (gdate),
                          g_date_get_month (gdate),
                          g_date_get_year (gdate));
  g_date_free (gdate);
  return date;
}

KPDate *
kp_training_log_get_latest_date (KPTrainingLog *log)
{
  GList *list;
  guint d, m, y;
  GDate *gdate;
  KPDate *date;
  TYear *ty;
  guint i;
  
  if (log->n_entries == 0)
    goto return_today;
  
  if (log->year_list == NULL)
    goto return_today;
  
  list = log->year_list;
  ty = list->data;
  
  while (list) {
    if (((TYear *) list->data)->year > ty->year)
      ty = list->data;

    list = list->next;
  }
  y = ty->year;

  for (i=11; i > 0; i--) {
    if ((list = g_list_last (ty->workouts[i]))) {

      gdate = kp_calendar_time_get_date (kp_calendar_entry_get_date 
        (KP_CALENDAR_ENTRY (list->data)));

      g_return_val_if_fail (g_date_valid (gdate), NULL);

      d = g_date_get_day (gdate);
      m = g_date_get_month (gdate);
      y = g_date_get_year (gdate);

      date = kp_date_new_dmy (d, m, y);
      g_return_val_if_fail (kp_date_valid (date), NULL);

      kp_debug ("Last date: %u.%u.%u\n", d, m, y);
    
      return date;
    }
  }

return_today:

  gdate = g_date_new ();
  g_date_set_time (gdate, time (NULL));

  date = kp_date_new_dmy (g_date_get_day (gdate),
                          g_date_get_month (gdate),
                          g_date_get_year (gdate));
  g_date_free (gdate);
  return date;
}


guint
kp_training_log_get_n_years (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), 0);

  if (log->year_list == NULL)
    return 0;

  return g_list_length (log->year_list);
}



/**
 * kp_training_log_remove_mark:
 * @log: a #KPTrainingLog
 * @d: the day
 * @m: the month
 * @y: the year
 * @mark: the mark to remove
 *
 * Removes the mark from the log. 
 */
void
kp_training_log_remove_mark (KPTrainingLog *log, guint d, guint m, guint y,
                             const gchar *mark)
{
  GList *list;
  gchar *m_str;

  list = kp_training_log_get_day (log, d, m, y);
  while (list) {
    m_str = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));

    if (strcmp (m_str, mark) == 0) {
      g_free (m_str);
      kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));
      return;
    }
    else
      g_free (m_str);
    
    list = list->next;
  }
}

/**
 * kp_training_log_remove_by_date:
 * @log: a #KPTrainingLog
 * @d: Day number or -1
 * @m: Month number or -1
 * @y: Year number or -1
 *
 * Remove one or more calendar entries from the log. If d, m and y are all
 * not -1 and mark is not NULL, only one entry is removed. In more complex cases
 * all entries in the day, month, year or even in the whole log may be removed.
 * 
 * By setting d, m, or y -1 or setting mark NULL mean that all those will be
 * deleted, like if d is -1, all entries in all days in month m will be deleted.
 */
void
kp_training_log_remove_by_date (KPTrainingLog *log, gint d, gint m, gint y,
                                const gchar *mark)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  if (mark == NULL) {
    kp_training_log_remove_day (log, d, m, y);
    return;
  }
  
  if (d < 0) {
    kp_training_log_remove_month (log, y, m);
    return;
  }
  
  if (m < 0) {
    /* TODO:
    kp_training_log_remove_year (log, y);*/
    return;
  }

  kp_training_log_remove_mark (log, d, m, y, mark);
}

/**
 * kp_training_log_remove_month:
 * @log: A #KPTrainingLog
 * @y: the day
 * @m: the month
 *
 * Remove all entries in a month.
 */
void
kp_training_log_remove_month (KPTrainingLog *log, guint y, guint m)
{
  GList *list;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  list = kp_training_log_get_month (log, y, m);
  while (list) {
    kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));
    list = list->next;
  }
}

/**
 * kp_training_log_remove_day:
 * @d: the day
 * @m: the month
 * @y: the year
 *
 * Remove all entries of a day.
 */
void
kp_training_log_remove_day (KPTrainingLog *log, guint d, guint m, guint y)
{
  GList *list;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  list = kp_training_log_get_day (log, d, m, y);

  while (list) {
    kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));  
    list = list->next;
  }
}


/**
 * kp_training_log_to_xml:
 * @log: a #KPTrainingLog
 * 
 * Convert log to xml. This function just tries to convert data from
 * KPTrainingLog to xmlDocPtr. KPTrainingLog can be empty.
 *
 * Returns: #xmlDocPtr or NULL if some error occurs.
 */
xmlDocPtr
kp_training_log_to_xml (KPTrainingLog * log)
{
  xmlDocPtr doc;
  xmlNodePtr node;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  doc = xmlNewDoc ("1.0");
  node = xmlNewChild ((xmlNodePtr) doc, NULL, "traininglog", NULL);

  if (log->year_list)
    g_list_foreach (log->year_list, kp_training_log_year_to_xml, node);

  return doc;
}

static void
kp_training_log_year_to_xml (gpointer val, gpointer data)
{
  xmlDocPtr doc = (xmlDocPtr) data;
  GList *el;
  guint i;
  guint n = 0;
  TYear *ty;

  ty = (TYear *) val;

  for (i = 0; i < 12; i++) {
    if (ty->workouts[i] != NULL) {

      el = ty->workouts[i];

      while (el) {
        g_assert (KP_IS_CALENDAR_ENTRY (el->data));
        
        xmlAddChild ((xmlNodePtr) doc,
                     kp_calendar_entry_to_xml (KP_CALENDAR_ENTRY (el->data)));
        kp_debug ("Made XML node from entry %u.!", n++);
        el = el->next;
      }
    }
  }
}

/**
 * kp_training_log_get_entry:
 * @log: A #KPTrainingLog
 * @d: Number of day between 1 and 31
 * @m: Number of month between 1 and 12
 * @y: Number of in format yyyy 
 * @mark:
 *
 * Get a #CalendarEntry. DMY must be valid.
 *
 * Returns: A #KPCalendarEntry or NULL if there is no
 *          such entry in the log.
 */
KPCalendarEntry *
kp_training_log_get_entry (KPTrainingLog *log, guint d, guint m, guint y,
                           const gchar *mark)
{
  GList *list;
  gchar *m_str;

  list = kp_training_log_get_day (log, d, m, y);
  while (list) {
    m_str = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));
    
    if (strcmp (m_str, mark) == 0) {
      g_free (m_str);
      return KP_CALENDAR_ENTRY (list->data);
    }
    g_free (m_str);
    
    list = list->next;
  }
  return NULL;
}

/**
 * kp_training_log_get_year:
 * @log: A #KPTrainingLog
 * @year: Year in format yyyy
 * 
 * Get a year-structure for any year. If there aren't any kp_workouts
 * with a date in the given year in the log, NULL will be returned.
 *
 * Otherwise, pointer to a TYear will be returned. It won't be copied,
 * so it MUST NOT be freed by the caller of this function.
 *
 * Returns: Year-structure that represents a year.
 */
TYear *
kp_training_log_get_year (KPTrainingLog * log, guint year)
{
  GList *list;
  TYear *ty;

  list = g_list_first (log->year_list);

  while (list) {
    ty = list->data;

    if (ty->year == year)
      return ty;

    list = list->next;
  }
  return NULL;
}

/**
 * kp_training_log_get_size:
 * @log: A #KPTrainingLog
 *
 * Returns the number of entries in the log
 *
 * Returns: Number of entries in the log.
 */
guint
kp_training_log_get_size (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), 0);
  return log->n_entries;
}

/**
 * kp_training_loG_get_month:
 * @log: A #KPTrainingLog
 * @year: Year number
 * @month: Month number (1-12)
 * 
 * Get all calendar entries in any month. Return the list
 * of those entries or NULL if some error occurs. Entries
 * are not copied, so they MUST NOT be freed.
 *
 * Returns: List of all entries in the month
 */
GList *
kp_training_log_get_month (KPTrainingLog *log, gint year, gint month)
{
  GList *y_list;
  TYear *ty;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (g_date_valid_month (month), NULL);

  ty = NULL;
  for (y_list = g_list_first (log->year_list); y_list != NULL;
       y_list = y_list->next) {
    ty = y_list->data;
    if (ty->year == (guint) year)
      break;
    else
      ty = NULL;
  }

  if (!ty)
    return NULL;

  return ty->workouts[month - 1];
}

/**
 * kp_training_log_get_day:
 * @log: A #KPTrainingLog
 * @d: Day number
 * @m: Month number (1-12)
 * @y: Year number
 * 
 * Get list of all kp_workouts and other entries for any day in
 * the log. As usual, if some error happens, NULL will be returned.
 * Entries are not copied, so they MUST NOT be freed.
 *
 * Returns: List of all the entries in a day.
 */
GList *
kp_training_log_get_day (KPTrainingLog *log, guint d, guint m, guint y)
{
  KPCalendarEntry *entry;
  GList *list = NULL;
  GList *mon;
  GDate *date;
  TYear *ty;
  
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (g_date_valid_month (m), NULL);

  ty = kp_training_log_get_year (log, y);
  if (!ty)
    return NULL; /* There is no kp_workout in year y */
  
  mon = ty->workouts[m-1];

  while (mon) {
    entry = mon->data;
    g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), NULL);
    
    date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (entry)->datetime);
    if (g_date_get_day (date) == d)
      list = g_list_prepend (list, entry);
    mon = mon->next;
  }
  return list; 
}


/**
 * kp_training_log_get_all_workouts: 
 * @log: A training log.
 * 
 * Get a list of ALL entries in the log. List items are not copied
 * (maybe they should be?) so they MUST NOT be freed!
 */
GList *
kp_training_log_get_all_entries (KPTrainingLog * log)
{
  GList *wo_list;
  GList *y_list;
  TYear *ty;
  guint i;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  y_list = g_list_first (log->year_list);
  wo_list = NULL;

  while (y_list) {
    ty = y_list->data;
    for (i = 0; i < 12; i++) {
      /* Go through all months and add items to wo_list */
      g_list_foreach (ty->workouts[i], list_add_item_to_passed_list, &wo_list);
    }
    y_list = y_list->next;
  }
  
  return wo_list;
}


/**
 * Just private helper-func to use with g_list_foreach ().
 */
static void
list_add_item_to_passed_list (gpointer val, gpointer data)
{
  GList **list = (GList **) data;
  *list = g_list_prepend (*list, val);
}

/**
 * kp_training_log_get_workout_params_between:
 * @log: A #KPTrainingLog
 * @mode: #KPTrainingLogDataMode
 * @sport: Show only params when the sport of the workout is this, NULL = all
 * @param: The name of the param
 * @start: Starting date.
 * @end: Ending date.
 * @days_between: Number of days between start and end will be stored
 *   in @days_between.
 * 
 * Get the distance of every kp_workout between two dates and return
 * data as dynamic two-dimensional array. It must be freed with
 * kp_array_2d_free()-func.
 * 
 * If days_between is not NULL, number of days between two dates will
 * be stored in it.
 *
 * Returns: Two-dimensional gdouble-array of data
 */
gdouble **
kp_training_log_get_workout_params_between (KPTrainingLog *log,
                                            KPTrainingLogDataMode mode,
                                            const gchar *sport,
                                            const gchar *param, GDate *start,
                                            GDate *end, guint *days_between)
{
  KPWorkout *wo;
  GDate *date;
  GList *list;
  guint days;
  gdouble **data;
  gdouble item;
  guint idx;

  /* For avg */
  gdouble *avg_sum;
  guint *avg_n;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (start != NULL, NULL);
  g_return_val_if_fail (end != NULL, NULL);
  g_return_val_if_fail (g_date_compare (start, end) <= 0, NULL);
  
  days = g_date_days_between (start, end) + 1;
  data = (gdouble **) kp_array_2d_alloc_init0 (1, days, sizeof (gdouble));
  list = kp_training_log_get_all_entries_between (log, start, end, NULL);

  if (mode == KP_TRAINING_LOG_DATA_AVG) {
    avg_sum = g_malloc0 (days * sizeof (*avg_sum));
    avg_n = g_malloc0 (days * sizeof (*avg_n));
  }
  else {
    /* Prevent a compiler warning */
    avg_sum = NULL;
    avg_n = NULL;
  }
      
  while (list) {
    wo = (KP_IS_WORKOUT (list->data)) ? KP_WORKOUT (list->data) : NULL;
   
    /* If this entry is a workout AND the sport is the same as wanted
     * OR NULL, then add param to the statistics */
    
    if (wo && (!sport || strcasecmp (sport, kp_workout_get_sport (wo)) == 0)) {
      date = kp_calendar_time_get_date (
          KP_CALENDAR_ENTRY (list->data)->datetime);
      g_assert (g_date_valid (date));

      idx = g_date_days_between (start, date);  
      g_assert (idx < days);
  
      item = kp_workout_get_param_double (KP_WORKOUT (list->data), param);
      switch (mode)
      {
        case KP_TRAINING_LOG_DATA_SUM:      
          data[0][idx] += item;
          break;
          
        case KP_TRAINING_LOG_DATA_MIN:
          if (data[0][idx] > 0.0) {
            if (item < data[0][idx])
              data[0][idx] = item;
          } else
            data[0][idx] = item;
          
          break;

        case KP_TRAINING_LOG_DATA_MAX:
          if (item > data[0][idx])
            data[0][idx] = item;
          break;

        case KP_TRAINING_LOG_DATA_AVG:
          if (item > 0.0) {
            avg_n[idx]++;
            avg_sum[idx] += item;
            data[0][idx] = avg_sum[idx] / (gdouble) avg_n[idx];
          }
          break;

        default:
          g_return_val_if_reached (NULL);
      }
    }
    list = list->next;
  }
  if (days_between)
    *days_between = days;

  return data; 
}

/**
 * kp_training_log_get_workout_params_year:
 * @log:
 * @year:
 * @param:
 * @days_between: 
 *
 * Returns: Two-dimensional gdouble-array of data 
 */
gdouble **
kp_training_log_get_workout_params_year (KPTrainingLog * log,
                                         const gchar *param, guint year,
                                         guint * days_between)
{
  gdouble **ret_data;
  gdouble **data;
  guint days;
  GDate *s;
  GDate *e;
  guint m;
  guint i;
 
  ret_data = (gdouble **) kp_array_2d_alloc_init0 (1, 12, sizeof (gdouble));

  s = g_date_new ();
  e = g_date_new ();

  for (m=1; m <= 12; m++) {
    
    g_date_set_dmy (s, 1, m, year);
    g_date_set_dmy (e, kp_get_month_len (m, kp_leap (year)), m, year);
    
    data = kp_training_log_get_workout_params_between (log,
        KP_TRAINING_LOG_DATA_SUM, NULL, param, s, e, &days);
 
    for (i=0, ret_data[0][m-1]=0; i < days; i++)
      ret_data[0][m-1] += data[0][i];
    
    kp_array_2d_free ((void **)data, 1, days);
  }
  *days_between = 12;
  return ret_data;
}

/**
 * traiing_log_get_all_entries_between:
 * @log: a training log.
 * @start: starting date.
 * @end: ending date.
 * 
 * Get all CalendarEntries between two dates.
 *
 * Returns: List of all workouts between start and end dates. Note that
 * the list is NOT a copy. It must NOT be freed. If there some error occurs,
 * NULL will be returned.
 */
GList *
kp_training_log_get_all_entries_between (KPTrainingLog *log, GDate *start,
                                         GDate *end, GList **sports)
{
  GList *dlist = NULL;
  GList *list = NULL;
  KPWorkout *wo;
  GDate *date;
  guint days;

  if (sports)
    *sports = NULL;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (start != NULL, NULL);
  g_return_val_if_fail (end != NULL, NULL);
  g_return_val_if_fail (g_date_compare (start, end) <= 0, NULL);
  
  date = g_date_new_dmy (g_date_get_day (start),
                         g_date_get_month (start),
                         g_date_get_year (start));

  /* +1 is to prevent problem with unsigned int and days >= 0 -thing */
  for (days = g_date_days_between (start, end) + 1; days > 0; days--) {
    dlist = kp_training_log_get_day (log, g_date_get_day (date),
                                          g_date_get_month (date),
                                          g_date_get_year (date));
    list = g_list_concat (list, dlist);

    while (sports && dlist) {
      wo = (KP_IS_WORKOUT (dlist->data)) ? KP_WORKOUT (dlist->data) : NULL;
     
      if (wo && !g_list_find_custom (*sports, kp_workout_get_sport (wo), (GCompareFunc) strcmp))
        *sports = g_list_prepend (*sports, g_strdup (kp_workout_get_sport (wo)));
      
      dlist = dlist->next;
    }
    
    g_date_add_days (date, 1);
  }
  g_date_free (date);

  return list;
}


static void
print_year (gpointer val, gpointer data)
{
  KPWorkout *wo;
  GList *list;
  TYear *y;
  gint i;
  
  y = (TYear *) val;
  g_print ("\nStatistics for year: %d\n\n", y->year);

  for (i = 0; i < 12; i++) {
    list = y->workouts[i];
    while (list) {
      wo = list->data;
      kp_calendar_entry_to_string (KP_CALENDAR_ENTRY ((wo)));
      list = list->next;
    }
  }
}
/*static void
print_param (gpointer key, gpointer val, gpointer data)
{
  printf ("   - %s=\"%s\"\n", (gchar *) key, (gchar *) val);
}*/

/**
 * kp_training_log_dump:
 * @log: A #KPTrainingLog
 *
 * Dump the contents of the log to stdout.
 * Could be used for debugging.
 */
void
kp_training_log_dump (KPTrainingLog *log)
{
  g_return_if_fail (log != NULL);
  g_print ("KPTrainingLog dump:\n");
  g_list_foreach (log->year_list, print_year, NULL);
}



static KPTrainingLog *
kp_training_log_parse_from_xml (const gchar *filename, KPTrainingLog *log_,
                                GError **error)
{
  KPCalendarEntry *entry;
  KPTrainingLog *log;
  xmlNodePtr node;
  xmlDocPtr doc;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  if (log_)
    log = log_;
  else
    log = kp_training_log_new ();
  
  doc = xmlParseFile (filename);
  if (!doc) 
    goto err;

  node = xmlDocGetRootElement (doc);
  if (!node || !node->name || !KP_TAG_MATCH (node, "traininglog"))
    goto err;

  node = node->children;
  
  while (node) {
    /** <workout> **/
    if (KP_TAG_MATCH (node, "workout")) {
      entry = KP_CALENDAR_ENTRY (kp_workout_new ());
      
      if (!kp_calendar_entry_parse (KP_CALENDAR_ENTRY (entry), node))
        return NULL;
      g_return_val_if_fail (KP_IS_WORKOUT (entry), NULL);
      kp_training_log_add (log, entry);
    }
    
    /** <comment> **/
    else if (KP_TAG_MATCH (node, "comment")) {
      entry = KP_CALENDAR_ENTRY (kp_comment_new (NULL, NULL));

      if (!kp_calendar_entry_parse (KP_CALENDAR_ENTRY (entry), node))
        return NULL;
      g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), NULL);
      kp_training_log_add (log, entry);
    }

    /* some text content.. */
    else if (KP_TAG_MATCH (node, "text")) {
      /* DO NOTHING */
    }
    else { 
      g_warning ("Unknown tag in the log: %s\n", node->name);
      goto err;
    }
    node = node->next;
  }
  return log;

err:

  g_set_error (error,
               KP_TRAINING_LOG_ERROR,
               KP_TRAINING_LOG_OPEN_ERROR,
            _("The log (%s) couldn't be opened: %s!"), 
               filename, last_error->str);

  if (doc)
    xmlFreeDoc (doc);
  
  xmlCleanupParser ();

  /* Don't touch the log if it was an existing one and
   * was passed to the function */
  if (log_) {
    return NULL;
  }
  else
    g_object_unref (log);
    
  return NULL;
}


gdouble
kp_training_log_get_week_total (KPTrainingLog *log,
                                GDate *date, const gchar *param)
{
  gdouble sum = 0.0;
  guint i;

  g_return_val_if_fail (date != NULL, 0.0);
  g_return_val_if_fail (g_date_valid (date), 0.0);

  for (i = 0; i < 7; i++) {
    sum += kp_training_log_get_param_day (log, date, param);
    g_date_add_days (date, 1);
  }
  return sum;
}

/**
 * kp_training_log_get_param_day:
 * @log: A #KPTrainingLog.
 * @date: The date of the day as #GDate.
 * @param: Param of which value to get.
 *
 * Return a double value of sum of the entries' params in a day.
 *
 * Returns: a gdouble which can be zero if the requested param is invalid
 * or if its value is zero or if some other error happens.
 */
gdouble
kp_training_log_get_param_day (KPTrainingLog *log, GDate *date,
                               const gchar *param)
{
  KPCalendarEntry *entry;
  gdouble sum = 0.0;
  GList *list;
  GDate *wo_date;
  guint year;
  guint mon;
  TYear *ty;
  gint comp;

  if (log == NULL || log->year_list == NULL)
    return 0.0;

  g_return_val_if_fail (param != NULL, 0.0);
  g_return_val_if_fail (g_date_valid (date), 0.0);

  year = g_date_get_year (date);
  mon = g_date_get_month (date);

  g_return_val_if_fail (mon > 0, 0.0);
  g_return_val_if_fail (mon <= 12, 0.0);

  ty = kp_training_log_get_year (log, year);
  if (!ty)
    return 0.0;

  for (list = ty->workouts[mon - 1]; list != NULL; list = list->next) {
    entry = list->data;

    g_return_val_if_fail (entry != NULL, 0.0);

    wo_date = kp_calendar_time_get_date (entry->datetime);
    g_return_val_if_fail (wo_date != NULL, 0.0);

    comp = g_date_compare (wo_date, date);

    if (comp > 0)
      return sum;

    if (KP_IS_WORKOUT (entry) && comp == 0) {
      if (strcmp (param, "distance") == 0)
        sum += kp_workout_get_distance (KP_WORKOUT (entry));
      else
        sum += kp_workout_get_param_double (KP_WORKOUT (entry), param);
    }
  }

  return sum;
}


/**
 * kp_training_log_foreach:
 * @log: A #KPTrainingLog
 * @func: A func to execute
 * @data: Data to pass to the @func
 *
 * Call some function for each entry in the log.
 *
 * NOTE: This function can not be used to remove all the entries
 * from the log, because kp_training_log_remove () may destroy some
 * structures which are needed to iterate the log! So you must
 * use kp_training_log_remove_all ().
 */
void
kp_training_log_foreach (KPTrainingLog *log, GFunc func, gpointer data)
{
  GList *yl;
  GList *wl;
  TYear *ty;
  guint i;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (func != NULL);

  yl = log->year_list;
  
  while (yl) {
    ty = yl->data;

    g_return_if_fail (ty != NULL);
    
    for (i = 0; i < 12; i++) {
      wl = ty->workouts[i];
    
      while (wl) {
        g_return_if_fail (KP_IS_CALENDAR_ENTRY (wl->data));
        func (KP_CALENDAR_ENTRY (wl->data), data);
        wl = wl->next;
      }
    }
    yl = yl->next;
  }
}

