#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kputil.h"

#include "kpcalendarentryinfodialog.h"
#include "kpnewsplitworkoutdialog.h"
#include "kpnewworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpguiutils.h"
#include "kplogstore.h"
#include "kptreeview.h"

/* Callbacks */
static void       kp_tree_view_init             (KPTreeView *tv);

/* Callback functions */
static void       view_popup_menu               (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
static void       on_popup_add_comment_clicked  (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static void       on_popup_add_split_workout_clicked
                                                (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static void       on_popup_add_workout_clicked  (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static void       on_popup_delete_clicked       (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static void       on_popup_properties_clicked   (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static void       on_popup_delete_clicked       (GtkWidget *menuitem,
                                                 KPTreeView *tv);
static gboolean   on_popup_menu                 (GtkWidget *treeview,
                                                 gpointer data);
static gboolean   on_button_press_event         (GtkWidget *treeview,
                                                 GdkEventButton *event,
                                                 gpointer data);
/* Follow log changes.. */
static void       log_connect_signals           (KPTrainingLog *log,
                                                 KPTreeView *tv);
static void       log_disconnect_signals        (KPTrainingLog *log,
                                                 KPTreeView *tv);
static void       log_changed                   (KPTrainingLog *log,
                                                 gpointer data);
static void       log_entry_removed             (KPTrainingLog *log,
                                                 guint d, guint m, guint y,
                                                 const gchar * mark_str,
                                                 KPTreeView *tv);
static void       log_entry_added               (KPTrainingLog *log,
                                                 guint d, guint m, guint y,
                                                 const gchar *mark,
                                                 KPTreeView *tv);
static GtkWidget *create_popup_menu             (KPTreeView *tv);

typedef struct KPTreeViewPrivateData_
{
  GtkWidget        *popup_menu_title;
  GtkWidget        *popup_menu;

  GtkWidget        *popup_mi_properties;
  GtkWidget        *popup_mi_add;

  KPTrainingLog    *log;
} KPTreeViewPrivateData;

#define KP_TREE_VIEW_PRIVATE_DATA(widget) (((KPTreeViewPrivateData*) \
      (KP_TREE_VIEW (widget)->private_data)))


GType
kp_tree_view_get_type (void)
{
  static GType        kp_tree_view_type = 0;

  if (!kp_tree_view_type) {
    static const GTypeInfo kp_tree_view_info = {
      sizeof (KPTreeViewClass),
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      sizeof (KPTreeView),
      0,
      (GInstanceInitFunc) kp_tree_view_init,
      NULL
    };
    kp_tree_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                               "KPTreeView",
                                               &kp_tree_view_info, 0);
  }
  return kp_tree_view_type;
}


static void
kp_tree_view_init (KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;
  
  tv->private_data = g_new (KPTreeViewPrivateData, 1);
  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes (_("Entry Tree"), ren,
      "markup", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv), col);
  
  g_signal_connect (G_OBJECT (tv), "button-press-event",
                    G_CALLBACK (on_button_press_event), NULL);

  g_signal_connect (G_OBJECT (tv), "popup-menu",
                    G_CALLBACK (on_popup_menu), NULL);
  p_data->popup_menu = create_popup_menu (tv);
}


GtkWidget *
kp_tree_view_new (KPTrainingLog *log)
{
  KPTreeViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_tree_view_get_type (), NULL);

  p_data = KP_TREE_VIEW_PRIVATE_DATA (widget);
  p_data->log = log;
  
  if (p_data->log)
    log_connect_signals (p_data->log, KP_TREE_VIEW (widget));

  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

void
kp_tree_view_set_log (KPTreeView *tv, KPTrainingLog *log)
{
  KPTreeViewPrivateData *p_data;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);
  p_data->log = log;
  
  log_connect_signals (log, tv);
}

void
kp_tree_view_unset_log (KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  log_disconnect_signals (p_data->log, tv);
  p_data->log = NULL;
}


static gboolean
kp_tree_view_get_date (KPTreeView *tv, KPDate *date)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeView *treeview = GTK_TREE_VIEW (tv);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    kp_log_store_get_date (KP_LOG_STORE (model), &iter,
                          &date->d, &date->m, &date->y);

    kp_debug ("Log store gives date: %u.%u.%u", date->d, date->m, date->y);
    
    if (g_date_valid_year (date->y && g_date_valid_month (date->m)
     && g_date_valid_day (date->d))) {
      return TRUE;
    } else if (g_date_valid_year (date->y)&& g_date_valid_month (date->m)) {
      date->d = 1;
      return TRUE;
    } else if (g_date_valid_year (date->y)) {
      date->d = 1;
      date->m = 1;
      return TRUE;
    }
  }
  
  kp_date_set_time (date, time (NULL));
  return TRUE;
}


static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark_str, KPTreeView *tv)
{
  GtkTreeModel *store;
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_TREE_VIEW (tv));

  store = gtk_tree_view_get_model (GTK_TREE_VIEW (tv));
  g_return_if_fail (KP_IS_LOG_STORE (store));

  kp_debug ("Removing mark.");
  
  kp_log_store_remove_mark (KP_LOG_STORE (store), d, m, y, mark_str);
}


static void
log_entry_added (KPTrainingLog *log, guint d, guint m, guint y,
                 const gchar *mark, KPTreeView *tv)
{
  GtkTreeModel *store;
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_TREE_VIEW (tv));

  store = gtk_tree_view_get_model (GTK_TREE_VIEW (tv));
  g_return_if_fail (KP_IS_LOG_STORE (store));

  kp_log_store_add_mark (KP_LOG_STORE (store), d, m, y, mark);
}
              

static void
log_changed (KPTrainingLog *log, gpointer data)
{
}
  
static void
log_connect_signals (KPTrainingLog *log, KPTreeView *tv)
{
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), tv);
  g_signal_connect (G_OBJECT (log), "entry-added",
                    G_CALLBACK (log_entry_added), tv);
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), tv);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPTreeView *tv)
{
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, tv);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, tv);
  g_signal_handlers_disconnect_by_func (log, log_changed, tv);
}

static void
on_popup_delete_clicked (GtkWidget *menuitem, KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeView *treeview = GTK_TREE_VIEW (tv);
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint d,m,y;
  gchar *mark;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);
    gtk_tree_model_get (model, &iter, 0, &mark, -1);

    /* Remove from the log */
    kp_training_log_remove_mark (p_data->log, d, m, y, mark);
    
    /* Remove from log store */
    kp_log_store_remove (KP_LOG_STORE (model), &iter);
    
    kp_debug ("Node deleted.\n");
  }
}


static void
on_popup_add_workout_clicked (GtkWidget *menuitem, KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);
  g_return_if_fail (p_data != NULL);

  kp_tree_view_get_date (tv, &date);
 
  dialog = kp_new_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


static void
on_popup_add_split_workout_clicked (GtkWidget *menuitem, KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);
  g_return_if_fail (p_data != NULL);
 
  kp_tree_view_get_date (tv, &date);
 
  kp_debug ("Adding new split workout to the day %u.%u.%u",
             date.d, date.m, date.y);
  
  dialog = kp_new_split_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

static void
on_popup_add_comment_clicked (GtkWidget *menuitem, KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  KPComment *comment;
  GtkWidget *dialog;
  KPDate kdate;
  GDate *date;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);
  g_return_if_fail (p_data != NULL);

  kp_tree_view_get_date (tv, &kdate);

  date = g_date_new_dmy (kdate.d, kdate.m, kdate.y);
  comment = kp_comment_new (NULL, NULL);
  
  dialog = kp_new_comment_dialog_new (date, comment);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  g_date_free (date);

  if (strlen (comment->title->str) > 0) 
    kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (comment));
  else 
    g_object_unref (G_OBJECT (comment));
}


static void
on_popup_properties_clicked (GtkWidget *menuitem, KPTreeView *tv)
{
  GtkTreeView *treeview = GTK_TREE_VIEW (tv);
  KPTreeViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkWidget *dialog;
  KPCalendarEntry *entry;
  guint d,m,y;
  gchar *mark;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);
    gtk_tree_model_get (model, &iter, 0, &mark, -1);

    if (g_date_valid_dmy (d, m, y)) {

      entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
      g_return_if_fail (entry != NULL);
      
      dialog = kp_calendar_entry_info_dialog_new (entry);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
    }
  }
}


static gboolean
on_button_press_event (GtkWidget *treeview, GdkEventButton *event,
                       gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  if (event->type != GDK_BUTTON_PRESS || event->button != 3)
    return FALSE;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    view_popup_menu (treeview, event, data);
  }

  return TRUE; /* we handled this */
}


static gboolean
on_popup_menu (GtkWidget *treeview, gpointer data)
{
  view_popup_menu (treeview, NULL, data);
  return TRUE; 
}


static GtkWidget *
create_popup_menu (KPTreeView *tv)
{
  KPTreeViewPrivateData *p_data;
  GtkWidget *mi_split_workout;
  GtkWidget *mi_delete;
  GtkWidget *mi_workout;
  GtkWidget *mi_comment;
  GtkWidget *menu;

  p_data = KP_TREE_VIEW_PRIVATE_DATA (tv);

  GladeXML *xml = kp_gui_load ("popups", "cv_popup_menu");
  p_data->popup_menu_title = KP_W (xml, "title");
  p_data->popup_mi_properties = KP_W (xml, "properties"); 
  p_data->popup_mi_add = KP_W (xml, "add");
  mi_split_workout = KP_W (xml, "split_workout");
  mi_delete = KP_W (xml, "delete");
  mi_workout = KP_W (xml, "workout");
  mi_comment = KP_W (xml, "comment");
  menu = KP_W (xml, "cv_popup_menu");
  
  g_signal_connect (G_OBJECT (mi_workout), "activate",
                    G_CALLBACK (on_popup_add_workout_clicked), tv);
  g_signal_connect (G_OBJECT (mi_delete), "activate",
                    G_CALLBACK (on_popup_delete_clicked), tv);
  g_signal_connect (G_OBJECT (p_data->popup_mi_properties), "activate",
                    G_CALLBACK (on_popup_properties_clicked), tv);
  g_signal_connect (G_OBJECT (mi_comment), "activate",
                    G_CALLBACK (on_popup_add_comment_clicked), tv);
  g_signal_connect (G_OBJECT (mi_split_workout), "activate",
                    G_CALLBACK (on_popup_add_split_workout_clicked), tv);

  g_object_unref (xml);
  return menu;
}

static void
view_popup_menu (GtkWidget *treeview, GdkEventButton *event, gpointer data)
{
  KPTreeViewPrivateData *p_data;
  GtkTreeSelection *selection;
  KPLogStoreRecordType type;
  gboolean is_entry = FALSE;
  gboolean is_day = FALSE;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar date_str[32];
  GtkWidget *child;
  guint d, m, y;
  GDate *date;
  gchar *tmp;
  
  p_data = KP_TREE_VIEW_PRIVATE_DATA (treeview);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    type = kp_log_store_get_iter_type (KP_LOG_STORE (model), &iter);
    
    kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);

    switch (type)
    {
      case KP_LOG_STORE_REC_DAY:
        is_day = TRUE;
      case KP_LOG_STORE_REC_ENTRY:
        is_entry = is_day != TRUE;
        date = g_date_new_dmy (d, m, y);
        (void) g_date_strftime (date_str, sizeof (date_str)-1, "%x", date);
        break;

      case KP_LOG_STORE_REC_MONTH:
        tmp = kp_get_month_name (m-1);
        g_snprintf (date_str, sizeof (date_str)-1, "%s %u", tmp, y);
        g_free (tmp);
        break;
    
      case KP_LOG_STORE_REC_YEAR:
        g_snprintf (date_str, sizeof (date_str)-1, "%u", y);
        break;
      
      case KP_LOG_STORE_REC_ROOT:
        g_snprintf (date_str, sizeof (date_str)-1, _("root"));
        break;

      default:
        g_warning ("This is a BUG");
        g_return_if_reached ();
    }
    
    if (p_data->popup_menu_title) {
      child = gtk_bin_get_child (GTK_BIN (p_data->popup_menu_title));
      if (GTK_IS_LABEL (child))
        gtk_label_set_text (GTK_LABEL (child), date_str);
    } else {
      g_return_if_reached ();
    }
  } else
    return;

  gtk_widget_set_sensitive (p_data->popup_mi_properties, is_entry);
  gtk_widget_set_sensitive (p_data->popup_mi_add, is_day);
  
  gtk_menu_popup (GTK_MENU (p_data->popup_menu), NULL, NULL, NULL, NULL,
                  (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));
}

