#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kpworkout.h"
#include "../kpcomment.h"
#include "../kputil.h"

#include "kpviewmodel.h"
#include "kpstatusbar.h"

static void         kp_statusbar_class_init         (KPStatusbarClass *klass);
static void         kp_statusbar_init               (KPStatusbar *bar);
static void         kp_statusbar_finalize           (GObject *object);

static void         log_connect_signals             (KPTrainingLog *log,
                                                     KPStatusbar *bar);
static void         log_disconnect_signals          (KPTrainingLog *log,
                                                     KPStatusbar *bar);
static void         log_entry_removed               (KPTrainingLog *log,
                                                     guint d, guint m, guint y,
                                                     const gchar * mark_str,
                                                     KPStatusbar *bar);
static void         log_entry_added                 (KPTrainingLog *log,
                                                     guint d, guint m, guint y,
                                                     const gchar *mark,
                                                     KPStatusbar *bar);
static void         log_changed                     (KPTrainingLog *log,
                                                     KPStatusbar *bar);
static void         kp_statusbar_update_view_status (KPStatusbar *bar);
static void         kp_statusbar_update_entry_stats (KPStatusbar *bar);

typedef struct KPStatusbarPrivateData_
{
  GString          *message;
  GString          *viewer_name;
  GString          *view_type;

  guint             entries;

  GtkWidget        *label_msg;
  GtkWidget        *label_entries;
  GtkWidget        *label_view;
  
  KPTrainingLog    *log;
} KPStatusbarPrivateData;

#define KP_STATUSBAR_PRIVATE_DATA(widget) (((KPStatusbarPrivateData*) \
      (KP_STATUSBAR (widget)->private_data)))

KPStatusbarFieldData kp_statusbar_field_data[] = {
{KP_STATUSBAR_F_N_ENTRIES, "sb_num_entries_button" ,"sbar_show_entries"       },
{KP_STATUSBAR_F_N_COMMENTS,"sb_num_comments_button","sbar_show_comments"      },
{KP_STATUSBAR_F_N_WORKOUTS,"sb_num_workouts_button","sbar_show_workouts"      },
{KP_STATUSBAR_F_T_DISTANCE,"sb_t_distance_button",  "sbar_show_total_distance"},
{KP_STATUSBAR_F_T_DURATION,"sb_t_duration_button",  "sbar_show_total_duration"},
{KP_STATUSBAR_F_VIEW_TYPE, "sb_view_type",          "sbar_show_view_type"     },
{KP_STATUSBAR_F_VIEW,      "sb_view",               "sbar_show_view"          },
{KP_STATUSBAR_F_N,          NULL                   , NULL                     }
};




GType
kp_statusbar_get_type (void)
{
  static GType        kp_statusbar_type = 0;

  if (!kp_statusbar_type) {
    static const GTypeInfo kp_statusbar_info = {
      sizeof (KPStatusbarClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_statusbar_class_init,
      NULL,
      NULL,
      sizeof (KPStatusbar),
      0,
      (GInstanceInitFunc) kp_statusbar_init,
      NULL
    };
    kp_statusbar_type = g_type_register_static (GTK_TYPE_HBOX,
                                               "KPStatusbar",
                                               &kp_statusbar_info, 0);
  }
  return kp_statusbar_type;
}


static void
kp_statusbar_class_init (KPStatusbarClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_statusbar_finalize;
}


static void
kp_statusbar_init (KPStatusbar *bar)
{
  KPStatusbarPrivateData *p_data;
  
  bar->private_data = g_new (KPStatusbarPrivateData, 1);
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
  p_data->viewer_name = g_string_new (NULL);
  p_data->view_type = g_string_new (NULL);
  p_data->message = g_string_new (NULL);
  p_data->entries = 0;

  p_data->label_msg = gtk_label_new (NULL);
  gtk_box_pack_start (GTK_BOX (bar), p_data->label_msg, FALSE, TRUE, 6);
  gtk_widget_show (p_data->label_msg);

  p_data->label_view = gtk_label_new ("0");
  gtk_box_pack_end (GTK_BOX (bar), p_data->label_view, FALSE, TRUE, 6);
  gtk_widget_show (p_data->label_view);

  p_data->label_entries = gtk_label_new ("0");
  gtk_box_pack_end (GTK_BOX (bar), p_data->label_entries, FALSE, TRUE, 6);
  gtk_widget_show (p_data->label_entries);
}


static void
kp_statusbar_finalize (GObject *object)
{
  KPStatusbarPrivateData *p_data;
  
  kp_debug ("KPStatusbar finalizing!");

  p_data = KP_STATUSBAR_PRIVATE_DATA (object);

  g_string_free (p_data->viewer_name, TRUE);
  g_string_free (p_data->view_type, TRUE);
  g_string_free (p_data->message, TRUE);

  g_free (KP_STATUSBAR (object)->private_data);
}

/**
 * kp_statusbar_new:
 * @log: A #KPTrainingLog, can be NULL
 *
 * Create a new instance of #KPStatusbar.
 */
GtkWidget *
kp_statusbar_new (KPTrainingLog *log)
{
  KPStatusbarPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_statusbar_get_type (), NULL);

  p_data = KP_STATUSBAR_PRIVATE_DATA (widget);
  p_data->log = log;
  
  if (p_data->log)
    log_connect_signals (p_data->log, KP_STATUSBAR (widget));

  return widget;
}

/**
 * kp_statusbar_set_log
 * @bar: A #KPStatusbar
 * @log: A #KPTrainingLog
 *
 * Set the log to use when some log information is shown.
 */
void
kp_statusbar_set_log (KPStatusbar *bar, KPTrainingLog *log)
{
  KPStatusbarPrivateData *p_data;
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
  p_data->log = log;
  
  log_connect_signals (log, bar);

  kp_statusbar_update (bar);
}


/**
 * kp_statusbar_unset_log:
 * @bar: A #KPStatusbar
 *
 * Disconnects the signals and so on.
 */
void
kp_statusbar_unset_log (KPStatusbar *bar)
{
  KPStatusbarPrivateData *p_data;
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);

  if (!KP_IS_TRAINING_LOG (p_data->log))
    return;
  
  log_disconnect_signals (p_data->log, bar);
  p_data->log = NULL;

  kp_statusbar_update (bar);
}

/**
 * kp_statusbar_set_message:
 * @bar: #KPStatusbar
 * @message: Message to show in statusbar, can be NULL
 *
 * Show a message in the statusbar.
 */
void
kp_statusbar_set_message (KPStatusbar *bar, const gchar *message)
{
  KPStatusbarPrivateData *p_data;
  
  g_return_if_fail (KP_IS_STATUSBAR (bar));

  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
  
  g_string_assign (p_data->message, (message) ? message : "");

  kp_statusbar_update (bar);
}


void
kp_statusbar_set_view_type (KPStatusbar *bar, KPViewModelType type)
{
  KPStatusbarPrivateData *p_data;
  gchar *type_str;
  
  g_return_if_fail (KP_IS_STATUSBAR (bar));

  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);

  switch (type)
  {
    case KP_VIEW_MODEL_TYPE_YEAR:
      type_str = _("Year");
      break;

    case KP_VIEW_MODEL_TYPE_MONTH:
      type_str = _("Month");
      break;
      
    case KP_VIEW_MODEL_TYPE_WEEK:
      type_str = _("Week");
      break;
      
    case KP_VIEW_MODEL_TYPE_DAY:
      type_str = _("Day");
      break;

    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      type_str = _("All time");
      break;
      
    default:
      type_str = _("Unknown type");
  }

  g_string_assign (p_data->view_type, type_str);
  kp_statusbar_update_view_status (bar);
}


void
kp_statusbar_set_viewer_name (KPStatusbar *bar, const gchar *name)
{
  KPStatusbarPrivateData *p_data;
  
  g_return_if_fail (KP_IS_STATUSBAR (bar));
  g_return_if_fail (name != NULL);

  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
  g_string_assign (p_data->viewer_name, name);

  kp_statusbar_update_view_status (bar);
}


/**
 *
 *
 */
void
kp_statusbar_set_format_message (KPStatusbar *bar, const gchar *format, ...)
{
  va_list args;
  gchar *str;

  va_start (args, format);
  str = g_strdup_vprintf (format, args);
  va_end (args);

  kp_statusbar_set_message (bar, str);
  
  g_free (str);
}

/**
 * kp_statusbar_get_field:
 * @field: Field type
 *
 * Get some information about the field @field.
 * 
 * Returns: Newly-allocated struct that must be freed.
 */
KPStatusbarFieldData *
kp_statusbar_get_field (KPStatusbarFieldType field)
{
  KPStatusbarFieldData *data;
  
  g_return_val_if_fail (field < KP_STATUSBAR_F_N, NULL);
 
  data = g_malloc (sizeof *data);

  data->name = g_strdup (kp_statusbar_field_data[field].name);
  data->setting = g_strdup (kp_statusbar_field_data[field].setting);
  data->n = field;
  
  return data;
}


static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark_str, KPStatusbar *bar)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_STATUSBAR (bar));

  kp_statusbar_update (bar);
}


static void
log_entry_added (KPTrainingLog *log, guint d, guint m, guint y,
                 const gchar *mark, KPStatusbar *bar)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_STATUSBAR (bar));

  kp_statusbar_update (bar);
}
              

static void
log_changed (KPTrainingLog *log, KPStatusbar *bar)
{
  kp_statusbar_update (bar);
}


static void
log_connect_signals (KPTrainingLog *log, KPStatusbar *bar)
{
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), bar);
  g_signal_connect (G_OBJECT (log), "entry-added",
                    G_CALLBACK (log_entry_added), bar);
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), bar);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPStatusbar *bar)
{
  kp_debug ("Disconnecting log handlers.");
  
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, bar);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, bar);
  g_signal_handlers_disconnect_by_func (log, log_changed, bar);
}


void
kp_statusbar_update (KPStatusbar *bar)
{
  KPStatusbarPrivateData *p_data;

  g_return_if_fail (KP_IS_STATUSBAR (bar));
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);

  if (!KP_IS_TRAINING_LOG (p_data->log))
    return;
  
  kp_statusbar_update_entry_stats (bar);
  kp_statusbar_update_view_status (bar);

  gtk_label_set_text (GTK_LABEL (p_data->label_msg), p_data->message->str);
}


struct log_entry_data
{
  guint n_comments;
  guint n_workouts;

  gdouble distance;
  gdouble duration;
};


static void
foreach_count_func (KPCalendarEntry *entry, struct log_entry_data *data)
{
  if (KP_IS_WORKOUT (entry)) {
    data->duration += kp_workout_get_duration_in_seconds (KP_WORKOUT (entry));
    data->distance += kp_workout_get_distance (KP_WORKOUT (entry));
    data->n_workouts++;
    return;
  }

  if (KP_IS_COMMENT (entry)) {
    data->n_comments++;
    return;
  }
}


static gboolean
show_field (KPStatusbarFieldType type)
{
  KPStatusbarFieldData *data;
  data = &kp_statusbar_field_data[type];
  
  return kp_settings_get_bool (data->setting);
}


static void
kp_statusbar_update_view_status (KPStatusbar *bar)
{
  KPStatusbarPrivateData *p_data;
  GString *string;
  gboolean os; /* Is output started? */

  os = FALSE;
  g_return_if_fail (KP_IS_STATUSBAR (bar));
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
 
  string = g_string_new_len (NULL, 0);
 
  if (show_field (KP_STATUSBAR_F_VIEW)) {
    g_string_append_printf (string, _("Viewer: <b>%s</b>"),
                            p_data->viewer_name->str);
    os = 1;
  }

  if (show_field (KP_STATUSBAR_F_VIEW_TYPE)) {
    if (os)
      g_string_append (string, " ");
    g_string_append_printf (string, _("Type: <b>%s</b>"),
                            p_data->view_type->str);
    os = 1;
  }
  
  gtk_label_set_markup (GTK_LABEL (p_data->label_view), string->str);

  g_string_free (string, TRUE);
}

static void
kp_statusbar_update_entry_stats (KPStatusbar *bar)
{
  KPStatusbarPrivateData *p_data;
  struct log_entry_data data = { 0, 0, 0.0, 0.0 };
  GString *string;
  guint n_entries;
  gchar *time;
  gboolean os; /* Is output started? */

  os = FALSE;
  g_return_if_fail (KP_IS_STATUSBAR (bar));
  p_data = KP_STATUSBAR_PRIVATE_DATA (bar);
 
  n_entries = (p_data->log) ? kp_training_log_get_size (p_data->log) : 0;

  kp_training_log_foreach (p_data->log, (GFunc) foreach_count_func, &data);
 
  string = g_string_new_len (NULL, 0);
 
  if (show_field (KP_STATUSBAR_F_T_DURATION)) {
    if (os)
      g_string_append (string, ", ");
  
    time = kp_date_seconds_to_string (data.duration);
    g_string_append_printf (string, "%s", time);

    os = (strlen (time) != 0);
    g_free (time);
  }

  if (show_field (KP_STATUSBAR_F_T_DISTANCE)) {
    if (os)
      g_string_append (string, ", ");
    g_string_append_printf (string, _("<b>%.0f</b> km"), data.distance);
    os = 1;
  }

  /* Number of comments */
  if (show_field (KP_STATUSBAR_F_N_COMMENTS)) {
    if (os)
      g_string_append (string, ", ");
    g_string_append_printf (string, _("<b>%d</b> Comments"), data.n_comments);
    os = 1;
  }

  /* Number of workouts */
  if (show_field (KP_STATUSBAR_F_N_WORKOUTS)) {
    if (os)
      g_string_append (string, ", ");
    g_string_append_printf (string, _("<b>%d</b> Workouts"), data.n_workouts);
    os = 1;
  }
 
  /* Number of entries */
  if (show_field (KP_STATUSBAR_F_N_ENTRIES)) {
    if (os)
      g_string_append (string, ", ");
    g_string_append_printf (string, _("<b>%d</b> Entries"), n_entries);
    os = 1;
  }
  
  if (os) 
    g_string_prepend (string, _("Log: "));
 
  gtk_label_set_markup (GTK_LABEL (p_data->label_entries), string->str);

  g_string_free (string, TRUE);
}

