#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <time.h>

#include "kpnewworkoutdialog.h"
#include "kpguiutils.h"

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kputil.h"
#include "../kpworkout.h"

#define SPORT_MENU_TEXT    "Choose a Sport"
#define SPORT_MENU_OTHER   "Other"

static void     kp_new_workout_dialog_class_init (KPNewWorkoutDialogClass *klass);
static void     kp_new_workout_dialog_init       (KPNewWorkoutDialog * dialog);
static void     kp_new_workout_dialog_finalize   (GObject *object);
static void     kp_new_workout_dialog_response_cb(GtkDialog *gtkdialog,
                                                  int response_id,
                                                  KPNewWorkoutDialog *dialog);

static gboolean export_list_row                  (GtkTreeModel *model,
                                                  GtkTreePath *path,
                                                  GtkTreeIter *iter,
                                                  gpointer data);
static void     detail_list_row_selected         (GtkTreeSelection *selection,
                                                  gpointer data);
static gboolean set_time                         (KPNewWorkoutDialog *dialog,
                                                  KPWorkout *wo);
static void     set_params                       (KPNewWorkoutDialog *dialog);
static void     set_comment                      (KPNewWorkoutDialog *dialog);
static void     set_detail_tree_view             (KPNewWorkoutDialog *dialog);
static void     build_type_menu                  (KPNewWorkoutDialog *dialog);
static void     build_detail_menu                (KPNewWorkoutDialog *dialog);
static gboolean check_time                       (KPNewWorkoutDialog *dialog);
static void     check_values                     (GtkButton *button,
                                                  KPNewWorkoutDialog *dialog);
static void     on_event_entry_changed           (GtkEntry *entry,
                                                  gpointer data);
static void     on_new_wo_d_add_button_clicked   (GtkButton *button,
                                                  gpointer data);
static void     on_new_wo_d_delete_button_clicked(GtkButton *button,
                                                  gpointer data);
static void     on_sport_menu_activate           (GtkMenuItem *menuitem,
                                                  gpointer data);
static void     on_time_togglebutton_clicked     (GtkToggleButton *button,
                                                  gpointer data);
static void     on_simple_timefmt_button_clicked (GtkCheckButton *button,
                                                  gpointer data);
static gchar   *get_given_sport                  (KPNewWorkoutDialog *dialog);

static GuiModuleSignalsData signals_data[] = {
{"new_wo_d_add_button",   "clicked",CB(on_new_wo_d_add_button_clicked),   NULL},
{"new_wo_d_delete_button","clicked",CB(on_new_wo_d_delete_button_clicked),NULL},
{"new_wo_type_menu",      "clicked",CB(on_sport_menu_activate),           NULL},
{"time_togglebutton",     "toggled",CB(on_time_togglebutton_clicked),     NULL},
{"simple_timefmt_button", "clicked",CB(on_simple_timefmt_button_clicked), NULL},
{"simple_timefmt_button", "toggled",CB(check_values),                     NULL},
{"event_entry",           "changed",CB(on_event_entry_changed),           NULL},
{"time_entry",            "changed",CB(check_values),                     NULL},
{"new_wo_type_menu",      "clicked",CB(check_values),                     NULL},
{"new_wo_distance_entry", "changed",CB(check_values),                     NULL},
{"new_wo_duration_entry", "changed",CB(check_values),                     NULL},
{ NULL,                    NULL,    NULL,                                 NULL},
};


typedef struct KPNewWorkoutDialogPrivateData_
{
  KPTrainingLog *log;
  KPWorkout *wo;

  GtkWidget *details_view;
  GtkWidget *detail_add_button;
  GtkWidget *detail_delete_button;
  GtkWidget *detail_preset_menu;
  GtkWidget *detail_name;
  GtkWidget *detail_value;

  GtkWidget *time_entry;
  GtkWidget *time_button;
  GtkWidget *time_hint;
  GtkWidget *time_fmt_button;
 
  GtkWidget *event_entry;
  GtkWidget *okbutton;
  GtkWidget *cancelbutton;
  GtkWidget *duration_entry;
  GtkWidget *distance_entry;
  GtkWidget *type_menu;
  GtkWidget *comment;
  GtkWidget *calendar;
} KPNewWorkoutDialogPrivateData;

#define KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA(widget) \
  (((KPNewWorkoutDialogPrivateData*) \
  (KP_NEW_WORKOUT_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_new_workout_dialog_get_type (void)
{
  static GType kp_new_workout_dialog_type = 0;

  if (kp_new_workout_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPNewWorkoutDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_new_workout_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPNewWorkoutDialog),
      0,
      (GInstanceInitFunc) kp_new_workout_dialog_init,
      NULL,
    };

    kp_new_workout_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                         "KPNewWorkoutDialog",
                                                         &our_info, 0);
  }
  return kp_new_workout_dialog_type;
}


static void
kp_new_workout_dialog_class_init (KPNewWorkoutDialogClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_new_workout_dialog_finalize;
}


static void
kp_new_workout_dialog_init (KPNewWorkoutDialog * dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GladeXML *xml;

  xml = kp_gui_load ("new_workout", "new_workout");
  
  dialog->private_data = g_new0 (KPNewWorkoutDialogPrivateData, 1);
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  g_signal_connect (G_OBJECT (dialog),
                   "response",
                    G_CALLBACK (kp_new_workout_dialog_response_cb), dialog);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("New Workout"));

  p_data->cancelbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                GTK_STOCK_CANCEL,
                                                GTK_RESPONSE_CANCEL);

  p_data->okbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                            GTK_STOCK_ADD,
                                            GTK_RESPONSE_OK);

  gtk_widget_set_sensitive (p_data->okbutton, FALSE);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "new_workout"));

  p_data->wo = kp_workout_new ();
  
  /* Put widgets to private data */
  p_data->details_view = KP_W (xml, "new_wo_details_treeview");
  p_data->detail_add_button = KP_W (xml, "new_wo_d_add_button");  
  p_data->detail_delete_button = KP_W (xml, "new_wo_d_delete_button");  
  p_data->detail_value = KP_W (xml, "new_wo_detail_value_entry");
  p_data->detail_preset_menu = KP_W (xml, "detail_preset_menu");

  p_data->duration_entry = KP_W (xml, "new_wo_duration_entry");
  p_data->distance_entry = KP_W (xml, "new_wo_distance_entry");
  p_data->type_menu = KP_W (xml, "new_wo_type_menu");
  p_data->comment = KP_W (xml, "new_wo_comment");
  p_data->event_entry = KP_W (xml, "event_entry");
  p_data->time_fmt_button = KP_W (xml, "simple_timefmt_button");

  p_data->time_button = KP_W (xml, "time_togglebutton");
  p_data->time_entry = KP_W (xml, "time_entry");
  
  build_type_menu (dialog);
  build_detail_menu (dialog);

  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  g_object_unref (G_OBJECT (xml));
}


static void
kp_new_workout_dialog_finalize (GObject *object)
{
  KPNewWorkoutDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_NEW_WORKOUT_DIALOG (object));

  dialog = KP_NEW_WORKOUT_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


GtkWidget *
kp_new_workout_dialog_new (KPDate *date, KPTrainingLog *log)
{
  KPNewWorkoutDialogPrivateData *p_data;
  KPNewWorkoutDialog *dialog;

  dialog = g_object_new (KP_TYPE_NEW_WORKOUT_DIALOG, NULL);
  
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  p_data->log = log;

  KP_CALENDAR_ENTRY (p_data->wo)->datetime = kp_calendar_time_new ();
  kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                            date->d, date->m, date->y);
  
  set_detail_tree_view (dialog);

  return GTK_WIDGET (dialog);
}


/*
 * button is not used, so it can be NULL and this can be called
 * from other functions and not only as callback for buttons.
 */
static void
check_values (GtkButton *button, KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  gboolean valid_distance = FALSE;
  gboolean valid_duration = FALSE;
  gboolean valid = TRUE;
  gchar *sport;
  
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  kp_debug ("Checking values of the workout.\n");
 
  /* If simple time isn't used, the field must be correct */
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (p_data->time_fmt_button)))
    if (!check_time (dialog))
      valid = FALSE;
   
  valid_distance =
    (g_strtod (gtk_entry_get_text (GTK_ENTRY (p_data->distance_entry)), NULL) > 0.0);
   
  valid_duration = kp_is_valid_duration_entry_str (
      gtk_entry_get_text (GTK_ENTRY (p_data->duration_entry)));

  if (!valid_distance && !valid_duration)
    valid = FALSE;

  if (!(sport = get_given_sport (dialog)))
    valid = FALSE;
 
  gtk_widget_set_sensitive (p_data->okbutton, valid);
}

/* 
 * This function returns the selected sport or NULL
 * if nothing valid is selected.
 */
static gchar *
get_given_sport (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  const gchar *sport_str;
  GtkWidget *menu;
  GtkWidget *active;
  gchar *tmp;

  g_return_val_if_fail (KP_IS_NEW_WORKOUT_DIALOG (dialog), NULL);
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (p_data->type_menu));
  g_return_val_if_fail (menu != NULL, NULL);

  active = gtk_menu_get_active (GTK_MENU (menu));
  g_return_val_if_fail (active != NULL, NULL);
  
  tmp = g_object_get_data (G_OBJECT (active), "str");
  g_return_val_if_fail (tmp != NULL, NULL);
  
  /* Sport is not what it was initially set, so OK */
  if (!strcmp (SPORT_MENU_TEXT, tmp) == 0) {
    /* User has selected "Other" from menu and maybe written the sport */
    if (strcmp (SPORT_MENU_OTHER, tmp) == 0) {
      sport_str = gtk_entry_get_text (GTK_ENTRY (p_data->event_entry));
      if (!sport_str || strlen (sport_str) == 0)
        return NULL;

      tmp = g_strdup (sport_str);
    }
    /* User has written a valid sport,
       or user has selected a sport from the menu */
    return g_strdup (tmp);
  }
  /* Sport is not selected by user, so this workout can't be added yet */
  else 
    return NULL;
}


static void
kp_new_workout_dialog_response_cb (GtkDialog *gtk_dialog, int response_id,
                                   KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkEntry *duration_entry;
  gchar *sport;
  guint32 sec;
  
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  if (response_id == GTK_RESPONSE_CANCEL)
    return;

  duration_entry = GTK_ENTRY (p_data->duration_entry);
  sec = kp_duration_str_to_ms (gtk_entry_get_text (duration_entry));

  kp_workout_set_duration_in_milliseconds (p_data->wo, sec);

  g_return_if_fail (set_time (dialog, p_data->wo));
  
  set_params (dialog);
  set_comment (dialog);

  sport = get_given_sport (KP_NEW_WORKOUT_DIALOG (gtk_dialog));
  g_return_if_fail (sport != NULL);
  kp_workout_set_sport (p_data->wo, sport);
    
  g_return_if_fail (kp_workout_is_valid (p_data->wo));

  kp_debug ("Adding workout to the log.\n");
  kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (p_data->wo));
}


static void 
build_type_menu (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *sports;
  GSList *list;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label (_(SPORT_MENU_TEXT));
  g_object_set_data (G_OBJECT (item), "str", g_strdup (SPORT_MENU_TEXT));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (G_OBJECT (item), "activate",
                    G_CALLBACK (on_sport_menu_activate), dialog);
  
  sports = kp_settings_get_list ("preset_sport");
  list = sports;
  
  while (list) {
    item = gtk_menu_item_new_with_label (list->data);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    g_object_set_data (G_OBJECT (item), "str", g_strdup (list->data));
    g_signal_connect (G_OBJECT (item), "activate",
                      G_CALLBACK (on_sport_menu_activate), dialog);
    list = list->next;
  }
  kp_settings_list_free (sports);

  item = gtk_menu_item_new_with_label (_(SPORT_MENU_OTHER));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_object_set_data (G_OBJECT (item), "str", g_strdup (SPORT_MENU_OTHER));
  g_signal_connect (G_OBJECT (item), "activate",
                    G_CALLBACK (on_sport_menu_activate), dialog);
  
  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->type_menu), menu);
  gtk_widget_show_all (menu);
}


static void
on_simple_timefmt_button_clicked (GtkCheckButton *button, gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkWidget *show;
  GtkWidget *hide;
  gboolean use_simple;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  use_simple = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
  
  hide = (use_simple) ? p_data->time_entry : p_data->time_button;
  show = (use_simple) ? p_data->time_button : p_data->time_entry;
  
  gtk_widget_hide (hide);
  gtk_widget_show (show);
}


static void
on_sport_menu_activate (GtkMenuItem *menuitem, gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  gchar *label_text;
  GtkWidget *label;
  
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  label = gtk_bin_get_child (GTK_BIN (menuitem));

  g_return_if_fail (GTK_IS_LABEL (label));
  label_text = g_strdup (gtk_label_get_text (GTK_LABEL (label)));

  if (strcmp (label_text, _(SPORT_MENU_TEXT)) == 0)
    goto check;

  if (strcmp (label_text, _(SPORT_MENU_OTHER))  == 0) {
    gtk_widget_show (p_data->event_entry);
    kp_workout_set_sport (p_data->wo, NULL);
  } else {
    kp_workout_set_sport (p_data->wo, g_strstrip (label_text));
    g_free (label_text);

    if (GTK_WIDGET_VISIBLE (p_data->event_entry))
      gtk_widget_hide (p_data->event_entry);
  }

check:
  check_values (NULL, KP_NEW_WORKOUT_DIALOG (data));
}


static void
on_time_togglebutton_clicked (GtkToggleButton *button, gpointer data)
{
  gchar *str;
  gtk_toggle_button_set_mode (button, FALSE);
  
  if (gtk_toggle_button_get_active (button) == TRUE)
    str = _("PM");
  else
    str = _("AM");
      
  gtk_button_set_label (GTK_BUTTON (button), str);
}


static void
on_event_entry_changed (GtkEntry *entry, gpointer data)
{
  check_values (NULL, KP_NEW_WORKOUT_DIALOG (data));  
}


static void
set_comment (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkTextBuffer *buffer;
  GtkTextView *textview;
  GtkTextIter start;
  GtkTextIter end;
  GValue comment;
  gchar *text;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  textview = GTK_TEXT_VIEW (p_data->comment);
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
  text = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);

  comment.g_type = 0;  
  
  g_assert (text != NULL);
  
  g_value_init (&comment, G_TYPE_STRING);
  g_value_set_string (&comment, text); 
  kp_workout_set_param (p_data->wo, TRUE, "comment", &comment);
  g_value_unset (&comment);
  g_free (text);
}


static gboolean
set_time (KPNewWorkoutDialog *dialog, KPWorkout * wo)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkToggleButton *time_button;
  const gchar *time;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  time_button = GTK_TOGGLE_BUTTON (p_data->time_fmt_button);
  
  if (gtk_toggle_button_get_active (time_button)) {
    time = gtk_button_get_label (GTK_BUTTON (p_data->time_button));
    if (strcmp (time, _("AM")) == 0)
      kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime, 10, 10, 10, 0);
    else
      kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime, 16, 16, 16, 0);
  } else {
    time = gtk_entry_get_text (GTK_ENTRY (p_data->time_entry));

    if (!check_time (dialog))
      return FALSE;
   
    kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (wo)->datetime,
                           (guint) g_ascii_strtod (&time[0], NULL),
                           (guint) g_ascii_strtod (&time[3], NULL),
                           /* we don't need seconds or smaller units here. */
                           0, 0);
  }
  return TRUE;
}


static gboolean
check_time (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkToggleButton *button;
  const gchar *time;
  guint i;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  button = GTK_TOGGLE_BUTTON (p_data->time_fmt_button);
  
  if (!gtk_toggle_button_get_active (button)) {
    /* if simple times are not used */
   
    time = gtk_entry_get_text (GTK_ENTRY (p_data->time_entry));
    
    if (strlen (time) != 5)
      return FALSE;

    for (i=0; i < 5; i++)
      if (i != 2 && !g_ascii_isdigit (time[i]))
        return FALSE;

    if (time[2] != ':')
      return FALSE;
  } 
  return TRUE;
}


static void
set_params (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkTreeView *view;
  GValue dist_value;
  gdouble distance;
  
  const gchar *dist_str;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  dist_value.g_type = 0;
  
  view = GTK_TREE_VIEW (p_data->details_view);

  dist_str = gtk_entry_get_text (GTK_ENTRY (p_data->distance_entry));
  distance = kp_number (dist_str);

  g_value_init (&dist_value, G_TYPE_DOUBLE);
  g_value_set_double (&dist_value, distance);

  if (distance > 0.0)
    if (!kp_workout_set_param (p_data->wo, TRUE, "distance", &dist_value))
      g_warning ("Couldn't add param distance to workout which IS SERIOUS!\n");
  
  gtk_tree_model_foreach (gtk_tree_view_get_model (view), export_list_row,
                          dialog);
}


static gboolean
export_list_row (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
                 gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  const gchar *key;
  const gchar *val;
  GValue *value;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  g_return_val_if_fail (p_data != NULL, TRUE); 
  
  gtk_tree_model_get (model, iter, 0, &key, 1, &val, -1);

  value = kp_g_value_from_string (val);
  kp_workout_set_param (p_data->wo, TRUE, key, value);

  return FALSE;
}

/*
 * notebook's note 'DETAILS'
 * 
 */
static void
set_detail_tree_view (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkTreeViewColumn *column2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->details_view);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Field"), renderer,
                                                     "text", 0, NULL);
  column2 = gtk_tree_view_column_new_with_attributes (_("Value"), renderer,
                                                     "text", 1, NULL);
  gtk_tree_view_append_column (treeview, column1);
  gtk_tree_view_append_column (treeview, column2);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);
}

static void
detail_list_row_selected (GtkTreeSelection * selection,
                          gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  gtk_widget_set_sensitive (p_data->detail_delete_button, TRUE);
}


static void
on_new_wo_d_delete_button_clicked (GtkButton * button,
                                   gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->details_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
}


static void
on_new_wo_d_add_button_clicked (GtkButton * button,
                                gpointer data)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkWidget *treeview;
  GtkTreeModel *model;
  GtkTreeIter iter;

  gchar *value_str;
  gchar *name;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (KP_NEW_WORKOUT_DIALOG (data));
  
  treeview = p_data->details_view;

  value_str = g_strdup (gtk_entry_get_text (GTK_ENTRY (p_data->detail_value)));

  name = kp_gui_get_option_menu_active (
      GTK_OPTION_MENU (p_data->detail_preset_menu));
  g_return_if_fail (name != NULL);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, name, 1, value_str, -1);

  g_free (value_str);
  g_free (name);
  
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_value), "");
}


static void
build_detail_menu (KPNewWorkoutDialog *dialog)
{
  KPNewWorkoutDialogPrivateData *p_data;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *details;
  GSList *list;

  p_data = KP_NEW_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label (_("Choose preset detail"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  
  
  details = kp_settings_get_list ("preset_detail");
  list = details;
  
  while (list) {
    item = gtk_menu_item_new_with_label (list->data);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    
    list = list->next;
  }
  kp_settings_list_free (details);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->detail_preset_menu), menu);
  gtk_widget_show_all (menu);
}

