#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <time.h>

#include "kpnewsplitworkoutdialog.h"
#include "kpguiutils.h"

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kputil.h"
#include "../kpworkout.h"

static void     kp_new_split_workout_dialog_class_init  (KPNewSplitWorkoutDialogClass *klass);
static void     kp_new_split_workout_dialog_init        (KPNewSplitWorkoutDialog *dialog);
static void     kp_new_split_workout_dialog_finalize    (GObject *object);
static void     kp_new_split_workout_dialog_response_cb (GtkDialog *gtkdialog,
                                                         int response_id,
                                                         KPNewSplitWorkoutDialog * dialog);
static void     build_detail_menu                       (KPNewSplitWorkoutDialog *dialog);
static void     set_detail_tree_view                    (KPNewSplitWorkoutDialog *dialog);
static gboolean export_list_row                         (GtkTreeModel *model,
                                                         GtkTreePath *path,
                                                         GtkTreeIter *iter,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     detail_list_row_selected                (GtkTreeSelection * selection,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_detail_add_button_clicked            (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_detail_delete_button_clicked         (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
/* SPLITS */
static void     split_list_row_selected                 (GtkTreeSelection *selection,
                                                         KPNewSplitWorkoutDialog *dialog);
static gboolean export_split_list_row                   (GtkTreeModel *model,
                                                         GtkTreePath *path,
                                                         GtkTreeIter *iter,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     set_splits_tree_view                    (KPNewSplitWorkoutDialog *dialog);
static void     on_split_add_button_clicked             (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_split_delete_button_clicked          (GtkButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static gboolean set_time                                (KPNewSplitWorkoutDialog *dialog,
                                                         KPWorkout *wo);
static void     set_date                                (KPNewSplitWorkoutDialog *dialog,
                                                         guint d,
                                                         guint m,
                                                         guint y);
static void     set_comment                             (KPNewSplitWorkoutDialog *dialog);
static void     build_sports_menu                       (KPNewSplitWorkoutDialog *dialog);
static gboolean check_time                              (KPNewSplitWorkoutDialog *dialog);
static void     check_values                            (GtkWidget *dont_use,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_sport_entry_changed                  (GtkEditable *editable,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_sports_menu_activate                 (GtkMenuItem *menuitem,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_time_togglebutton_clicked            (GtkToggleButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);
static void     on_simple_timefmt_button_clicked        (GtkCheckButton *button,
                                                         KPNewSplitWorkoutDialog *dialog);

static GuiModuleSignalsData signals_data[] = {
{"detail_add_button",     "clicked",CB (on_detail_add_button_clicked),    NULL},
{"detail_delete_button",  "clicked",CB (on_detail_delete_button_clicked), NULL},
{"split_add_button",      "clicked",CB (on_split_add_button_clicked),     NULL},
{"split_delete_button" ,  "clicked",CB (on_split_delete_button_clicked),  NULL},
{"sports_menu",           "clicked",CB (on_sports_menu_activate),         NULL},
{"time_togglebutton",     "toggled",CB (on_time_togglebutton_clicked),    NULL},
{"simple_timefmt_button", "clicked",CB (on_simple_timefmt_button_clicked),NULL},
{"sport_entry",          "activate",CB (on_sport_entry_changed),          NULL},
{"sport_entry",          "activate",CB (check_values),                    NULL},
{"sports_menu",           "clicked",CB (check_values),                    NULL},
{"simple_timefmt_button", "toggled",CB (check_values),                    NULL},
{"time_entry",            "changed",CB (check_values),                    NULL},
{ NULL,                    NULL,    NULL,                                 NULL},
};


typedef struct KPNewSplitWorkoutDialogPrivateData_
{
  KPTrainingLog  *log;
  KPWorkout      *wo;
 
  GtkWidget    *details_view;
  GtkWidget    *detail_add_button;
  GtkWidget    *detail_delete_button;
  GtkWidget    *detail_preset_menu;
  GtkWidget    *detail_value;

  GtkWidget    *time_entry;
  GtkWidget    *time_button;
  GtkWidget    *time_hint_simple;
  GtkWidget    *time_hint;
  GtkWidget    *time_fmt_button;

  GtkWidget    *split_spin;
  GtkWidget    *split_type_menu;
  GtkWidget    *split_distance_spin;
  GtkWidget    *split_duration_entry;
  GtkWidget    *split_comment_entry;
  GtkWidget    *split_treeview;
  GtkWidget    *split_delete_button;
  GtkWidget    *split_add_button;

  GtkWidget    *sports_menu;
  GtkWidget    *okbutton;
  GtkWidget    *cancelbutton;
  GtkWidget    *comment;
  GtkWidget    *sport_entry;
  GtkWidget    *calendar;
} KPNewSplitWorkoutDialogPrivateData;

#define KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA(widget) (((KPNewSplitWorkoutDialogPrivateData*) \
        (KP_NEW_SPLIT_WORKOUT_DIALOG (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_new_split_workout_dialog_get_type (void)
{
  static GType kp_new_split_workout_dialog_type = 0;

  if (kp_new_split_workout_dialog_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPNewSplitWorkoutDialogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_new_split_workout_dialog_class_init,
      NULL,
      NULL,
      sizeof (KPNewSplitWorkoutDialog),
      0,
      (GInstanceInitFunc) kp_new_split_workout_dialog_init,
      NULL,
    };

    kp_new_split_workout_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                         "KPNewSplitWorkoutDialog",
                                                         &our_info, 0);
  }

  return kp_new_split_workout_dialog_type;
}

static void
kp_new_split_workout_dialog_class_init (KPNewSplitWorkoutDialogClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_new_split_workout_dialog_finalize;
}

static void
kp_new_split_workout_dialog_init (KPNewSplitWorkoutDialog * dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GladeXML *xml;

  xml = kp_gui_load ("new_split_workout_dialog", "split_workout");
  
  dialog->private_data = g_new0 (KPNewSplitWorkoutDialogPrivateData, 1);
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  g_signal_connect (G_OBJECT (dialog),
                   "response",
                    G_CALLBACK (kp_new_split_workout_dialog_response_cb), dialog);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_NONE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("New Workout"));

  p_data->cancelbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                GTK_STOCK_CANCEL,
                                                GTK_RESPONSE_CANCEL);

  p_data->okbutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                            GTK_STOCK_ADD,
                                            GTK_RESPONSE_OK);

  gtk_widget_set_sensitive (p_data->okbutton, FALSE);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "split_workout"));

  p_data->wo = kp_workout_new ();
  
  /* Put widgets to private data */
  p_data->details_view = KP_W (xml, "details_view");
  p_data->detail_add_button = KP_W (xml, "detail_add_button");  
  p_data->detail_delete_button = KP_W (xml, "detail_delete_button");  
  p_data->detail_value = KP_W (xml, "detail_value_entry");
  p_data->detail_preset_menu = KP_W (xml, "detail_preset_menu");

  p_data->split_spin = KP_W (xml, "split_spin");
  p_data->split_type_menu = KP_W (xml, "split_type_menu");
  p_data->split_distance_spin = KP_W (xml, "distance_spin");
  p_data->split_duration_entry = KP_W (xml, "duration_entry");
  p_data->split_comment_entry = KP_W (xml, "comment_entry");
  p_data->split_treeview = KP_W (xml, "split_view");
  p_data->split_delete_button = KP_W (xml, "split_delete_button");
  p_data->split_add_button = KP_W (xml, "split_add_button");

  p_data->sports_menu = KP_W (xml, "sports_menu");
  p_data->comment = KP_W (xml, "comment_textview");
  p_data->sport_entry = KP_W (xml, "sport_entry");
  p_data->time_fmt_button = KP_W (xml, "simple_timefmt_button");

  p_data->time_button = KP_W (xml, "time_togglebutton");
  p_data->time_entry = KP_W (xml, "time_entry");
  
  p_data->time_hint = KP_W (xml, "time_hint_label");
  p_data->time_hint_simple = KP_W (xml, "time_hint_simple");
  p_data->calendar = KP_W (xml, "calendar");
  
  build_sports_menu (dialog);
  build_detail_menu (dialog);

  kp_gui_module_signals_connect_data (xml, signals_data, dialog);
  g_object_unref (G_OBJECT (xml));
}

static void
kp_new_split_workout_dialog_finalize (GObject *object)
{
  KPNewSplitWorkoutDialog *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_NEW_SPLIT_WORKOUT_DIALOG (object));

  dialog = KP_NEW_SPLIT_WORKOUT_DIALOG (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
kp_new_split_workout_dialog_new (KPDate *date, KPTrainingLog *log)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPNewSplitWorkoutDialog *dialog;

  dialog = g_object_new (KP_TYPE_NEW_SPLIT_WORKOUT_DIALOG, NULL);
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  p_data->log = log;

  KP_CALENDAR_ENTRY (p_data->wo)->datetime = kp_calendar_time_new ();
  kp_calendar_time_set_dmy (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                            date->d, date->m, date->y);
  
  set_date (dialog, date->d, date->m, date->y);
  set_detail_tree_view (dialog);
  set_splits_tree_view (dialog);

  return GTK_WIDGET (dialog);
}


static void
check_values (GtkWidget *dont_use, KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  gboolean valid = TRUE; 
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  kp_debug ("Checking values of the workout.");
 
  /* If simple time isn't used, the field must be correct */
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (p_data->time_fmt_button)))
    if (!check_time (dialog))
      valid = FALSE;
   
  if (!kp_workout_get_sport (p_data->wo))
    valid = FALSE;
  
  gtk_widget_set_sensitive (p_data->okbutton, valid);
}


static void
kp_new_split_workout_dialog_response_cb (GtkDialog * gtk_dialog,
    int response_id, KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  if (response_id == GTK_RESPONSE_CANCEL)
    return;

  g_return_if_fail (set_time (dialog, p_data->wo));
 
  /* Add details to the workout */
  gtk_tree_model_foreach (gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view)),
                         (GtkTreeModelForeachFunc) export_list_row,
                         (gpointer) dialog);

  /* Add splits to the workout */
  gtk_tree_model_foreach (gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->split_treeview)),
                         (GtkTreeModelForeachFunc) export_split_list_row,
                         (gpointer) dialog);
  
  /* Add text from the textview to the workout */
  set_comment (dialog);
 
  /* If workout is valid, add it to the log. */
  if (kp_workout_is_valid (p_data->wo)) 
    kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (p_data->wo));
  else
    g_return_if_reached ();
}


static void 
build_sports_menu (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *list;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  list = kp_settings_get_list ("preset_sport");
  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label (_("Choose a sport"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  
  while (list) {
    item = gtk_menu_item_new_with_label (list->data);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

    g_signal_connect (G_OBJECT (item), "activate",
                      G_CALLBACK (on_sports_menu_activate), dialog);
    g_signal_connect (G_OBJECT (item), "activate",
                      G_CALLBACK (check_values), dialog);
    list = list->next;
  }
  kp_settings_list_free (list);

  item = gtk_menu_item_new_with_label (_("Other"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  g_signal_connect (G_OBJECT (item), "activate",
                    G_CALLBACK (on_sports_menu_activate), dialog);
 
  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->sports_menu), menu);
  gtk_widget_show_all (menu);
}


static void
on_simple_timefmt_button_clicked (GtkCheckButton *button,
                                  KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  gboolean use_simple;
  GtkWidget *show;
  GtkWidget *hide;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  use_simple = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
  
  hide = (use_simple) ? p_data->time_entry : p_data->time_button;
  show = (use_simple) ? p_data->time_button : p_data->time_entry;
  
  gtk_widget_hide (hide);
  gtk_widget_show (show);

  hide = (use_simple) ? p_data->time_hint : p_data->time_hint_simple;
  show = (use_simple) ? p_data->time_hint_simple : p_data->time_hint;

  gtk_widget_hide (hide);
  gtk_widget_show (show);
}


static void
on_sports_menu_activate (GtkMenuItem * menuitem,
                         KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  gchar *label_text;
  GtkWidget *label;
  gchar *sport;
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  label = gtk_bin_get_child (GTK_BIN (menuitem));

  g_return_if_fail (GTK_IS_LABEL (label));
  label_text = g_strdup (gtk_label_get_text (GTK_LABEL (label)));

  if (strcmp (label_text, _("Other"))  == 0)
    gtk_widget_show (p_data->sport_entry);
  else {
    sport = g_strstrip (label_text);
    kp_workout_set_sport (p_data->wo, sport);
    g_free (label_text);

    if (GTK_WIDGET_VISIBLE (p_data->sport_entry))
      gtk_widget_hide (p_data->sport_entry);
  }
}

static void
on_time_togglebutton_clicked (GtkToggleButton *button,
                              KPNewSplitWorkoutDialog *dialog)
{
  gchar *str;
  gtk_toggle_button_set_mode (button, FALSE);
  
  if (gtk_toggle_button_get_active (button) == TRUE)
    str = _("PM");
  else
    str = _("AM");
      
  gtk_button_set_label (GTK_BUTTON (button), str);
}

static void
on_sport_entry_changed (GtkEditable * editable,
                        KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  gchar *sport;
  gchar *text;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  text = gtk_editable_get_chars (editable, 0, -1);

  sport = g_strstrip (g_utf8_strdown (text, -1));
  kp_workout_set_sport (p_data->wo, sport);
  
  g_free (sport);
  g_free (text);
}

/*
 * Month (m) is between 1 and 12.
 */
static void
set_date (KPNewSplitWorkoutDialog *dialog,
          guint d,
          guint m,
          guint y)
{
  KPNewSplitWorkoutDialogPrivateData *p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  (void) gtk_calendar_select_month (GTK_CALENDAR (p_data->calendar), m-1, y);
  (void) gtk_calendar_select_day (GTK_CALENDAR (p_data->calendar), d);
}  

static void
set_comment (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTextBuffer *buffer;
  GtkTextView *textview;
  GtkTextIter start;
  GtkTextIter end;
  GValue comment;
  gchar *text;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  textview = GTK_TEXT_VIEW (p_data->comment);
  buffer = gtk_text_view_get_buffer (textview);

  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
  text = gtk_text_buffer_get_text (buffer, &start, &end, TRUE);

  comment.g_type = 0;  
  
  g_assert (text != NULL);
  
  g_value_init (&comment, G_TYPE_STRING);
  g_value_set_string (&comment, text); 
  kp_workout_set_param (p_data->wo, TRUE, "comment", &comment);
  g_value_unset (&comment);
  g_free (text);
}

static gboolean
set_time (KPNewSplitWorkoutDialog *dialog, KPWorkout * wo)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkToggleButton *time_button;
  const gchar *time;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  time_button = GTK_TOGGLE_BUTTON (p_data->time_fmt_button);
  
  if (gtk_toggle_button_get_active (time_button)) {
    time = gtk_button_get_label (GTK_BUTTON (p_data->time_button));
    if (strcmp (time, _("AM")) == 0)
      kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                                 10, 10, 10, 0);
    else
      kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->wo)->datetime,
                                 16, 16, 16, 0);
  } else {
    time = gtk_entry_get_text (GTK_ENTRY (p_data->time_entry));

    if (!check_time (dialog))
      return FALSE;
   
    kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (wo)->datetime,
                              (guint) g_ascii_strtod (&time[0], NULL),
                              (guint) g_ascii_strtod (&time[3], NULL),
                              /* we don't need seconds or smaller units here. */
                              0, 0);
  }
  return TRUE;
}

static gboolean
check_time (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  const gchar *time;
  guint i;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  
  if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (p_data->time_fmt_button))) {
    /* if simple times are not used */
   
    time = gtk_entry_get_text (GTK_ENTRY (p_data->time_entry));
    
    if (strlen (time) != 5)
      return FALSE;

    for (i=0; i < 5; i++)
      if (i != 2 && !g_ascii_isdigit (time[i]))
        return FALSE;

    if (time[2] != ':')
      return FALSE;
  } 
  return TRUE;
}

static gboolean
export_list_row (GtkTreeModel * model,
                 GtkTreePath * path,
                 GtkTreeIter * iter,
                 KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  const gchar *key;
  const gchar *val;
  GValue *value;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  g_return_val_if_fail (p_data != NULL, TRUE); 
  
  gtk_tree_model_get (model, iter, 0, &key, 1, &val, -1);

  value = kp_g_value_from_string (val);
  kp_workout_set_param (p_data->wo, TRUE, key, value);

  return FALSE;
}

/*
 * notebook's note 'DETAILS'
 * 
 */

static void
set_detail_tree_view (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *column1;
  GtkTreeViewColumn *column2;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeView *treeview;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  treeview = GTK_TREE_VIEW (p_data->details_view);
  
  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));

  renderer = gtk_cell_renderer_text_new ();

  column1 = gtk_tree_view_column_new_with_attributes (_("Field"), renderer,
                                                     "text", 0, NULL);
  column2 = gtk_tree_view_column_new_with_attributes (("Value"), renderer,
                                                     "text", 1, NULL);
  gtk_tree_view_append_column (treeview, column1);
  gtk_tree_view_append_column (treeview, column2);

  selection = gtk_tree_view_get_selection (treeview);

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (detail_list_row_selected), dialog);
}

static void
detail_list_row_selected (GtkTreeSelection * selection,
                          KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  gtk_widget_set_sensitive (p_data->detail_delete_button, TRUE);
}

static void
on_detail_delete_button_clicked (GtkButton * button,
                                 KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->details_view));
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
}

static void
on_detail_add_button_clicked (GtkButton * button,
                              KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;

  gchar *value_str;
  gchar *name;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  value_str = g_strdup (gtk_entry_get_text (GTK_ENTRY (p_data->detail_value)));
  name = kp_gui_get_option_menu_active (
      GTK_OPTION_MENU (p_data->detail_preset_menu));
  g_return_if_fail (name != NULL);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->details_view));
  gtk_list_store_append (GTK_LIST_STORE (model), &iter);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, name, 1, value_str, -1);

  g_free (value_str);
  g_free (name);

  gtk_entry_set_text (GTK_ENTRY (p_data->detail_value), "");
}

/*static void
on_detail_menu_item_activate (GtkMenuItem * menuitem,
                              KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  const gchar *text;
  GtkWidget *label;
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  label = gtk_bin_get_child (GTK_BIN (menuitem));
  text = gtk_label_get_text (GTK_LABEL (label));
  gtk_entry_set_text (GTK_ENTRY (p_data->detail_name), text);
}*/

static void
build_detail_menu (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkWidget *item;
  GtkWidget *menu;
  GSList *list;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  list = kp_settings_get_list ("preset_detail");
  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label (_("Choose preset detail"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
  
  while (list) {
    item = gtk_menu_item_new_with_label (list->data);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    list = list->next;
  }
  kp_settings_list_free (list);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->detail_preset_menu), menu);
  gtk_widget_show_all (menu);
}









/*
 * notebook's note 'SPLITS'
 *
 */

enum
{
  S_COL_LAP,
  S_COL_RECOVERY,
  S_COL_DISTANCE,
  S_COL_TIME,
  S_COL_COMMENT,
  S_COL_N
};

static void
set_splits_tree_view (KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *c[5];
  GtkCellRenderer *r_text;
  GtkListStore *store;
  gint i;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  store = gtk_list_store_new (5, G_TYPE_INT,
                              G_TYPE_STRING,
                              G_TYPE_FLOAT,
                              G_TYPE_STRING,
                              G_TYPE_STRING);

  gtk_tree_view_set_model (GTK_TREE_VIEW (p_data->split_treeview),
                           GTK_TREE_MODEL (store));

  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (p_data->split_treeview), TRUE);

  r_text = gtk_cell_renderer_text_new ();

  c[0] =
    gtk_tree_view_column_new_with_attributes (_("Split"), r_text, "text", 0, NULL);
  c[1] =
    gtk_tree_view_column_new_with_attributes (_("Type"), r_text, "text", 1,
                                              NULL);
  c[2] =
    gtk_tree_view_column_new_with_attributes (_("Distance"), r_text, "text", 2,
                                              NULL);
  c[3] =
    gtk_tree_view_column_new_with_attributes (_("Time"), r_text, "text", 3, NULL);
  c[4] =
    gtk_tree_view_column_new_with_attributes (_("Comment"), r_text, "text", 4,
                                              NULL);

  for (i = 0; i < 5; i++) {
    gtk_tree_view_column_set_clickable (c[i], TRUE);
    gtk_tree_view_column_set_sort_column_id (c[i], i);
    gtk_tree_view_append_column (GTK_TREE_VIEW (p_data->split_treeview), c[i]);
  }

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->split_treeview));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (split_list_row_selected), dialog);
}

/*
static void
recovery_toggled (GtkCellRendererToggle * cell,
                  gchar * path_str,
                  KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  gboolean recovery;
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  model = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->split_treeview));
  
  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter, S_COL_RECOVERY, &recovery, -1);

  recovery ^= 1;

  gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                      S_COL_RECOVERY, recovery, -1);

  gtk_tree_path_free (path);
}*/

static void
split_list_row_selected (GtkTreeSelection * selection,
                         KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  gtk_widget_set_sensitive (p_data->split_delete_button, TRUE);
}


void
on_split_delete_button_clicked (GtkButton * button,
                                KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter iter;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->split_treeview));

  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    (void) gtk_list_store_remove (GTK_LIST_STORE (model), &iter);

    if (gtk_tree_model_iter_n_children (model, NULL) == 0)
      gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
  }
}

void
on_split_add_button_clicked (GtkButton * button,
                             KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  GtkTreeModel *store;
  GtkTreeIter iter;
  const gchar *comment;
  const gchar *time;
  gchar *time_str;
  gchar *type;
  gint32 msec;
  
  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
 
  type = kp_gui_get_option_menu_active (GTK_OPTION_MENU (p_data->split_type_menu));
  g_return_if_fail (type != NULL);
  comment = gtk_entry_get_text (GTK_ENTRY (p_data->split_comment_entry));
  time = gtk_entry_get_text (GTK_ENTRY (p_data->split_duration_entry));
 
  if ((msec = kp_duration_str_to_ms (time)) > 0) {
    kp_gui_report_error (GTK_WINDOW (dialog), NULL,
                      _("Duration entry is invalid!"));
    return;
  }
  time_str = kp_date_mseconds_to_std_string (msec);
  
  store = gtk_tree_view_get_model (GTK_TREE_VIEW (p_data->split_treeview));
  gtk_list_store_append (GTK_LIST_STORE (store), &iter);
  gtk_list_store_set (GTK_LIST_STORE (store), &iter,
   0, gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (p_data->split_spin)),
   1, type,
   2, gtk_spin_button_get_value (GTK_SPIN_BUTTON (p_data->split_distance_spin)),
   3, time_str,
   4, comment,
  -1);
  g_free (time_str);
  gtk_spin_button_spin (GTK_SPIN_BUTTON (p_data->split_spin),
                        GTK_SPIN_STEP_FORWARD, 1.0);

  gtk_entry_set_text (GTK_ENTRY (p_data->split_comment_entry), "");
  gtk_entry_set_text (GTK_ENTRY (p_data->split_duration_entry), "");
}

static gboolean
export_split_list_row (GtkTreeModel * model,
                       GtkTreePath * path,
                       GtkTreeIter * iter,
                       KPNewSplitWorkoutDialog *dialog)
{
  KPNewSplitWorkoutDialogPrivateData *p_data;
  KPWorkoutSplitType split_type;
  KPCalendarTime *ctime;
  guint lap;
  gfloat distance;
  gchar *split_type_str;
  gchar *time;
  gchar *comment;
  gchar *tmp;

  p_data = KP_NEW_SPLIT_WORKOUT_DIALOG_PRIVATE_DATA (dialog);
  
  gtk_tree_model_get (model, iter,
                      S_COL_LAP, &lap,
                      S_COL_RECOVERY, &split_type_str,
                      S_COL_DISTANCE, &distance,
                      S_COL_TIME, &time, S_COL_COMMENT, &comment, -1);

  ctime = kp_calendar_time_new ();

  if (strcmp (split_type_str, _("Normal")) == 0)
    split_type = KP_WORKOUT_SPLIT_TYPE_NORMAL;
  else if (strcmp (split_type_str, _("Warm Up")) == 0)
    split_type = KP_WORKOUT_SPLIT_TYPE_WARM_UP;
  else if (strcmp (split_type_str, _("Cool Down")) == 0)
    split_type = KP_WORKOUT_SPLIT_TYPE_COOL_DOWN;
  else if (strcmp (split_type_str, _("Recovery")) == 0)
    split_type = KP_WORKOUT_SPLIT_TYPE_RECOVERY;
  else
    g_return_val_if_reached (FALSE);

  /* Convert "1:20:00" -> "01:20:00" */
  if (time[1] == ':')
    tmp = g_strdup_printf("0%s", time);
  else
    tmp = g_strdup (time);
  
  if (kp_calendar_time_set_hmst_str (ctime, tmp) == FALSE) {
    g_warning ("Setting the time failed! (time = \"%s\"\n", tmp);
  }
 
  g_free (tmp);
  
  kp_workout_add_split (p_data->wo, split_type, lap, distance, ctime, comment);

  return FALSE;
}



