/*
 * widgets.cc -- Resusable widgets
 * Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "widgets.h"

extern "C" {
#include "interface.h"
#include "support.h"
}

// Required since libglade is now used in kino
GtkWidget *my_lookup_widget( GtkWidget *widget, const gchar *widget_name )
{
	GtkWidget *parent, *found_widget;

	for (;;)
    {
		if (GTK_IS_MENU (widget))
        	parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
		else
        	parent = widget->parent;
		if (!parent)
        	parent = (GtkWidget*) g_object_get_data (G_OBJECT (widget), "GladeParentKey");
		if (parent == NULL)
        	break;
		widget = parent;
    }

	found_widget = (GtkWidget*) g_object_get_data (G_OBJECT (widget), widget_name);
	if (!found_widget)
		g_warning ("Widget not found: %s", widget_name);
	return found_widget;
}


/** Key frame controller is a GUI widget which provides a simple key frame definition/editing tool. 
 
	This is currently represented by 4 components - a slide bar to select the position, a toggle 
	button which allows you to specify whether a key exists at the current position, and two buttons
	to allow navigation between the key frames. 

	Intended use is for a container to implement the controller client.
*/

static void on_control_key_frame( GtkToggleButton *button, gpointer user_data );
static void on_control_prev_key_clicked( GtkToggleButton *arrow, gpointer user_data );
static void on_control_next_key_clicked( GtkToggleButton *arrow, gpointer user_data );
static void on_control_position_changed( GtkHScale *position, gpointer user_data );

class KinoPlusKeyFrameController : public KeyFrameController
{
	private:
		bool gui_locked;
		KeyFrameControllerClient *client;
		GtkWidget *window;

		void SetCurrentPosition( double position )
		{
			GtkRange *range = GTK_RANGE( my_lookup_widget( window, "hscale" ) );
			GtkAdjustment *adjust = gtk_range_get_adjustment( range );
        	adjust->value = position;
        	gtk_range_set_adjustment( GTK_RANGE( range ), adjust);
			gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );
		}

	public:
		KinoPlusKeyFrameController( KeyFrameControllerClient *in_client ) : gui_locked( false ), client( in_client )
		{
			window = create_kino_key_frame_control( );
			GtkWidget *widget = my_lookup_widget( window, "togglebutton" );
			gtk_signal_connect( GTK_OBJECT( widget ), "toggled", GTK_SIGNAL_FUNC( on_control_key_frame ), this ); 
			widget = my_lookup_widget( window, "button_prev" );
			gtk_signal_connect( GTK_OBJECT( widget ), "toggled", GTK_SIGNAL_FUNC( on_control_prev_key_clicked ), this ); 
			widget = my_lookup_widget( window, "button_next" );
			gtk_signal_connect( GTK_OBJECT( widget ), "toggled", GTK_SIGNAL_FUNC( on_control_next_key_clicked ), this ); 
			widget = my_lookup_widget( window, "hscale" );
			gtk_signal_connect( GTK_OBJECT( GTK_RANGE( widget )->adjustment ), "value_changed", GTK_SIGNAL_FUNC( on_control_position_changed ), this );
		}

		virtual ~KinoPlusKeyFrameController()
		{
			gtk_widget_destroy( window );
		}

		/** GtkKinoWidget implementation
		*/

		GtkWidget *GetGtkWidget( )
		{
			return window;
		}

		/** Reflect the status of the current position.
		*/

		void ShowCurrentStatus( double position, frame_type type, bool hasPrev, bool hasNext )
		{
			// Make sure that updating the GUI doesn't trigger callbacks
			gui_locked = true;

			// Set the position
			SetCurrentPosition( position );

			// Determine Key button status
			GtkWidget *widget = my_lookup_widget( window, "togglebutton" );

			if ( type & LOCKED_KEY )	 // Locked Key
			{
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), true );
				gtk_widget_set_sensitive( widget, false );
			}
			else if ( type & KEY ) 		// Normal Key
			{
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), true );
				gtk_widget_set_sensitive( widget, true );
			}
			else 						// Not Key
			{
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), false );
				gtk_widget_set_sensitive( widget, true );
			}

			// Set the previous button availability/status
			widget = my_lookup_widget( window, "button_prev" );
			gtk_widget_set_sensitive( widget, hasPrev );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), false );

			// Set the next button availability/status
			widget = my_lookup_widget( window, "button_next" );
			gtk_widget_set_sensitive( widget, hasNext );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), false );

			// All done - gui is active again
			gui_locked = false;
		}

		/** Get the current position.
		*/

		double GetCurrentPosition( )
		{
			GtkRange *range = GTK_RANGE( my_lookup_widget( window, "hscale" ) );
			GtkAdjustment *adjust = gtk_range_get_adjustment( range );
			return adjust->value; 
		}

		/** Triggers the client callbacks.
		*/

		void OnControllerKeyChanged( )
		{
			if ( ! gui_locked )
			{
				GtkWidget *widget = my_lookup_widget( window, "togglebutton" );
				client->OnControllerKeyChanged( GetCurrentPosition( ), gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) );
			}
		}

		void OnControllerPrevKey( )
		{
			if ( ! gui_locked )
				client->OnControllerPrevKey( GetCurrentPosition( ) );
		}

		void OnControllerNextKey( )
		{
			if ( ! gui_locked )
				client->OnControllerNextKey( GetCurrentPosition( ) );
		}

		void OnControllerPositionChanged( )
		{
			if ( ! gui_locked )
				client->OnControllerPositionChanged( GetCurrentPosition( ) );
		}
};

static void on_control_key_frame( GtkToggleButton *button, gpointer user_data )
{
	( (KinoPlusKeyFrameController *)user_data )->OnControllerKeyChanged( );
}

static void on_control_prev_key_clicked( GtkToggleButton *arrow, gpointer user_data )
{
	( (KinoPlusKeyFrameController *)user_data )->OnControllerPrevKey( );
}

static void on_control_next_key_clicked( GtkToggleButton *arrow, gpointer user_data )
{
	( (KinoPlusKeyFrameController *)user_data )->OnControllerNextKey( );
}

static void on_control_position_changed( GtkHScale *position, gpointer user_data )
{
	( (KinoPlusKeyFrameController *)user_data )->OnControllerPositionChanged( );
}

/** Factory method for the key frame controller.
*/

KeyFrameController *CreateWidgetKeyFrameController( KeyFrameControllerClient *client )
{
	return new KinoPlusKeyFrameController( client );
}

/** KinoPlusPreview Area Widget. 
*/

static void on_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data );

class KinoPlusPreviewArea : public PreviewArea
{
	private:
		PreviewAreaClient *client;
		GtkWidget *window;
		GtkWidget *drawing_area;

	public:
		KinoPlusPreviewArea( PreviewAreaClient *m_client ) : client( m_client )
		{
			window = create_kino_preview_area( );
			drawing_area = my_lookup_widget( window, "drawingarea" );
			gtk_signal_connect( GTK_OBJECT( drawing_area ), "expose_event", GTK_SIGNAL_FUNC( on_drawingarea_refresh_required ), this );
		}

		virtual ~KinoPlusPreviewArea()
		{
			gtk_widget_destroy( window );
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		void ShowImage( int x, int y, uint8_t *image, int width, int height )
		{
			GtkWidget *widget = my_lookup_widget( window, "hruler" );
			SetRuler( GTK_RULER( widget ), x );
			widget = my_lookup_widget( window, "vruler" );
			SetRuler( GTK_RULER( widget ), y );

			GdkGC *gc = gdk_gc_new( drawing_area->window );
			GdkPixbuf *pix = gdk_pixbuf_new_from_data( image, GDK_COLORSPACE_RGB, FALSE, 8, width, height, width * 3, NULL, NULL );
			GdkPixbuf *im = gdk_pixbuf_scale_simple( pix, drawing_area->allocation.width, drawing_area->allocation.height, GDK_INTERP_NEAREST );
			gdk_draw_pixbuf( drawing_area->window, gc, im, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_NORMAL, 0, 0 );
			g_object_unref( im );
			g_object_unref( pix );
			g_object_unref( gc );
		}

		void Refresh( )
		{
			client->OnPreviewAreaRefresh( this );
		}

	protected:
		void SetRuler( GtkRuler *ruler, double value )
		{
			gtk_ruler_set_range( ruler, 0, 100, value, 100 );
		}
};

static void on_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data )
{
	( (KinoPlusPreviewArea *)user_data )->Refresh( );
}

/** Factory method for the PreviewArea.
*/

PreviewArea *CreateWidgetPreviewArea( PreviewAreaClient *client )
{
	return new KinoPlusPreviewArea( client );
}

/** Pair Picker widget - provides a widget for collecting x,y type info.
*/

static void on_pair_picker_key_frame_changed( GtkWidget *some_widget, gpointer user_data );
static gboolean on_pair_picker_arrow_north( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_pair_picker_arrow_east( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_pair_picker_arrow_south( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_pair_picker_arrow_west( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_pair_picker_arrow_released( GtkWidget *widget, GdkEventButton *event, gpointer user_data );

enum direction_button_type { NORTH, EAST, SOUTH, WEST };

class KinoPairPicker : public PairPicker
{
	private:
		GtkWidget *window;
		PairPickerClient *client;
		bool gui_locked;
		double first;
		double second;
		bool held;
		double first_min;
		double first_max;
		double second_min;
		double second_max;

	public:
		KinoPairPicker( PairPickerClient *in_client ) : client( in_client ), gui_locked( false ), first( 50 ), second( 50 ), held( false ),
														first_min( 0 ), first_max( 100 ), second_min( 0 ), second_max( 100 )
		{
			window = create_kino_pair_picker( );
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_first" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_pair_picker_key_frame_changed ), this );
			widget = my_lookup_widget( window, "spinbutton_second" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_pair_picker_key_frame_changed ), this );
			widget = my_lookup_widget( window, "eventbox_north" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_north ), this );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_released ), this );
			widget = my_lookup_widget( window, "eventbox_east" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_east ), this );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_released ), this );
			widget = my_lookup_widget( window, "eventbox_south" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_south ), this );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_released ), this );
			widget = my_lookup_widget( window, "eventbox_west" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_west ), this );
			gtk_signal_connect( GTK_OBJECT( widget ), "event", GTK_SIGNAL_FUNC( on_pair_picker_arrow_released ), this );
		}

		virtual ~KinoPairPicker()
		{
			gtk_widget_destroy( window );
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		void SetValues( double first, double second )
		{
			gui_locked = true;
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_first" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), first );
			widget = my_lookup_widget( window, "spinbutton_second" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), second );
			gui_locked = false;
		}

		void SetSpinButtonMinMax( GtkSpinButton *widget, double min, double max )
		{
			GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( widget ) );
			adjust->lower = min < max ? min : max;
			adjust->upper = max > min ? max : min;
			gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );
		}

		void SetFirstMinMax( double min, double max )
		{
			gui_locked = true;
			first_min = min;
			first_max = max;
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_first" );
			SetSpinButtonMinMax( GTK_SPIN_BUTTON( widget ), min, max );
			gui_locked = false;
		}

		void SetSecondMinMax( double min, double max )
		{
			gui_locked = true;
			second_min = min;
			second_max = max;
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_second" );
			SetSpinButtonMinMax( GTK_SPIN_BUTTON( widget ), min, max );
			gui_locked = false;
		}

		void OnChange( )
		{
			GtkWidget *widget = my_lookup_widget( window, "spinbutton_first" );
			first = atoi( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			widget = my_lookup_widget( window, "spinbutton_second" );
			second = atoi( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

			if ( !gui_locked )
				client->OnPairPickerChangeValue( this, first, second );
		}

		void OnDirectionButton( direction_button_type direction )
		{
			bool first_iteration = true;

			GtkSpinButton *first_spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton_first" ) );
			GtkSpinButton *second_spin = GTK_SPIN_BUTTON( my_lookup_widget( window, "spinbutton_second" ) );

			int forward_first = first_min < first_max ? 1 : -1;
			int forward_second = second_min < second_max ? 1 : -1;

			held = true;

			while( held )
			{
				switch( direction )
				{
					case WEST:
						first -= forward_first;
						gtk_spin_button_set_value( GTK_SPIN_BUTTON( first_spin ), first );
						break;
					case NORTH:
						second -= forward_second;
						gtk_spin_button_set_value( GTK_SPIN_BUTTON( second_spin ), second );
						break;
					case EAST:
						first += forward_first;
						gtk_spin_button_set_value( GTK_SPIN_BUTTON( first_spin ), first );
						break;
					case SOUTH:
						second += forward_second;
						gtk_spin_button_set_value( GTK_SPIN_BUTTON( second_spin ), second );
						break;
				}

				while (gtk_events_pending()) 
        			gtk_main_iteration();

				if ( held && first_iteration )
				{
					struct timespec tm = { 0, 50000000 };
					nanosleep( &tm, NULL );
					first_iteration = false;
				}
				else if ( held )
				{
					struct timespec tm = { 0, 2500000 };
					nanosleep( &tm, NULL );
					first_iteration = false;
				}

				while (gtk_events_pending()) 
        			gtk_main_iteration();

			}
		}

		void OnDirectionButtonReleased( )
		{
			held = false;
		}
};

static void on_pair_picker_key_frame_changed( GtkWidget *some_widget, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnChange( );
}

static gboolean on_pair_picker_arrow_north( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnDirectionButton( NORTH );
	return FALSE;
}

static gboolean on_pair_picker_arrow_east( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnDirectionButton( EAST );
	return FALSE;
}

static gboolean on_pair_picker_arrow_south( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnDirectionButton( SOUTH );
	return FALSE;
}

static gboolean on_pair_picker_arrow_west( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnDirectionButton( WEST );
	return FALSE;
}

static gboolean on_pair_picker_arrow_released( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPairPicker *)user_data )->OnDirectionButtonReleased( );
	return FALSE;
}

/** Factory method for the PairPicker.
*/

PairPicker *CreateWidgetPairPicker( PairPickerClient *client )
{
	return new KinoPairPicker( client );
}

static gboolean on_file_selector_delete_event( GtkWidget *widget, GdkEvent *event, gpointer user_data );
static gboolean on_file_selector_ok( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_file_selector_cancel( GtkWidget *widget, GdkEventButton *event, gpointer user_data );

class KinoPlusFileSelector : public FileSelector
{
	private:
		GtkWidget *window;
		FileSelectorClient *client;
		bool deleted;

	public:
		KinoPlusFileSelector( FileSelectorClient *in_client ) : client( in_client ), deleted( true )
		{
		}

		virtual ~KinoPlusFileSelector( )
		{
		}

		void SetDefaultDirectory( char *dir )
		{
		}

		void SelectFile( ) 
		{
			window = create_fileselection( );
			deleted = false;
			gtk_signal_connect( GTK_OBJECT( window ), "delete_event", GTK_SIGNAL_FUNC( on_file_selector_delete_event ), this );

			GtkWidget *widget = my_lookup_widget( window, "ok_button" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_file_selector_ok ), this );
			widget = my_lookup_widget( window, "cancel_button" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_press_event", GTK_SIGNAL_FUNC( on_file_selector_cancel ), this );

			gtk_widget_show( window );
			gtk_main();

			if ( !deleted )
				gtk_widget_destroy( window );
		}

		void OnConfirm( )
		{
			const gchar *filename = gtk_file_selection_get_filename( GTK_FILE_SELECTION( window ) );
			client->OnFileSelected( this, (char *)filename );
			gtk_main_quit( );
		}

		void OnCancel( )
		{
			gtk_main_quit( );
		}

		void OnDestroy( )
		{
			deleted = true;
			gtk_main_quit( );
		}
};

static gboolean on_file_selector_ok( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPlusFileSelector *)user_data )->OnConfirm( );
	return FALSE;
}

static gboolean on_file_selector_cancel( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPlusFileSelector *)user_data )->OnCancel( );
	return FALSE;
}

static gboolean on_file_selector_delete_event( GtkWidget *widget, GdkEvent *event, gpointer user_data )
{
	( ( KinoPlusFileSelector *)user_data )->OnDestroy( );
	return FALSE;
}

FileSelector *CreateWidgetFileSelector( FileSelectorClient *client )
{
	return new KinoPlusFileSelector( client );
}

static void on_luma_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data );
static gboolean on_luma_picker_choose( GtkWidget *widget, GdkEventButton *event, gpointer user_data );
static gboolean on_luma_softness_change( GtkWidget *widget, gpointer user_data );

class KinoPlusLumaPicker : public LumaPicker, private FileSelectorClient
{
	private:
		GtkWidget *window;
		LumaPickerClient *client;
		FileSelector *selector;
		bool gui_active;
		char *file;
		uint8_t *last_image;

	public:
		KinoPlusLumaPicker( LumaPickerClient *in_client ) : client( in_client ), gui_active( true ), file( NULL )
		{
			window = create_kino_luma_picker( );
			selector = CreateWidgetFileSelector( this );

			GtkWidget *widget = my_lookup_widget( window, "button" );
			gtk_signal_connect( GTK_OBJECT( widget ), "button_release_event", GTK_SIGNAL_FUNC( on_luma_picker_choose ), this );

			widget = my_lookup_widget( window, "spinbutton" );
			gtk_signal_connect( GTK_OBJECT( widget ), "changed", GTK_SIGNAL_FUNC( on_luma_softness_change ), this );

			widget = my_lookup_widget( window, "drawingarea" );
			gtk_signal_connect( GTK_OBJECT( widget ), "expose_event", GTK_SIGNAL_FUNC( on_luma_drawingarea_refresh_required ), this );

			last_image = new uint8_t[ 90 * 72 * 3 ];
			memset( last_image, 0, 90 * 72 * 3 );

			SetLuma( NULL, 0.20 );
		}

		virtual ~KinoPlusLumaPicker()
		{
			gtk_widget_destroy( window );
			delete selector;
			delete last_image;
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		bool SetLuma( char *file, double softness )
		{
			gui_active = false;
			GtkWidget *widget = my_lookup_widget( window, "spinbutton" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), softness );
			gui_active = true;

#if 0
			if ( file != NULL )
			{
				GdkImlibImage *image = gdk_imlib_load_image( file );
				if ( image != NULL )
				{
					this->file = file;
					GdkImlibImage *scaled= gdk_imlib_clone_scaled_image( image, 90, 72 );
					memcpy( last_image, scaled->rgb_data, 90 * 72 * 3 );
					GtkWidget *widget = my_lookup_widget( window, "drawingarea" );
					gdk_imlib_apply_image( scaled, widget->window );
					gdk_imlib_destroy_image( scaled );
					gdk_imlib_destroy_image( image );
					return true;
				}
				else
				{
					return false;
				}
			}
			else
			{
				this->file = file;
				GtkWidget *widget = my_lookup_widget( window, "drawingarea" );
				GdkImlibImage *image = gdk_imlib_create_image_from_data( last_image, NULL, 90, 72 );
				gdk_imlib_apply_image( image, widget->window );
				gdk_imlib_destroy_image( image );
				return true;
			}
#endif
		}

		void OnChoose( )
		{
			selector->SelectFile( );
		}

		void OnFileSelected( FileSelector *selector, char *file )
		{
			GtkWidget *widget = my_lookup_widget( window, "spinbutton" );
			double softness = atof( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			if ( SetLuma( file, softness ) )
				client->OnLumaPickerChange( this, file, softness );
		}

		void OnSoftnessChange( )
		{
			if ( gui_active )
				OnFileSelected( selector, file );
		}

		void Refresh( )
		{
#if 0 
			GtkWidget *widget = my_lookup_widget( window, "drawingarea" );
			GdkImlibImage *image = gdk_imlib_create_image_from_data( last_image, NULL, 90, 72 );
			gdk_imlib_apply_image( image, widget->window );
			gdk_imlib_destroy_image( image );
#endif
		}
};

static void on_luma_drawingarea_refresh_required( GtkWidget *some_widget, GdkEventConfigure *event, gpointer user_data )
{
	( ( KinoPlusLumaPicker * )user_data )->Refresh( );
}

static gboolean on_luma_picker_choose( GtkWidget *widget, GdkEventButton *event, gpointer user_data )
{
	( ( KinoPlusLumaPicker * )user_data )->OnChoose( );
	return FALSE;
}

static gboolean on_luma_softness_change( GtkWidget *widget, gpointer user_data )
{
	( ( KinoPlusLumaPicker * )user_data )->OnSoftnessChange( );
	return FALSE;
}

/** Factory method for the LumaPicker.
*/

LumaPicker *CreateWidgetLumaPicker( LumaPickerClient *client )
{
	return new KinoPlusLumaPicker( client );
}

class KinoInterlaceOptions : public InterlaceOptions
{
	private:
		GtkWidget *window;

	public:
		KinoInterlaceOptions( )
		{
			window = create_kino_interlace( );
		}

		virtual ~KinoInterlaceOptions()
		{
			gtk_widget_destroy( window );
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		bool IsInterlaceOn( )
		{
			return gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( my_lookup_widget( window, "checkbutton_interlace" ) ) );
		}

		bool IsFirstFieldOn( )
		{
			return gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( my_lookup_widget( window, "checkbutton_first_field" ) ) );
		}
};

InterlaceOptions *CreateWidgetInterlaceOptions( )
{
	return new KinoInterlaceOptions( );
}


class KinoMultiMap : public MultiMap
{
	private:
		MultiMapClient *client;
		GtkWidget *window;

	public:
		KinoMultiMap( MultiMapClient *in_client ) : client( in_client )
		{
			window = create_kino_multimap( );
		}

		virtual ~KinoMultiMap()
		{
			gtk_widget_destroy( window );
		}

		GtkWidget *GetGtkWidget()
		{
			return window;
		}

		void Refresh( MultiMapProperties *set )
		{
			int count = set->Count();
			GtkWidget *widget = my_lookup_widget( window, "vbox_properties" );

			for ( int index = 0; index < count; index ++ )
			{
			}
		}
};

extern MultiMap *CreateWidgetMultiMap( MultiMapClient *client )
{
	return new KinoMultiMap( client );
}



