/*
* image_create.cc -- RGB24 image create
* Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glade/glade.h>
#include "image_create.h"
#include "page_magick.h"
#include "message.h"

extern "C"
{
#include "support.h"
	extern GladeXML* magick_glade;
}

/*
 * Note that for speed of development I elected to make everything GUI here, but
 * this will limit its use a bit. For every creator, there should be a GUI component
 * which extends a non-GUI component. The intention is that we should be able
 * to propogate state via the playlist and some unspecified sml/smil aware class
 * (ummm... I think that's the way I want it anyway... :-/).
 */

/** Create a number of images from the specified colour.
*/

class ImageCreateColour : public GDKAudioImport
{
protected:
	static uint8_t start_r, start_g, start_b, start_a;
	static int frames;
	GtkWidget *picker;
	GtkWidget *frame_counter;

public:
	char *GetDescription( ) const
	{
		return _( "Fixed Colour" );
	}

	void CreateFrame( uint8_t *pixels, int width, int height, double position, double frame_delta )
	{
		if ( position == 0 )
		{
			uint8_t * p = pixels;
			while ( p < ( pixels + width * height * 3 ) )
			{
				*p ++ = start_r;
				*p ++ = start_g;
				*p ++ = start_b;
			}
		}
	}

	int GetNumberOfFrames( )
	{
		return frames;
	}

	void AddFrameCollector( GtkWidget *table, int row )
	{
		GtkWidget * label = gtk_label_new( _( "Frames " ) );
		gtk_widget_show( label );

		GtkObject *frame_counter_adj = gtk_adjustment_new( frames, 0, 1000, 1, 10, 10 );
		frame_counter = gtk_spin_button_new( GTK_ADJUSTMENT( frame_counter_adj ), 1, 0 );
		gtk_widget_show( frame_counter );
		gtk_widget_set_usize( frame_counter, 60, -2 );

		gtk_table_attach( GTK_TABLE( table ), label, 0, 1, row, row + 1, ( GtkAttachOptions ) ( 0 ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), frame_counter, 1, 2, row, row + 1, ( GtkAttachOptions ) ( GTK_EXPAND | GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
	}

	void InterpretFrameCollector( )
	{
		GtkEntry * entry = GTK_ENTRY( frame_counter );
		frames = atoi( gtk_entry_get_text( entry ) );
	}

	void AttachWidgets( GtkBin *bin )
	{
		GtkWidget * table = gtk_table_new ( 2, 2, FALSE );

		GtkWidget *label = gtk_label_new( _( "Colour " ) );
		gtk_widget_show( label );

		picker = gnome_color_picker_new();
		gnome_color_picker_set_i8( GNOME_COLOR_PICKER( picker ), start_r, start_g, start_b, start_a );
		gtk_widget_show( picker );

		AddFrameCollector( table, 0 );

		gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 1, 2, ( GtkAttachOptions ) ( 0 ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), picker, 1, 2, 1, 2, ( GtkAttachOptions ) ( GTK_EXPAND | GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );

		gtk_widget_show( table );
		gtk_container_add( GTK_CONTAINER( bin ), table );
	}

	void DetachWidgets( GtkBin *bin )
	{
		if ( bin->child != NULL )
		{
			InterpretFrameCollector( );
			gnome_color_picker_get_i8( GNOME_COLOR_PICKER( picker ), &start_r, &start_g, &start_b, &start_a );
			gtk_container_remove( GTK_CONTAINER( bin ), bin->child );
		}
	}

	void InterpretWidgets( GtkBin *bin )
	{
		InterpretFrameCollector( );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( picker ), &start_r, &start_g, &start_b, &start_a );
	}

	void CreateAudio( int16_t **buffer, short int *channels, int *frequency, int *samples )
	{}
}
;

uint8_t ImageCreateColour::start_r = 0;
uint8_t ImageCreateColour::start_g = 0;
uint8_t ImageCreateColour::start_b = 0;
uint8_t ImageCreateColour::start_a = 0;
int ImageCreateColour::frames = 25;

/** Create noise.
*/

class ImageCreateNoise : public ImageCreateColour
{
public:
	char *GetDescription( ) const
	{
		return _( "Random noise" );
	}

	void CreateFrame( uint8_t *pixels, int width, int height, double position, double frame_delta )
	{
		uint8_t value;
		uint8_t *p = pixels;
		while ( p < ( pixels + width * height * 3 ) )
		{
			value = ( uint8_t ) ( 255.0 * rand() / ( RAND_MAX + 1.0 ) );
			*p ++ = value;
			*p ++ = value;
			*p ++ = value;
		}
	}

	void AttachWidgets( GtkBin *bin )
	{
		GtkWidget * table = gtk_table_new ( 2, 2, FALSE );
		AddFrameCollector( table, 0 );
		gtk_widget_show( table );
		gtk_container_add( GTK_CONTAINER( bin ), table );
	}

	void DetachWidgets( GtkBin *bin )
	{
		if ( bin->child != NULL )
		{
			InterpretFrameCollector( );
			gtk_container_remove( GTK_CONTAINER( bin ), bin->child );
		}
	}

	void InterpretWidgets( GtkBin *bin )
	{
		InterpretFrameCollector( );
	}
};

/** Create a number of images that move from one colour to the next.
*/

class ImageCreateColourRange : public ImageCreateColour
{
protected:
	static uint8_t end_r, end_g, end_b, end_a;
	GtkWidget *start_picker;
	GtkWidget *end_picker;

public:
	char *GetDescription( ) const
	{
		return _( "Colour Range" );
	}

	void CreateFrame( uint8_t *pixels, int width, int height, double position, double frame_delta )
	{
		uint8_t r = start_r + ( uint8_t ) ( ( end_r - start_r ) * position );
		uint8_t g = start_g + ( uint8_t ) ( ( end_g - start_g ) * position );
		uint8_t b = start_b + ( uint8_t ) ( ( end_b - start_b ) * position );
		uint8_t *p = pixels;
		while ( p < ( pixels + width * height * 3 ) )
		{
			*p ++ = r;
			*p ++ = g;
			*p ++ = b;
		}
	}

	void AttachWidgets( GtkBin *bin )
	{
		GtkWidget * table = gtk_table_new( 2, 4, FALSE );

		GtkWidget *start_label = gtk_label_new( _( "From " ) );
		gtk_widget_show( start_label );

		start_picker = gnome_color_picker_new();
		gnome_color_picker_set_i8( GNOME_COLOR_PICKER( start_picker ), start_r, start_g, start_b, start_a );
		gtk_widget_show( start_picker );

		GtkWidget *end_label = gtk_label_new( _( " to " ) );
		gtk_widget_show( end_label );

		end_picker = gnome_color_picker_new();
		gnome_color_picker_set_i8( GNOME_COLOR_PICKER( end_picker ), end_r, end_g, end_b, end_a );
		gtk_widget_show( end_picker );

		AddFrameCollector( table, 0 );

		gtk_table_attach( GTK_TABLE( table ), start_label, 0, 1, 1, 2, ( GtkAttachOptions ) ( GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), start_picker, 1, 2, 1, 2, ( GtkAttachOptions ) ( GTK_EXPAND | GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), end_label, 2, 3, 1, 2, ( GtkAttachOptions ) ( GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), end_picker, 3, 4, 1, 2, ( GtkAttachOptions ) ( GTK_EXPAND | GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );

		gtk_widget_show( table );
		gtk_container_add( GTK_CONTAINER( bin ), table );
	}

	void DetachWidgets( GtkBin *bin )
	{
		if ( bin->child != NULL )
		{
			InterpretFrameCollector( );
			gnome_color_picker_get_i8( GNOME_COLOR_PICKER( start_picker ), &start_r, &start_g, &start_b, &start_a );
			gnome_color_picker_get_i8( GNOME_COLOR_PICKER( end_picker ), &end_r, &end_g, &end_b, &end_a );
			gtk_container_remove( GTK_CONTAINER( bin ), bin->child );
		}
	}

	void InterpretWidgets( GtkBin *bin )
	{
		InterpretFrameCollector( );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( start_picker ), &start_r, &start_g, &start_b, &start_a );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( end_picker ), &end_r, &end_g, &end_b, &end_a );
	}
};

uint8_t ImageCreateColourRange::end_r = 255;
uint8_t ImageCreateColourRange::end_g = 255;
uint8_t ImageCreateColourRange::end_b = 255;
uint8_t ImageCreateColourRange::end_a = 0;

/** Create a number of images that from gradiated colours.
*/

class ImageCreateGradiate : public GDKImageCreate
{
private:
	GtkWidget *window;
	uint8_t start_left_r, start_left_g, start_left_b, start_left_a;
	uint8_t start_right_r, start_right_g, start_right_b, start_right_a;
	uint8_t end_left_r, end_left_g, end_left_b, end_left_a;
	uint8_t end_right_r, end_right_g, end_right_b, end_right_a;
	uint8_t start_r, start_g, start_b, start_a;
	uint8_t end_r, end_g, end_b, end_a;
	int type;
	int frames;

public:
	ImageCreateGradiate()
	{
		window = glade_xml_get_widget( magick_glade, "image_create_gradiate" );
		gnome_color_picker_set_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_start_right" ) ), 255, 255, 255, 0 );
		gnome_color_picker_set_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_end_right" ) ), 255, 255, 255, 0 );
	}

	virtual ~ImageCreateGradiate()
	{
		gtk_widget_destroy( window );
	}

	char *GetDescription( ) const
	{
		return _( "Gradiant" );
	}

	void LeftMiddle( uint8_t *pixels, int width, int height, double position )
	{
		double rr = ( double ) ( end_r - start_r ) / ( double ) ( width ) * 2;
		double rg = ( double ) ( end_g - start_g ) / ( double ) ( width ) * 2;
		double rb = ( double ) ( end_b - start_b ) / ( double ) ( width ) * 2;
		uint8_t *p = pixels;
		for ( int y = 0; y < height; y ++ )
		{
			for ( int x = 0; x < width; x ++ )
			{
				if ( x < width / 2 )
				{
					*p ++ = ( guint8 ) ( start_r + rr * x );
					*p ++ = ( guint8 ) ( start_g + rg * x );
					*p ++ = ( guint8 ) ( start_b + rb * x );
				}
				else
				{
					int o = width / 2 - ( x - width / 2 );
					*p ++ = ( guint8 ) ( start_r + rr * o );
					*p ++ = ( guint8 ) ( start_g + rg * o );
					*p ++ = ( guint8 ) ( start_b + rb * o );
				}
			}
		}
	}

	void LeftRight( uint8_t *pixels, int width, int height, double position )
	{
		double rr = ( double ) ( end_r - start_r ) / ( double ) ( width );
		double rg = ( double ) ( end_g - start_g ) / ( double ) ( width );
		double rb = ( double ) ( end_b - start_b ) / ( double ) ( width );
		uint8_t *p = pixels;
		for ( int y = 0; y < height; y ++ )
		{
			for ( int x = 0; x < width; x ++ )
			{
				*p ++ = ( guint8 ) ( start_r + rr * x );
				*p ++ = ( guint8 ) ( start_g + rg * x );
				*p ++ = ( guint8 ) ( start_b + rb * x );
			}
		}
	}

	void TopBottom( uint8_t *pixels, int width, int height, double position )
	{
		double rr = ( double ) ( end_r - start_r ) / ( double ) ( height );
		double rg = ( double ) ( end_g - start_g ) / ( double ) ( height );
		double rb = ( double ) ( end_b - start_b ) / ( double ) ( height );
		uint8_t *p = pixels;
		for ( int y = 0; y < height; y ++ )
		{
			for ( int x = 0; x < width; x ++ )
			{
				*p ++ = ( guint8 ) ( start_r + rr * y );
				*p ++ = ( guint8 ) ( start_g + rg * y );
				*p ++ = ( guint8 ) ( start_b + rb * y );
			}
		}
	}

	void TopMiddle( uint8_t *pixels, int width, int height, double position )
	{
		double rr = ( double ) ( end_r - start_r ) / ( double ) ( height ) * 2;
		double rg = ( double ) ( end_g - start_g ) / ( double ) ( height ) * 2;
		double rb = ( double ) ( end_b - start_b ) / ( double ) ( height ) * 2;
		uint8_t *p = pixels;
		for ( int y = 0; y < height; y ++ )
		{
			for ( int x = 0; x < width; x ++ )
			{
				if ( y < height / 2 )
				{
					*p ++ = ( guint8 ) ( start_r + rr * y );
					*p ++ = ( guint8 ) ( start_g + rg * y );
					*p ++ = ( guint8 ) ( start_b + rb * y );
				}
				else
				{
					int o = height / 2 - ( y - height / 2 );
					*p ++ = ( guint8 ) ( start_r + rr * o );
					*p ++ = ( guint8 ) ( start_g + rg * o );
					*p ++ = ( guint8 ) ( start_b + rb * o );
				}
			}
		}
	}

	void CreateFrame( uint8_t *pixels, int width, int height, double position, double frame_delta )
	{
		start_r = start_left_r + ( uint8_t ) ( ( end_left_r - start_left_r ) * position );
		start_g = start_left_g + ( uint8_t ) ( ( end_left_g - start_left_g ) * position );
		start_b = start_left_b + ( uint8_t ) ( ( end_left_b - start_left_b ) * position );

		end_r = start_right_r + ( uint8_t ) ( ( end_right_r - start_right_r ) * position );
		end_g = start_right_g + ( uint8_t ) ( ( end_right_g - start_right_g ) * position );
		end_b = start_right_b + ( uint8_t ) ( ( end_right_b - start_right_b ) * position );

		if ( type == 0 )
			LeftMiddle( pixels, width, height, position );
		else if ( type == 1 )
			LeftRight( pixels, width, height, position );
		else if ( type == 2 )
			TopBottom( pixels, width, height, position );
		else if ( type == 3 )
			TopMiddle( pixels, width, height, position );
	}

	void AttachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( window ) ) ->child, GTK_WIDGET( bin ) );
	}

	void DetachWidgets( GtkBin *bin )
	{
		gtk_widget_reparent( ( GTK_BIN( bin ) ) ->child, GTK_WIDGET( window ) );
	}

	void InterpretWidgets( GtkBin *bin )
	{
		GtkMenu * menu = GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( window, "optionmenu_gradiate" ) ) ) );
		GtkWidget *active_item = gtk_menu_get_active( menu );
		type = g_list_index ( GTK_MENU_SHELL ( menu ) ->children, active_item );

		GtkEntry *entry = GTK_ENTRY( lookup_widget( window, "spinbutton_frames" ) );
		frames = atoi( gtk_entry_get_text( entry ) );

		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_start_left" ) ),
		                           &start_left_r, &start_left_g, &start_left_b, &start_left_a );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_start_right" ) ),
		                           &start_right_r, &start_right_g, &start_right_b, &start_right_a );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_end_left" ) ),
		                           &end_left_r, &end_left_g, &end_left_b, &end_left_a );
		gnome_color_picker_get_i8( GNOME_COLOR_PICKER( lookup_widget( window, "colorpicker_end_right" ) ),
		                           &end_right_r, &end_right_g, &end_right_b, &end_right_a );
	}

	int GetNumberOfFrames( )
	{
		return frames;
	}
};

/** Create a number of images that are generated from the input file.
*/

class ImageCreateFromFile : public ImageCreateColourRange
{
protected:
	static char file[ PATH_MAX + NAME_MAX ];
	GtkWidget *file_entry;

public:
	char *GetDescription( ) const
	{
		return _( "Create From File" );
	}

	void CreateFrame( uint8_t *pixels, int width, int height, double position, double frame_delta )
	{
		if ( position == 0 )
		{
			GError * gerror = NULL;
			GdkPixbuf *splash = gdk_pixbuf_new_from_file( file, &gerror );
			if ( gerror != NULL )
			{
				modal_message( gerror->message );
				g_error_free( gerror );
			}
			else
			{
				GdkPixbuf *image = gdk_pixbuf_scale_simple( splash, width, height, GDK_INTERP_HYPER );
				memcpy( pixels, gdk_pixbuf_get_pixels( image ), width * height * 3 );
				g_object_unref( splash );
				g_object_unref( image );
			}
		}
	}

	void AttachWidgets( GtkBin *bin )
	{
		GtkWidget * table = gtk_table_new ( 2, 2, FALSE );

		GtkWidget *label = gtk_label_new( _( "File " ) );
		gtk_widget_show( label );

		file_entry = gnome_file_entry_new( NULL, _( "Select an Image" ) );
		gtk_entry_set_text( GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( file_entry ) ) ), file );
		gtk_widget_show( file_entry );

		AddFrameCollector( table, 0 );
		gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 1, 2, ( GtkAttachOptions ) ( 0 ), ( GtkAttachOptions ) ( 0 ), 0, 0 );
		gtk_table_attach( GTK_TABLE( table ), file_entry, 1, 2, 1, 2, ( GtkAttachOptions ) ( GTK_EXPAND | GTK_FILL ), ( GtkAttachOptions ) ( 0 ), 0, 0 );

		gtk_widget_show( table );
		gtk_container_add( GTK_CONTAINER( bin ), table );
	}

	void DetachWidgets( GtkBin *bin )
	{
		if ( bin->child != NULL )
		{
			// Make sure we save the current settings (this should be a method, rather than a duplicate
			// of InterpretWidgets ... since Interpret may throw exceptions, it can't be called directly
			// here...)
			InterpretFrameCollector( );
			GtkEntry *entry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( file_entry ) ) );
			strcpy( file, gtk_entry_get_text( entry ) );
			gtk_container_remove( GTK_CONTAINER( bin ), bin->child );
		}
	}

	void InterpretWidgets( GtkBin *bin )
	{
		InterpretFrameCollector( );
		GtkEntry *entry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( file_entry ) ) );
		strcpy( file, gtk_entry_get_text( entry ) );

		if ( !strcmp( file, "" ) )
			throw _( "No image file name specified - aborting." );
	}
};

char ImageCreateFromFile::file[ PATH_MAX + NAME_MAX ] = "";

/** Callback for selection change.
*/

static void
on_optionmenu_selected ( GtkMenuItem *menu_item, gpointer user_data )
{
	( ( GDKImageCreateRepository * ) user_data ) ->SelectionChange();
}

/** Constructor for the image creator repository.
 
  	Registers an instance of each known creator for later GUI exposure via the Initialise method.
*/

GDKImageCreateRepository::GDKImageCreateRepository() : selected_creator( NULL ), menu( NULL ), container( NULL )
{
	printf( ">> image creator repository created\n" );
	// Register an instance of each object
	Register( new ImageCreateColour() );
	Register( new ImageCreateNoise() );
	Register( new ImageCreateColourRange() );
	Register( new ImageCreateGradiate() );
	Register( new ImageCreateFromFile() );
}

/** Destructor for the image repository - clears all the registered creators
*/

GDKImageCreateRepository::~GDKImageCreateRepository()
{
	printf( ">> image creator repository destroyed\n" );
	// Remove the creators in the repository
	for ( unsigned int index = 0; index < creators.size(); index ++ )
		delete creators[ index ];
}

/** Register an image creator creator.
*/

void GDKImageCreateRepository::Register( GDKImageCreate *creator )
{
	printf( ">>> Image Create: %s\n", creator->GetDescription( ) );
	creators.push_back( creator );
}

/** Initialise the option menu with the current list of registered creators.
*/

void GDKImageCreateRepository::Initialise( GtkOptionMenu *menu, GtkBin *container )
{
	// Store these for future reference
	this->menu = menu;
	this->container = container;

	// Add the creators to the menu
	GtkMenu *menu_new = GTK_MENU( gtk_menu_new( ) );
	for ( unsigned int index = 0; index < creators.size(); index ++ )
	{
		GtkWidget *item = gtk_menu_item_new_with_label( creators[ index ] ->GetDescription( ) );
		gtk_widget_show( item );
		gtk_menu_append( menu_new, item );
		g_signal_connect( G_OBJECT( item ), "activate", G_CALLBACK( on_optionmenu_selected ), this );
	}
	gtk_menu_set_active( menu_new, 0 );
	gtk_option_menu_set_menu( menu, GTK_WIDGET( menu_new ) );

	// Register the selected items widgets
	SelectionChange();
}

/** Get the currently selected image creator creator.
*/

GDKImageCreate *GDKImageCreateRepository::Get( ) const
{
	GtkMenu * creatorMenu = GTK_MENU( gtk_option_menu_get_menu( menu ) );
	GtkWidget *active_item = gtk_menu_get_active( creatorMenu );
	return creators[ g_list_index( GTK_MENU_SHELL( creatorMenu ) ->children, active_item ) ];
}

/** Handle attach/detach widgets on last selected/selected items.
*/

void GDKImageCreateRepository::SelectionChange( )
{
	// Detach the selected creators widgets
	if ( selected_creator != NULL )
		selected_creator->DetachWidgets( container );

	// Get the new selection
	selected_creator = Get();

	// Inform the main page of the change
	if ( common != NULL && common->getPageMagick( ) != NULL )
		common->getPageMagick( ) ->RefreshStatus( );

	// Attach the new creators widgets
	if ( selected_creator != NULL )
		selected_creator->AttachWidgets( container );
}
