/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "filetransactions.h"

//Qt
#include <qvector.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qprocess.h>

//KDE
#include <kurl.h>
#include <kmountpoint.h>
#include <ktemporaryfile.h>
#include <ksavefile.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>

void FileTransactions::fileInput( const KUrl menulst, const KUrl devicemap, GRUB::ConfigFile::Settings *settings, QVector<GRUB::ConfigFile::Entry> *entries, QVector<GRUB::Misc::Device> *devices )
{
	settings->clear();
	entries->clear();
	devices->clear();

	QTextStream stream;
	QRegExp space( "\\s+" );
	QRegExp separator( "(\\s+|=|\\s+=\\s+)" );
	QString line, output, tmpMenulst, tmpDevicemap;

	if ( KIO::NetAccess::download( menulst, tmpMenulst, 0 ) )
	{
		QFile fileMenulst( tmpMenulst );
		if ( fileMenulst.open( QIODevice::ReadOnly ) )
		{
			stream.setDevice( &fileMenulst );
			while ( !stream.atEnd() )
			{
				line = stream.readLine().trimmed();
				//GRUB::ConfigFile::Settings identification
				if ( line.startsWith( "splashimage", Qt::CaseInsensitive ) )
				{
					settings->setSplashImage( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "gfxmenu", Qt::CaseInsensitive ) )
				{
					settings->setGfxMenu( line.section( separator, 1 ) );
					continue;
				}
				if (line.startsWith( "default", Qt::CaseInsensitive ) )
				{
					( line.section( separator, 1 ).toLower() == "saved" ? settings->setDefault( -2 ) : settings->setDefault( line.section( separator, 1 ).toInt() ) );
					continue;
				}
				if (line.startsWith( "fallback", Qt::CaseInsensitive ) )
				{
					settings->setFallback( line.section( separator, 1 ).toInt() );
					continue;
				}
				if ( line.startsWith( "timeout", Qt::CaseInsensitive ) )
				{
					settings->setTimeout( line.section( separator, 1 ).toInt() );
					continue;
				}
				if ( line.startsWith( "hiddenmenu", Qt::CaseInsensitive ) )
				{
					settings->setHiddenMenu( true );
					continue;
				}
				if ( line.startsWith( "map", Qt::CaseInsensitive ) )
				{
					settings->addMap( GRUB::ComplexCommand::Map( line.section( separator, 1 ) ) );
					continue;
				}
				if ( line.startsWith( "color", Qt::CaseInsensitive ) )
				{
					settings->setColor( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "password", Qt::CaseInsensitive ) )
				{
					settings->setPassword( line.section( separator, 1 ) );
					continue;
				}
				//GRUB::ConfigFile::Entry identified
				if ( line.startsWith( "title", Qt::CaseInsensitive ) )
				{
					GRUB::ConfigFile::Entry tmp_entry;
					tmp_entry.setTitle( line.section( separator, 1 ) );

					while ( !stream.atEnd() )
					{
						line = stream.readLine().trimmed();
						if ( line.startsWith( "title", Qt::CaseInsensitive ) )
						{
							entries->append( tmp_entry );
							tmp_entry.clear();
							tmp_entry.setTitle( line.section( separator, 1 ) );
						}
						if ( line.startsWith( "lock", Qt::CaseInsensitive ) )
						{
							tmp_entry.setLock( true );
							continue;
						}
						if ( line.startsWith( "password", Qt::CaseInsensitive ) )
						{
							tmp_entry.setPassword( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "root", Qt::CaseInsensitive ) )
						{
							tmp_entry.setRoot( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "kernel", Qt::CaseInsensitive ) )
						{
							tmp_entry.setKernel( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "initrd", Qt::CaseInsensitive ) )
						{
							tmp_entry.setInitrd( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "map", Qt::CaseInsensitive ) )
						{
							tmp_entry.addMap( GRUB::ComplexCommand::Map( line.section( separator, 1 ) ) );
							continue;
						}
						if ( line.startsWith( "color", Qt::CaseInsensitive ) )
						{
							tmp_entry.setColor( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "chainloader", Qt::CaseInsensitive ) )
						{
							tmp_entry.setChainLoader( line.section( separator, 1 ) );
							continue;
						}
						if ( line.startsWith( "savedefault", Qt::CaseInsensitive ) )
						{
							tmp_entry.setSaveDefault( true );
							continue;
						}
						if ( line.startsWith( "makeactive", Qt::CaseInsensitive ) )
						{
							tmp_entry.setMakeActive( true );
							continue;
						}
					}
					entries->append( tmp_entry );
					tmp_entry.clear();
				}
			}
			fileMenulst.close();
		}
		else
		{
			kWarning() << fileMenulst.errorString();
			KMessageBox::error( 0, fileMenulst.errorString() );
		}
		KIO::NetAccess::removeTempFile( tmpMenulst );
	}
	else
	{
		kWarning() << KIO::NetAccess::lastErrorString();
		KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
	}

	KMountPoint::List mountPoints = KMountPoint::currentMountPoints();
	for( KMountPoint::List::ConstIterator it = mountPoints.begin(); it != mountPoints.end(); ++it )
	{
		const KMountPoint::Ptr mp = ( *it );
		if ( mp->mountedFrom().startsWith( "/dev" ) )
			devices->append( GRUB::Misc::Device( mp->mountedFrom().remove( QRegExp( "\\d" ) ), mp->mountedFrom(), mp->mountPoint() ) );
	}

	if ( KIO::NetAccess::download( devicemap, tmpDevicemap, 0 ) )
	{
		QFile fileDevicemap( tmpDevicemap );
		if ( fileDevicemap.open( QIODevice::ReadOnly ) )
		{
			stream.setDevice( &fileDevicemap );
			while ( !stream.atEnd() )
			{
				line = stream.readLine();
				for ( int i = 0 ; i < devices->size() ; i++ )
				{
					if ( devices->at( i ).device() == line.section( space, 1, 1 ) )
					{
						(*devices)[i].setGrubDevice( line.section( space, 0, 0 ) );
						(*devices)[i].setGrubPartition( (*devices)[i].grubDevice().replace( QString( ")" ), QString( "," ) + QString().setNum( (*devices)[i].partition().right( 1 ).toInt() - 1 ) + QString( ")" ) ) );
					}
				}
			}
			fileDevicemap.close();
		}
		else
		{
			kWarning() << fileDevicemap.errorString();
			KMessageBox::error( 0, fileDevicemap.errorString() );
		}
		KIO::NetAccess::removeTempFile( tmpDevicemap );
	}
	else
	{
		kWarning() << KIO::NetAccess::lastErrorString();
		KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
	}

	QProcess uuid;
	uuid.start( "ls" , QStringList() << "/dev/disk/by-uuid/" << "-l" );
	uuid.waitForFinished();
	output = uuid.readAllStandardOutput();
	for ( int j = 1 ; j < output.count( "\n" ) ; j++ )
	{
		for ( int i = 0 ; i < devices->size() ; i++ )
		{
			if ( devices->at( i ).partition() == output.section( "\n", j, j ).section( space, 9, 9 ).remove( "../.." ).prepend( "/dev" ) )
			{
				(*devices)[i].setUuid( output.section( "\n", j, j ).section( space, 7, 7 ) );
			}
		}
	}

}
void FileTransactions::fileOutput( const KUrl menulst, const GRUB::ConfigFile::Settings *settings, const QVector<GRUB::ConfigFile::Entry> *entries )
{
	QString tmpMenulst;
	if ( menulst.isLocalFile() )
		tmpMenulst = menulst.path();
	else
	{
		KTemporaryFile tmp;
		if ( !tmp.open() )
		{
			kWarning() << tmp.errorString();
			KMessageBox::error( 0, tmp.errorString() );
			return;
		}
		tmpMenulst = tmp.fileName();
	}

	KSaveFile fileMenulst( tmpMenulst );
	if ( !fileMenulst.open() )
	{
		kWarning() << fileMenulst.errorString();
		KMessageBox::error( 0, fileMenulst.errorString() );
		return;
	}

	QTextStream stream( &fileMenulst );
	if ( !settings->splashImage().isEmpty() )
			stream << "splashimage " << settings->splashImage() << endl;
	if ( !settings->gfxMenu().isEmpty() )
			stream << "gfxmenu " << settings->gfxMenu() << endl;
	if ( settings->hiddenMenu() )
			stream << "hiddenmenu" << endl;
	if ( settings->_default() != -1 )
			stream << "default " << ( settings->_default() != -2 ? QString().setNum( settings->_default() ) : "saved" ) << endl;
	if ( settings->fallback() != -1 )
			stream << "fallback " << settings->fallback() << endl;
	if ( settings->timeout() != -1 )
			stream << "timeout " << settings->timeout() << endl;
	if ( !settings->maps().isEmpty() )
		for ( int i = 0; i < settings->maps().size(); i++ )
			stream << "map " << settings->maps().at( i ) << endl;
	if ( !settings->color().isEmpty() )
			stream << "color " << settings->color() << endl;
	if ( !settings->password().isEmpty() )
			stream << "password " << settings->password() << endl;
	for ( int i = 0; i < entries->size(); i++ )
	{
			stream << endl;
			stream << "title " << entries->at( i ).title() << endl;
		if ( entries->at( i ).lock() )
			stream << "lock" << endl;
		if ( !entries->at( i ).password().isEmpty() )
			stream << "password " << entries->at( i ).password() << endl;
		if ( !entries->at( i ).root().isEmpty() )
			stream << "root " << entries->at( i ).root() << endl;
		if ( !entries->at( i ).kernel().isEmpty() )
			stream << "kernel " << entries->at( i ).kernel() << endl;
		if ( !entries->at( i ).initrd().isEmpty() )
			stream << "initrd " << entries->at( i ).initrd() << endl;
		if ( !entries->at( i ).maps().isEmpty() )
			for ( int j = 0; j < entries->at( i ).maps().size(); j++ )
				stream << "map " << entries->at( i ).maps().at( j ) << endl;
		if ( !entries->at( i ).color().isEmpty() )
			stream << "color " << entries->at( i ).color() << endl;
		if (!entries->at( i ).chainLoader().isEmpty())
			stream << "chainloader " << entries->at( i ).chainLoader() << endl;
		if ( entries->at( i ).saveDefault() )
			stream << "savedefault" << endl;
		if ( entries->at( i ).makeActive() )
			stream << "makeactive" << endl;
	}
	stream.flush();

	if ( !fileMenulst.finalize() )
	{
		kWarning() << fileMenulst.errorString();
		KMessageBox::error( 0, fileMenulst.errorString() );
		return;
	}

	if ( !menulst.isLocalFile() )
	{
		if ( !KIO::NetAccess::upload( tmpMenulst, menulst, 0 ) )
		{
			kWarning() << KIO::NetAccess::lastErrorString();
			KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
			return;
		}
	}
}
QString FileTransactions::view( const KUrl file )
{
	QString tmpFile, contents;
	if ( KIO::NetAccess::download( file, tmpFile, 0 ) )
	{
		QFile _file( tmpFile );
		if ( _file.open( QIODevice::ReadOnly ) )
		{
			QTextStream stream( &_file );
			contents = stream.readAll();
		}
		else
		{
			kWarning() << file.errorString();
			KMessageBox::error( 0, file.errorString() );
			return QString();
		}
		KIO::NetAccess::removeTempFile( tmpFile );
		return contents;
	}
	else
	{
		kWarning() << KIO::NetAccess::lastErrorString();
		KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
		return QString();
	}
}

void FileTransactions::backup( const KUrl source, const KUrl target )
{
	if ( KIO::NetAccess::exists( target, false, 0 ) )
	{
		if ( KIO::NetAccess::del( target, 0 ) )
		{
			if ( KIO::NetAccess::file_copy( source, target ) )
			{
				kDebug() << QString( "Operation was successfully completed! %1 has been backed up to %2." ).arg( source.path() ).arg( target.path() );
				KMessageBox::information( 0, i18n( "Operation was successfully completed! %1 has been backed up to %2.", source.path(), target.path() ), i18n( "Success!" ) );
			}
			else
			{
				kWarning() << KIO::NetAccess::lastErrorString();
				KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
			}
		}
		else
		{
			kWarning() << KIO::NetAccess::lastErrorString();
			KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
		}
	}
	else
	{
		if ( KIO::NetAccess::file_copy( source, target ) )
		{
			kDebug() << QString( "Operation was successfully completed! %1 has been backed up to %2." ).arg( source.path() ).arg( target.path() );
			KMessageBox::information( 0, i18n( "Operation was successfully completed! %1 has been backed up to %2.", source.path(), target.path() ), i18n( "Success!" ) );
		}
		else
		{
			kWarning() << KIO::NetAccess::lastErrorString();
			KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
		}
	}
}
void FileTransactions::restoreBackup( const KUrl source, const KUrl target)
{
	if ( KIO::NetAccess::del( target, 0 ) )
	{
		if ( KIO::NetAccess::file_copy( source, target ) )
		{
			kDebug() << QString( "Operation was successfully completed! %1 has been restored to %2." ).arg( source.path() ).arg( target.path() );
			KMessageBox::information( 0, i18n( "Operation was successfully completed! %1 has been restored to %2.", source.path(), target.path() ), i18n( "Success!" ) );
		}
		else
		{
			kWarning() << KIO::NetAccess::lastErrorString();
			KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
		}
	}
	else
	{
		kWarning() << KIO::NetAccess::lastErrorString();
		KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
	}
}
void FileTransactions::deleteBackup( const KUrl backup )
{
	kDebug() << "Deleting backup" << backup;
	if ( KIO::NetAccess::del( backup, 0 ) )
	{
		kDebug() << backup << "was successfully deleted.";
		KMessageBox::information( 0, i18n( "Operation was successfully completed! %1 has been deleted.", backup.path() ) );
	}
	else
	{
		kWarning() << KIO::NetAccess::lastErrorString();
		KMessageBox::error( 0, KIO::NetAccess::lastErrorString() );
	}
}

void FileTransactions::moveEntry( const int source, const int target, const KUrl menulst, const GRUB::ConfigFile::Settings *settings, QVector<GRUB::ConfigFile::Entry> *entries )
{
	if ( source != target )
	{
		GRUB::ConfigFile::Entry tmp_entry = (*entries)[source];
		entries->remove( source );
		entries->insert( target, tmp_entry );
		fileOutput( menulst, settings, entries );
	}
}

QString FileTransactions::convertToGRUBPath( const QString path, const QVector<GRUB::Misc::Device> devices )
{
	if ( path.isEmpty() )
		return QString();

	if ( KMountPoint::Ptr mp = KMountPoint::currentMountPoints().findByPath( path ) )
	{
		for ( int i = 0; i < devices.size() ; i++ )
		{
			if ( mp->mountPoint() == devices.at( i ).mountPoint() )
			{
				if ( mp->mountPoint() == "/" )
					return QString( path ).prepend( devices.at( i ).grubPartition() );
				else
					return QString( path ).replace( mp->mountPoint(), devices.at( i ).grubPartition() );
			}
		}
	}
	return QString();
}
QString FileTransactions::convertToGenericPath( const QString path, const QVector<GRUB::Misc::Device> devices )
{
	if ( path.isEmpty() )
		return QString();

	for ( int i = 0; i < devices.size() ; i++ )
	{
		if ( path.startsWith( devices.at( i ).grubPartition() ) )
		{
			return QString( path ).replace( devices.at( i ).grubPartition(), devices.at( i ).mountPoint() );
		}
	}
	return QString();
}
