/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 * Copyright (C) 2004 Markus Brueffer <markus@brueffer.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "bookmarks/editortls.h"
#include "kftpquickconnectdialog.h"
#include "misc/config.h"
#include "kftpbookmarks.h"
#include "misc.h"
#include "kftpclientthread.h"
#include "ftpsocket.h"

#include <kglobal.h>
#include <kcharsets.h>
#include <kcombobox.h>
#include <klocale.h>
#include <klineedit.h>
#include <kpassdlg.h>
#include <kmessagebox.h>
#include <kpushbutton.h>

/* KSSL includes */
#include <ksslpkcs12.h>

#include <qcheckbox.h>
#include <qspinbox.h>

using namespace KFTPGrabberBase;

KFTPQuickConnectDialog::KFTPQuickConnectDialog(QWidget *parent, const char *name)
 : KDialogBase(parent, name, true, i18n("Quick Connect"), Ok|Cancel, Ok),
   m_noUrlChange(false),
   m_protocolAdvancedDialog(0L),
   m_portChanged(false)
{
  // Create the main widget
  m_layout = new KFTPQuickConnectLayout(this);

  // Set the dialog options
  setMainWidget(m_layout);
  setInitialSize(QSize(200,300));
  setMaximumHeight(300);

  m_layout->protoAdvanced->setIconSet(loadSmallIcon("configure"));
  
  // Connect the slots
  connect(m_layout->urlBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotUrlChanged(const QString &)));
  connect(m_layout->hostBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotHostChanged(const QString&)));
  connect(m_layout->protocolBox, SIGNAL(activated(int)), this, SLOT(slotProtocolChanged(int)));
  connect(m_layout->protoAdvanced, SIGNAL(clicked()), this, SLOT(slotProtoAdvancedClicked()));
  connect(m_layout->usernameBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotUserChanged()));
  connect(m_layout->passwordBox, SIGNAL(textChanged(const QString&)), this, SLOT(slotPassChanged()));
  connect(m_layout->portBox, SIGNAL(valueChanged(int)), this, SLOT(slotPortChanged(int)));
  connect(m_layout->anonLogin, SIGNAL(clicked()), this, SLOT(slotAnonClicked()));

  // Init url
  m_curUrl.setProtocol("ftp");
  m_curUrl.setPort(21);
  m_curUrl.setPath("/");
  m_layout->urlBox->setURL(m_curUrl);
  
  // Use anonymous account by default
  m_layout->anonLogin->setChecked(true);
  slotAnonClicked();
  
  // Init the protocol advanced button
  m_layout->protoAdvanced->setEnabled(false);
  
  // Populate charsets
  QStringList charsets = KGlobal::charsets()->descriptiveEncodingNames();
  m_layout->serverEncoding->insertStringList(charsets);
  
  QString defEncoding = KFTPCore::Config::defEncoding();
  defEncoding = QString("%1 ( %2 )").arg(KGlobal::charsets()->languageForEncoding(defEncoding)).arg(defEncoding);
  m_layout->serverEncoding->setCurrentText(defEncoding);
}

KFTPQuickConnectDialog::~KFTPQuickConnectDialog()
{
  if (m_protocolAdvancedDialog)
    delete m_protocolAdvancedDialog;
}

KURL KFTPQuickConnectDialog::getServerURL()
{
  return m_curUrl;
}

void KFTPQuickConnectDialog::slotUrlChanged(const QString &text)
{
  if (m_noUrlChange) return;
  
  m_layout->anonLogin->setChecked(false);
  m_layout->usernameBox->setEnabled(true);
  m_layout->passwordBox->setEnabled(true);
  
  KURL tmpUrl = text;
  if (!tmpUrl.isValid())
    return;
  else
    m_curUrl = tmpUrl;

  m_noUrlChange = true;
  
  if (m_curUrl.protocol() == "ftp")
    m_layout->protocolBox->setCurrentItem(SP_FTP);
  else if (m_curUrl.protocol() == "sftp")
    m_layout->protocolBox->setCurrentItem(SP_SFTP);
  else {
    // Force FTP protocol
    m_curUrl.setProtocol("ftp");
    m_layout->protocolBox->setCurrentItem(SP_FTP);
  }
  
  m_layout->hostBox->setText(m_curUrl.host());
  m_layout->usernameBox->setText(m_curUrl.user());
  
  if (m_curUrl.hasPass()) {
    m_layout->passwordBox->erase();
    m_layout->passwordBox->insert(m_curUrl.pass());
  }
  
  if (m_curUrl.port() == 0) {
    switch (m_layout->protocolBox->currentItem()) {
      case SP_SFTP: {
        m_layout->portBox->setValue(22);
        m_curUrl.setPort(22);
        break;
      }
      default: {
        m_layout->portBox->setValue(21);
        m_curUrl.setPort(21);
        break;
      }
    }
  } else {
    m_layout->portBox->setValue(m_curUrl.port());
  }
  
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotHostChanged(const QString &text)
{
  if (m_noUrlChange) return;

  m_noUrlChange = true;
  m_curUrl.setHost(text);
  m_layout->urlBox->setURL(m_curUrl);
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotPortChanged(int port)
{
  if (m_noUrlChange) return;

  m_noUrlChange = true;
  m_curUrl.setPort(port);
  m_layout->urlBox->setURL(m_curUrl);
  m_portChanged = true;
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotUserChanged()
{
  if (m_noUrlChange) return;

  m_noUrlChange = true;
  m_curUrl.setUser(m_layout->usernameBox->text());
  m_layout->urlBox->setURL(m_curUrl);
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotPassChanged()
{
  if (m_noUrlChange) return;

  m_noUrlChange = true;
  m_curUrl.setPass(m_layout->passwordBox->password());
  m_layout->urlBox->setURL(m_curUrl);
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotAnonClicked()
{
  static QString tmpUser, tmpPass;
  
  if (m_layout->anonLogin->isChecked()) {
    m_layout->usernameBox->setEnabled(false);
    m_layout->passwordBox->setEnabled(false);
    tmpUser = m_layout->usernameBox->text();
    tmpPass = m_layout->passwordBox->text();
    m_layout->usernameBox->setText("anonymous");
    m_layout->passwordBox->erase();
    
    // Use the appropriate e-mail address for anonymous accounts
    if (!KFTPCore::Config::anonMail().isEmpty())
      m_layout->passwordBox->insert(KFTPCore::Config::anonMail());
    else
      m_layout->passwordBox->insert("userlogin@anonymo.us");
  } else {
    m_layout->usernameBox->setText(tmpUser);
    m_layout->passwordBox->erase();
    m_layout->passwordBox->insert(tmpPass);
    m_layout->usernameBox->setEnabled(true);
    m_layout->passwordBox->setEnabled(true);
  }
  
  slotUserChanged();
  slotPassChanged();
}

void KFTPQuickConnectDialog::slotOk()
{
  // Construct a nice error message
  QString errorMessage;
  
  if (m_curUrl.host().isEmpty())
    errorMessage = i18n("a hostname");
     
  if (m_curUrl.port() < 1)
    errorMessage += (errorMessage.isEmpty() ? QString::null : QString::fromLatin1(", ")) + i18n("a valid port");
  
  if (m_curUrl.user().isEmpty())
    errorMessage += (errorMessage.isEmpty() ? QString::null : QString::fromLatin1(", ")) + i18n("your username");
   
  if (m_curUrl.pass().isEmpty())
    errorMessage += (errorMessage.isEmpty() ? QString::null : QString::fromLatin1(", ")) + i18n("your password");
  
  if (errorMessage.findRev(",") != -1)
    errorMessage = errorMessage.replace(errorMessage.findRev(","), 1 , i18n(" and"));
  
  if (!errorMessage.isEmpty()) {
    KMessageBox::sorry(0, i18n("Please enter ") +  errorMessage + ".");
    return;
  }
  
  if (m_layout->addBookmark->isChecked()) {
    // Add the current connection to bookmarks. Use hostname as the bookmark name.
    KFTPBookmarks::Site *root = FTPBookmarkManager->findCategory("root");
    KFTPBookmarks::Site *site = root->addSite();
    site->setAttribute("name", m_curUrl.host());
    
    site->setProperty("host", m_curUrl.host());
    site->setProperty("port", m_curUrl.port());
    site->setProperty("username", m_curUrl.user());
    site->setProperty("password", encodePassword(m_curUrl.pass()));
    site->setProperty("defremotepath", "/");
    site->setProperty("protocol", m_layout->protocolBox->currentItem() == SP_SFTP ? "sftp" : "ftp");
    site->setProperty("use_tls", m_layout->protocolBox->currentItem() == SP_SSL_EXPLICIT);
    site->setProperty("use_implicit", m_layout->protocolBox->currentItem() == SP_SSL_IMPLICIT);
    site->setProperty("encoding", KGlobal::charsets()->encodingForName(m_layout->serverEncoding->currentText()));
    
    // Save TLS options
    if (m_protocolAdvancedDialog) {
      site->setProperty("tls_mode", QString(m_protocolAdvancedDialog->getTLSMode()));
      site->setProperty("use_cert", m_protocolAdvancedDialog->isCertChecked());
    
      if (m_protocolAdvancedDialog->isCertChecked())
        site->setProperty("tls_cert_path", m_protocolAdvancedDialog->getCertPath());
    }
    
    FTPBookmarkManager->emitUpdate();
  }

  // Close the dialog
  accept();
}

void KFTPQuickConnectDialog::setupClient(KFTPClientThr *client)
{
  // First activate the correct socket and reset the old flags
  client->getSocketManager()->setProtocol(KURL(QString("%1://test/").arg(m_layout->protocolBox->currentItem() == SP_SFTP ? "sftp" : "ftp")));
  client->getClient()->initConfig();
    
  client->getClient()->setConfig("do_retry", 0);
  
  client->getClient()->setConfig("use_tls", m_layout->protocolBox->currentItem() == SP_SSL_EXPLICIT);
  client->getClient()->setConfig("use_implicit", m_layout->protocolBox->currentItem() == SP_SSL_IMPLICIT);
  client->getClient()->setConfig("encoding", KGlobal::charsets()->encodingForName(m_layout->serverEncoding->currentText()));
  
  // Set TLS options
  if (m_protocolAdvancedDialog) {
    client->getClient()->setConfig("prot_mode", QString(m_protocolAdvancedDialog->getTLSMode()));
  
    // Should we use a X509 certificate ?
    if (m_protocolAdvancedDialog->isCertChecked() && m_layout->protocolBox->currentItem() == SP_FTP) {
      // Ask the user for the decryption password
      QCString p_pass;
      KPasswordDialog::getPassword(p_pass, i18n("Please provide your X509 certificate decryption password."));
      
      static_cast<KFTPNetwork::FtpSocket*>(client->getClient())->SSLSetClientCert(KSSLPKCS12::loadCertFile(m_protocolAdvancedDialog->getCertPath(), p_pass));
    }
  }
}

void KFTPQuickConnectDialog::slotProtocolChanged(int item)
{
  if (m_noUrlChange) return;
  
  // Enable/Disable the SSL/TLS settings if needed
  m_layout->protoAdvanced->setEnabled( item == SP_SSL_EXPLICIT || item == SP_SSL_IMPLICIT );
  
  // Set the default port
  if (!m_portChanged) {
    switch (item) {
      case SP_SSL_IMPLICIT: {
        m_layout->portBox->setValue(993);
        break;
      }
      case SP_SFTP: {
        m_layout->portBox->setValue(22);
        break;
      }
      default: {
        m_layout->portBox->setValue(21);
        break;
      }
    }
    
    m_portChanged = false;
  }
   
  m_noUrlChange = true;
  m_curUrl.setProtocol( item == SP_SFTP ? "sftp" : "ftp");
  m_curUrl.setPort( m_layout->portBox->value() );
  m_layout->urlBox->setURL(m_curUrl);
  m_noUrlChange = false;
}

void KFTPQuickConnectDialog::slotProtoAdvancedClicked()
{
  if (!m_protocolAdvancedDialog)
    m_protocolAdvancedDialog = new KFTPWidgets::Bookmarks::BookmarkEditorTLS(this);
    
  QChar tlsMode = m_protocolAdvancedDialog->getTLSMode();
  bool certChecked = m_protocolAdvancedDialog->isCertChecked();
  QString certPath = m_protocolAdvancedDialog->getCertPath();
  m_protocolAdvancedDialog->slotChangeActiveX509Group();
  
  if (!m_protocolAdvancedDialog->exec()) {
    m_protocolAdvancedDialog->setTLSMode(tlsMode);
    m_protocolAdvancedDialog->setCertChecked(certChecked);
    m_protocolAdvancedDialog->setCertPath(certPath);
    m_protocolAdvancedDialog->slotChangeActiveX509Group();
  }
}

QChar KFTPQuickConnectDialog::getTLSMode()
{
  if (!m_protocolAdvancedDialog)
    return false;
    
  return m_protocolAdvancedDialog->getTLSMode();
}

bool KFTPQuickConnectDialog::isCertChecked()
{
  if (!m_protocolAdvancedDialog)
    return false;
    
  return m_protocolAdvancedDialog->isCertChecked();
}

QString KFTPQuickConnectDialog::getCertPath()
{
  if (!m_protocolAdvancedDialog)
    return QString::null;
    
  return m_protocolAdvancedDialog->getCertPath();
}

int KFTPQuickConnectDialog::getFTPMode()
{
  return m_layout->protocolBox->currentItem();
}

void KFTPQuickConnectDialog::setHost(QString &host)
{
  m_layout->hostBox->setText(host);
}

void KFTPQuickConnectDialog::setPort(int port)
{
  m_layout->portBox->setValue(port);
}

void KFTPQuickConnectDialog::setFocusToUser()
{
  m_layout->usernameBox->setFocus();
}

#include "kftpquickconnectdialog.moc"
