/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "sftpsocket.h"
#include "misc/config.h"

#include <qdir.h>

#include <klocale.h>
#include <kio/job.h>
#include <kio/renamedlg.h>

#include <sys/stat.h>

#define F_STAT(x) FTPDirectoryItem(protoStat(KURL(x)))

namespace KFTPNetwork {

SftpSocket::SftpSocket(QObject *parent)
 : Socket(parent, "sftp"), m_curDir(0)
{
}

int SftpSocket::getFeatures()
{
  return 0;
}

void SftpSocket::initConfig()
{
  Socket::initConfig();
}

void SftpSocket::get(KURL source, KURL destination)
{
  protoGet(source.path(), destination.path());
}

void SftpSocket::put(const KURL &source, const KURL &destination)
{
  protoPut(source.path(), destination.path());
}

void SftpSocket::remove(const KURL &url)
{
  /* Discover if path is file or directory and call the right method */
  if (checkIsDir(url)) {
    m_stateInfo.enterSocketState(S_REMOVE, true);

    recursiveDelete(url);
    protoRmdir(url.path());

    m_stateInfo.enterSocketState(S_IDLE, true);
    return;
  }

  protoDelete(url.path());
}

void SftpSocket::rename(const KURL &source, const KURL &destination)
{
  protoRename(source.path(), destination.path());
}

void SftpSocket::chmod(const KURL &url, int mode)
{
  SFTP_ATTRIBUTES *attrs = static_cast<SFTP_ATTRIBUTES*>(malloc(sizeof(SFTP_ATTRIBUTES)));
  memset(attrs, 0, sizeof(*attrs));
  
  attrs->permissions = intToPosix(mode);
  attrs->flags = SSH_FILEXFER_ATTR_PERMISSIONS;
  
  sftp_setstat(m_sftpSession, (char*) url.path().ascii(), attrs);
  sftp_attributes_free(attrs);
}

void SftpSocket::mkdir(const KURL &url)
{
  protoMkdir(url.path());
}

void SftpSocket::stat(const KURL &url)
{
  emit sigStatResult(protoStat(url));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////  SFTP PROTOCOL  //////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int SftpSocket::protoConnect()
{
  // Set connection info
  m_sshOptions = options_new();
  options_set_username(m_sshOptions, (char*) getClientInfoUrl().user().ascii());
  options_set_host(m_sshOptions, getClientInfoUrl().host().ascii());
  options_set_port(m_sshOptions, getClientInfoUrl().port());
  
  // Connect to the remote host
  m_sftpSession = 0;
  m_sshSession = ssh_connect(m_sshOptions);
  
  if (!m_sshSession) {
    emit sigLogUpdate(4, i18n("Unable to establish SSH connection (%1)").arg(ssh_get_error(0)));
    return -1;
  }
  
  return 1;
}

void SftpSocket::protoDisconnect()
{
  if (m_curDir)
    protoCloseDir();
    
  if (m_sftpSession)
    sftp_free(m_sftpSession);
    
  ssh_disconnect(m_sshSession);
  m_sshSession = 0;
  
  m_isLoggedIn = false;
}

int SftpSocket::keyboardInteractiveLogin()
{
  int err = ssh_userauth_kbdint(m_sshSession, NULL, NULL);
  char *name, *instruction, *prompt;
  int i, n;
  char echo;
  
  while (err == SSH_AUTH_INFO) {
    name = ssh_userauth_kbdint_getname(m_sshSession);
    instruction = ssh_userauth_kbdint_getinstruction(m_sshSession);
    n = ssh_userauth_kbdint_getnprompts(m_sshSession);
    
    // FIXME Name and instruction are currently ignored. The libssh API reference
    // suggests displaying an interactive dialog box for the user to supply the
    // information requested from the server.
        
    for(i = 0; i < n; ++i) {
      prompt = ssh_userauth_kbdint_getprompt(m_sshSession, i, &echo);
      
      if (!echo) {
        // We should send the password (since only the password should be masked)
        ssh_userauth_kbdint_setanswer(m_sshSession, i, (char*) getClientInfoUrl().pass().ascii());
      } else {
        // FIXME Server requests something else ?
      }
    }
    
    err = ssh_userauth_kbdint(m_sshSession, NULL, NULL);
  }
  
  return err;
}

int SftpSocket::protoLogin()
{
  if (m_isLoggedIn) return 1;

  m_stateInfo.enterSocketState(S_LOGIN);
  
  // Try the public key auth with the set password (if any)
  int pkey_ret = ssh_userauth_autopubkey(m_sshSession, getConfigStr("pkey_pass").ascii());
  if (pkey_ret == -666) {
    // Make a password request
    m_errorHandler->dispatchError(EC_SSH_PKEY_PASSWD);
    FTP_EXCEPTION;
  } else if (pkey_ret != SSH_AUTH_SUCCESS) {
    // First let's try the keyboard-interactive authentification
    if (keyboardInteractiveLogin() != SSH_AUTH_SUCCESS) {
      // If this fails, let's try the password authentification
      if (ssh_userauth_password(m_sshSession, NULL, (char*) getClientInfoUrl().pass().ascii()) != SSH_AUTH_SUCCESS) {
        m_stateInfo.enterSocketState(S_IDLE);
        return -1;
      }
    }
  }
  
  m_sftpSession = sftp_new(m_sshSession);
  if (!m_sftpSession) {
    emit sigLogUpdate(4, i18n("Unable to initialize SFTP channel."));
    m_stateInfo.enterSocketState(S_IDLE);
    return -1;
  }
  
  if (sftp_init(m_sftpSession)) {
    emit sigLogUpdate(4, i18n("Unable to initialize SFTP."));
    m_stateInfo.enterSocketState(S_IDLE);
    return -1;
  }
  
  emit sigLogUpdate(3, i18n("Connection established."));
  
  m_stateInfo.enterSocketState(S_IDLE);
  m_isLoggedIn = true;
  return 1;
}

char addPermStr(bool cond, char add)
{
  if (cond)
    return add;
  else
    return '-';
}

int addPermInt(int &x, int n, int add)
{
  if (x >= n) {
    x -= n;
    return add;
  } else {
    return 0;
  }
}

int SftpSocket::intToPosix(int permissions)
{
  int posix = 0;
  QString str = QString::number(permissions);
  
  int user = str.mid(0, 1).toInt();
  int group = str.mid(1, 1).toInt();
  int other = str.mid(2, 1).toInt();
  
  posix |= addPermInt(user, 4, S_IRUSR);
  posix |= addPermInt(user, 2, S_IWUSR);
  posix |= addPermInt(user, 1, S_IXUSR);
  
  posix |= addPermInt(group, 4, S_IRGRP);
  posix |= addPermInt(group, 2, S_IWGRP);
  posix |= addPermInt(group, 1, S_IXGRP);
  
  posix |= addPermInt(other, 4, S_IROTH);
  posix |= addPermInt(other, 2, S_IWOTH);
  posix |= addPermInt(other, 1, S_IXOTH);
  
  return posix;
}

QString SftpSocket::posixToString(int permissions)
{
  QString str;
  
  str += addPermStr(permissions & S_IFDIR, 'd');
    
  // User
  str += addPermStr(permissions & S_IRUSR, 'r');
  str += addPermStr(permissions & S_IWUSR, 'w');
  str += addPermStr(permissions & S_IXUSR, 'x');
  
  // Group
  str += addPermStr(permissions & S_IRGRP, 'r');
  str += addPermStr(permissions & S_IWGRP, 'w');
  str += addPermStr(permissions & S_IXGRP, 'x');
  
  // Other
  str += addPermStr(permissions & S_IROTH, 'r');
  str += addPermStr(permissions & S_IWOTH, 'w');
  str += addPermStr(permissions & S_IXOTH, 'x');
  
  return str;
}

FTPEntry SftpSocket::protoStat(const KURL &url)
{
  FTPEntry entry;
  
  if (!protoDirList(url.upURL())) {
    // We have failed - signal that as well!
    FTPEntry fail;
    return fail;
  }
  
  for (FTPDirList::iterator i = m_lastDirList.begin(); i != m_lastDirList.end(); ++i) {
    if ((*i).name() == url.fileName()) {
      entry = (*i).m_ftpEntry;
      break;
    }
  }

  // We have all the data, signal that!
  return entry;
}

bool SftpSocket::protoDirList(const KURL &url)
{
  if (!m_isLoggedIn) {
    // We are not logged in
    return false;
  }
  
  if (!protoCwdError(url.path()))
    return false;
  
  emit sigLogUpdate(0, i18n("SFTP Reading directory listing..."));
    
  m_stateInfo.enterSocketState(S_LIST);
  m_lastDirList.clear();
  
  // Read the specified directory
  SFTP_ATTRIBUTES *file;
  FTPEntry entry;
  while ((file = sftp_readdir(m_sftpSession, m_curDir))) {
    entry.name = file->name;
    
    if (entry.name != "." && entry.name != "..") {
      entry.name = m_remoteEncoding->decode(entry.name.ascii());
      entry.owner = file->owner;
      entry.group = file->group;
      entry.date = file->mtime;
      entry.size = file->size;
      entry.link = "";
      entry.permissions = posixToString(file->permissions);
      
      if (file->permissions & S_IFDIR)
        entry.type = 'd';
      else
        entry.type = 'f';
      
      m_lastDirList.append(FTPDirectoryItem(entry));
    }
    
    sftp_attributes_free(file);
  }
  
  protoCloseDir();
  m_stateInfo.enterSocketState(S_IDLE);
  
  return true;
}

void SftpSocket::protoCloseDir()
{
  if (m_curDir) {
    sftp_dir_close(m_curDir);
    m_curDir = 0L;
  }
}

int SftpSocket::protoCwdError(const QString &dir)
{
  protoCloseDir();
  m_curDir = sftp_opendir(m_sftpSession, (char*) dir.ascii());
  
  if (!m_curDir) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_UNABLE_TO_ENTER_DIR, ErrorData(dir));
    FTP_EXCEPTION;
  }
  
  m_lastDir = dir;

  emit sigLogUpdate(0, i18n("SFTP Directory changed to '%1'").arg(dir));

  return 1;
}

int SftpSocket::protoCwd(const QString &dir)
{
  protoCloseDir();
  m_curDir = sftp_opendir(m_sftpSession, (char*) dir.ascii());
  
  return m_curDir ? 1 : 0;
}

int SftpSocket::protoRmdir(const QString &dir)
{
  return sftp_rmdir(m_sftpSession, (char*) dir.ascii()) < 0 ? -1 : 1;
}

int SftpSocket::protoDelete(const QString &path)
{
  return sftp_rm(m_sftpSession, (char*) path.ascii()) < 0 ? -1 : 1;
}

int SftpSocket::protoMkdir(const QString &dir)
{
  SFTP_ATTRIBUTES *attrs = static_cast<SFTP_ATTRIBUTES*>(malloc(sizeof(SFTP_ATTRIBUTES)));
  memset(attrs, 0, sizeof(*attrs));
  
  int ret = sftp_mkdir(m_sftpSession, (char*) dir.ascii(), attrs) < 0 ? -1 : 1;
  
  free(attrs);
  return ret;
}

int SftpSocket::protoRename(const QString &source, const QString &destination)
{
  return sftp_rename(m_sftpSession, (char*) source.ascii(), (char*) destination.ascii()) < 0 ? -1 : 1;
}

int SftpSocket::protoSize(const QString &file)
{
  QString parentDir = KURL(file).directory();
  QString fileName = KURL(file).fileName();
  if (!protoCwd(parentDir))
    return -1;
  
  bool found = false;
  SFTP_ATTRIBUTES *f;
  while ((f = sftp_readdir(m_sftpSession, m_curDir))) {
    if (QString(f->name) == fileName) {
      m_lastSize = f->size;
      found = true;
      break;
    }
  }
  
  if (!found)
    return -1;
    
  protoCloseDir();
  return 1;
}

int SftpSocket::protoGet(const QString &source, const QString &destination)
{
  SFTP_FILE *rf;
  FILE *f;
  filesize_t offset = 0;
  QDir fs;

  /* Check the sizes (and if the file exists) */
  filesize_t size_local = 0;
  if (protoSize(source) == -1) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
    FTP_EXCEPTION;
  }

  filesize_t size_remote = m_lastSize;

  /* Does destination (local) file exist ? */
  if (fs.exists(destination)) {
    // If the error hasn't been handled, dispatch the signal and
    // abort processing.
    if (m_errorHandler->dispatchError(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination, F_STAT(source))))
      FTP_EXCEPTION;

    // If we are still here, that means that the signal has been
    // handled. Check the return argument from the handler.
    if (m_errorHandler->returnCode(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination)) != KIO::R_OVERWRITE) {
      setConfig("feat_resume", 1);

      // Check if we can resume
      f = fopen(destination.local8Bit(), "a");
      if (f != NULL) {
        if (fseek(f, 0, SEEK_END) != 0) {
          fclose(f);
          
          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_FD_ERR);
          FTP_EXCEPTION;
        }

        size_local = ftell(f);

        if (size_local < size_remote && size_local > 0) {
          offset = size_local;
        }
      } else {
        // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FD_ERR);
        FTP_EXCEPTION;
      }
    } else {
      setConfig("feat_resume", 0);

      // Overwrite the file
      f = fopen(destination.local8Bit(), "w");

      if (f == NULL) {
        // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FD_ERR);
        FTP_EXCEPTION;
      }
    }

    // Invalidate the error message
    m_errorHandler->errorDone(EC_FILE_EXISTS_DOWNLOAD, ErrorData(destination));
  } else {
    // Check if the file path exists, if not, create it
    QString dest_dir = fs.cleanDirPath(destination.mid(0, destination.findRev('/')));

    // Is the path available ?
    if (!fs.exists(dest_dir)) {
      // Create all dirs
      QString full_path;
      for(register int i = 1;i<=dest_dir.contains('/');i++) {
        full_path += "/" + dest_dir.section('/', i, i);
        if (!fs.exists(full_path)) {
          fs.mkdir(full_path);
        }
      }
    }
    
    // Create the file
    f = fopen(destination.ascii(), "w");
    
    if (f == NULL) {
      // Dispatch error message and throw an exception so the processing will stop
        m_errorHandler->dispatchError(EC_FD_ERR);
        FTP_EXCEPTION;
    }
  }
  
  if (size_remote == size_local && getConfig("feat_resume") == 1) {
    /* File is already downloaded - skip it if enabled */
    fclose(f);
    processedSize(size_local);

    return 1;
  }
  
  emit sigLogUpdate(3, i18n("Starting with '%1' file download").arg(source));
  
  m_speedLimiter.transferStart();
  
  // Download the file
  char buffer[16384];
  filesize_t bytes_received = offset;
  time_t stallTimeout = 0;
  
  m_stateInfo.enterSocketState(S_TRANSFER);
  setOffset(offset);
  emit sigResumedOffset(offset);
  
  rf = sftp_open(m_sftpSession, (char*) source.ascii(), O_RDONLY, 0);
  if (!rf) {
    fclose(f);
    processedSize(size_local);
    m_stateInfo.enterSocketState(S_IDLE);
    
    return -1;
  }
  
  // Use the offset we are given
  if (offset > 0)
    sftp_seek(rf, offset);
  
  int b_read = 1;
  
  while (b_read) {
    int allowed = m_speedLimiter.allowedBytes();
    
    if (allowed > 16384)
      allowed = 16384;
    else if (allowed == 0)
      continue;
      
    b_read = sftp_read(rf, buffer, allowed);
    
    if (b_read > 0) {
      fwrite(buffer, b_read, 1, f);
      bytes_received += b_read;
  
      processedSize(b_read);
    }
    
    if (b_read <= 0 && getSpeed() == 0) {
      if (stallTimeout == 0) {
        // We are stalled for the first time
        stallTimeout = time(0);
      } else {
        // We have been stalled for some time, let's get the duration
        if (time(0) - stallTimeout > KFTPCore::Config::dataTimeout()) {
          Socket::disconnect();
    
          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_TIMEOUT);
          FTP_EXCEPTION;
        }
      }
    } else {
      stallTimeout = 0;
    }

    if (m_stateInfo.abortInProgress())
      break;
  }
  
  fflush(f);
  fclose(f);
  sftp_file_close(rf);
  
  m_stateInfo.enterSocketState(S_IDLE);
  
  emit sigLogUpdate(3, i18n("Transferred 1 byte.", "Transferred %n bytes.", bytes_received));
  
  return 1;
}

int SftpSocket::protoPut(const QString &source, const QString &destination)
{
  int fd;
  SFTP_FILE *rf;
  filesize_t offset = 0;
  QDir fs;

  /* Check the sizes (and if the file exists) */
  filesize_t size_local;
  filesize_t size_remote = 0;

  if (!fs.exists(source)) {
    // Dispatch error message and throw an exception so the processing will stop
    m_errorHandler->dispatchError(EC_FILE_NOT_FOUND);
    FTP_EXCEPTION;
  } else {
    /* Get local filesize */
    fd = open(source.ascii(), O_RDONLY);
    
    if (fd < 0) {
      // Dispatch error message and throw an exception so the processing will stop
      m_errorHandler->dispatchError(EC_FD_ERR);
      FTP_EXCEPTION;
    } else {
      size_local = lseek(fd, 0, SEEK_END);
      lseek(fd, 0, SEEK_SET);
    }
  }
  
  /* Does the destination (remote) file exist ? */
  if (protoSize(destination) == 1) {
    // If the error hasn't been handled, dispatch the signal and
    // abort processing.
    if (m_errorHandler->dispatchError(EC_FILE_EXISTS_UPLOAD, ErrorData(destination, F_STAT(destination))))
      FTP_EXCEPTION;

    // If we are still here, that means that the signal has been
    // handled. Check the return argument from the handler.
    if (m_errorHandler->returnCode(EC_FILE_EXISTS_UPLOAD, ErrorData(destination)) != KIO::R_OVERWRITE) {
      setConfig("feat_resume", 1);
      size_remote = m_lastSize;

      if (size_remote < size_local) {
        offset = size_remote;
        lseek(fd, offset, SEEK_SET);
      }
    }

    // Invalidate the error message
    m_errorHandler->errorDone(EC_FILE_EXISTS_UPLOAD, ErrorData(destination));
  }

  if (size_remote == size_local && getConfig("feat_resume") == 1) {
    /* File is already on the server - skip it if enabled */
    close(fd);
    processedSize(size_remote);
    return 1;
  }
  
  // Does the destination directory exist ?
  if (size_remote == 0 && !checkIsDir(KURL(destination).directory())) {
    // It doesn't - create it
    QString dest_dir = KURL(destination).directory();
    QString full_path;
    for(register int i = 1; i <= dest_dir.contains('/'); i++) {
      full_path += "/" + dest_dir.section('/', i, i);
      
      // Create the directory
      protoMkdir(full_path);
    }
  }
  
  emit sigLogUpdate(3, i18n("Starting with '%1' file upload").arg(source));
  
  m_speedLimiter.transferStart();
  
  // Open the file and grab the data
  char buffer[16384];
  filesize_t bytes_sent = offset;
  int b_read = 1;
  time_t stallTimeout = 0;

  m_stateInfo.enterSocketState(S_TRANSFER);
  setOffset(offset);
  emit sigResumedOffset(offset);
  
  if (offset > 0)
    rf = sftp_open(m_sftpSession, (char*) destination.ascii(), O_WRONLY | O_APPEND, 0);
  else
    rf = sftp_open(m_sftpSession, (char*) destination.ascii(), O_WRONLY | O_CREAT, 0);
    
  if (!rf) {
    close(fd);
    processedSize(size_remote);
    m_stateInfo.enterSocketState(S_IDLE);
    
    return -1;
  }
  
  // Move the file position to the specified offset
  sftp_seek(rf, offset);

  while (b_read) {
    int allowed = m_speedLimiter.allowedBytes();
    
    if (allowed > 16384)
      allowed = 16384;
    else if (allowed == 0)
      continue;
      
    b_read = read(fd, buffer, allowed);
    
    int b_sent = sftp_write(rf, buffer, b_read);
    
    if (b_sent > 0) {
      bytes_sent += b_sent;
      processedSize(b_sent);
    }
    
    if (b_sent <= 0) {
      if (stallTimeout == 0) {
        // We are stalled for the first time
        stallTimeout = time(0);
      } else {
        // We have been stalled for some time, let's get the duration
        if (time(0) - stallTimeout > KFTPCore::Config::dataTimeout()) {
          Socket::disconnect();

          // Dispatch error message and throw an exception so the processing will stop
          m_errorHandler->dispatchError(EC_TIMEOUT);
          FTP_EXCEPTION;
        }
      }
    } else {
      stallTimeout = 0;
    }

    if (m_stateInfo.abortInProgress())
      break;
  }
  
  close(fd);
  sftp_file_close(rf);

  m_stateInfo.enterSocketState(S_IDLE);
  
  emit sigLogUpdate(3, i18n("Transferred 1 byte.", "Transferred %n bytes.", bytes_sent));
  
  return 1;
}

}
#include "sftpsocket.moc"
