/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftpextensions.h"
#include "misc.h"
#include "misc/config.h"

#include <qdir.h>

#include <ksocks.h>
#include <ksockaddr.h>
#include <klocale.h>
#include <kio/job.h>

KFTPCacheExtensions &FTPCache()
{
  static KFTPCacheExtensions cache;
  return cache;
}

/*
                                                          CLASS FTPDIRECTORYITEM
*/

FTPDirectoryItem::FTPDirectoryItem()
{
  // This should NOT be used
}

FTPDirectoryItem::FTPDirectoryItem(FTPEntry ftpEntry)
{
  m_ftpEntry = ftpEntry;
}

QString FTPDirectoryItem::name() const
{
  return m_ftpEntry.name;
}

QString FTPDirectoryItem::owner() const
{
  return m_ftpEntry.owner;
}

QString FTPDirectoryItem::group() const
{
  return m_ftpEntry.group;
}

QString FTPDirectoryItem::link() const
{
  return m_ftpEntry.link;
}

QString FTPDirectoryItem::permissions() const
{
  return m_ftpEntry.permissions;
}

filesize_t FTPDirectoryItem::size() const
{
  return m_ftpEntry.size;
}

char FTPDirectoryItem::type() const
{
  return m_ftpEntry.type;
}

time_t FTPDirectoryItem::date() const
{
  return m_ftpEntry.date;
}

QString FTPDirectoryItem::dateAsString()
{
  QDateTime dt;
  dt.setTime_t(m_ftpEntry.date);

  return KGlobal::locale()->formatDateTime(dt);
}

bool FTPDirectoryItem::operator<(const FTPDirectoryItem &item) const
{
  int p1 = KFTPCore::Config::self()->getFilePriority(name());
  int p2 = KFTPCore::Config::self()->getFilePriority(item.name());
  
  if (type() != item.type())
    return KFTPCore::Config::queueDirsFirst() ? type() == 'd' : type() != 'd';
  
  if (p1 == p2 || !KFTPCore::Config::enablePrioList())
    return name() < item.name();
    
  if (p1 == PRIO_NOT_FOUND)
    return p2 < 0 ? true : false;
  else if (p2 == PRIO_NOT_FOUND)
    return p1 < 0 ? false : true;
  
  return p1 > p2;
}

QDataStream &operator<< (QDataStream &s, const FTPDirectoryItem &a)
{
  s << a.name();
  s << a.owner();
  s << a.group();
  s << a.link();
  s << a.permissions();
  s << a.size();
  s << (int) a.type();
  s << a.date();

  return s;
}

QDataStream &operator>> (QDataStream &s, FTPDirectoryItem &a)
{
  int type;

  s >> a.m_ftpEntry.name;
  s >> a.m_ftpEntry.owner;
  s >> a.m_ftpEntry.group;
  s >> a.m_ftpEntry.link;
  s >> a.m_ftpEntry.permissions;
  s >> a.m_ftpEntry.size;
  s >> type;
  s >> a.m_ftpEntry.date;

  a.m_ftpEntry.type = (char) type;

  return s;
}

/*
                                                          CLASS FTPCACHEITEM
*/
FTPCacheItem::FTPCacheItem()
{
}

FTPCacheItem::FTPCacheItem(KURL url, FTPDirList dirList)
{
  m_url = url;
  m_dirList = dirList;
}

QDataStream &operator<< (QDataStream &s, const FTPCacheItem &a)
{
  // Encode (base64) the password, because it is in cleartext
  KURL tmp = a.m_url;
  if (tmp.hasPass()) {
    tmp.setPass(KFTPGrabberBase::encodePassword(a.m_url.pass()));
  }
  
  s << tmp;
  s << a.m_age;
  s << a.m_dirList;

  return s;
}

QDataStream &operator>> (QDataStream &s, FTPCacheItem &a)
{
  s >> a.m_url;
  s >> a.m_age;
  s >> a.m_dirList;
 
  // Decode (base64) the password, because it was encoded above 
  if (a.m_url.hasPass()) {
    a.m_url.setPass(KFTPGrabberBase::decodePassword(a.m_url.pass()));
  }

  return s;
}

/*
                                                        CLASS KFTPCACHEEXTENSIONS
*/
KFTPCacheExtensions::KFTPCacheExtensions()
{
  m_listCache.clear();
}

KFTPCacheExtensions::~KFTPCacheExtensions()
{
}

void KFTPCacheExtensions::listCache(KURL url, FTPDirList list)
{
  /* If it already exists, remove it first */
  listRemoveFromCache(url);

  /* Add the list to our cache */
  FTPCacheItem tmp(url, list);
  tmp.m_age = time(0L);

  m_listCache.append(tmp);
}

FTPCacheItem *KFTPCacheExtensions::listGetFromCache(KURL url)
{
  /* Go trough all cached items and */
rescan:
  FTPListCache::iterator end( m_listCache.end() );
  for(FTPListCache::iterator i( m_listCache.begin() );i != end; ++i) {
    // Delete entries that are too old
    // FIXME make it read the cache duration from the KConfig - now set to 24h
    if (time(0L) - (*i).m_age > 82800) {
      m_listCache.remove(i);
      
      // List has changed, we must start from the beginning
      goto rescan;
    }

    if ((*i).m_url == url) {
      static FTPCacheItem item;
      item = (*i);
      return &item;
    }
  }

  return 0L;
}

FTPDirectoryItem *KFTPCacheExtensions::listGetFileFromCache(KURL url)
{
  /* Find the parent cached item */
  FTPListCache::iterator listCacheEnd( m_listCache.end() );
  for(FTPListCache::iterator i( m_listCache.begin() );i != listCacheEnd; ++i) {
    if ((*i).m_url == url.upURL()) {
      FTPCacheItem item = (*i);
      
      FTPDirList::iterator dirListEnd( item.m_dirList.end() );
      for(FTPDirList::iterator j( item.m_dirList.begin() );j != dirListEnd; ++j) {
        if ((*j).name() == url.fileName()) {
          // We have found the fileitem, return it
          static FTPDirectoryItem f_item;
          f_item = (*j);
          return &f_item;
        }
      }
    }
  }

  return 0L;
}

void KFTPCacheExtensions::listRemoveFromCache(KURL url)
{
  FTPListCache::iterator end( m_listCache.end() );
  for(FTPListCache::iterator i( m_listCache.begin() );i != end; ++i) {
    if ((*i).m_url == url) {
      m_listCache.remove(i);
      return;
    }
  }
}

void KFTPCacheExtensions::cacheSaveToFile(const QString &filename)
{
  QFile f(filename);
  f.open(IO_WriteOnly);
  QDataStream stream(&f);

  // Write the header
  stream << (Q_UINT32) 0xCAC83; // format id
  stream << (Q_UINT32) 0x01;    // cache format version

  // Write the cache
  stream << m_listCache;

  f.close();
}

void KFTPCacheExtensions::cacheLoadFromFile(const QString &filename)
{
  QFile f(filename);
  if (!f.open(IO_ReadOnly)) {
    // First run, huh - so there is no cache yet
    return;
  }

  QDataStream stream(&f);

  // Read the header
  Q_UINT32 format_id;
  stream >> format_id;
  if (format_id != 0xCAC83) {
    // Invalid cachefile format!
    qDebug("[WARNING] Invalid cache file format!");
    return;
  }

  Q_UINT32 version_id;
  stream >> version_id;
  if (version_id != 0x01) {
    // Incompatible version
    qDebug("[WARNING] Incompatible cache format version!");
    return;
  }

  // Now read from stream to the cache
  stream >> m_listCache;

  f.close();
}

