/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPNETWORKFTPSOCKET_H
#define KFTPNETWORKFTPSOCKET_H

#include <sys/time.h>
#include <time.h>

#include <kssl.h>
#include <ksslcertificate.h>

#include <qmutex.h>

namespace KFTPNetwork {

class Socket;

/**
  * This class represents the FTP socket implementation.
  *
  * @author Jernej Kos
  */
class FtpSocket : public Socket
{
Q_OBJECT
public:
    FtpSocket(QObject *parent);
    
    FTPDirList *dirList(const KURL &url, bool ignoreCache = false);
    
    void get(KURL source, KURL destination);
    void put(const KURL &source, const KURL &destination);
    void remove(const KURL &url);
    void rename(const KURL &source, const KURL &destination);
    void chmod(const KURL &url, int mode);
    void mkdir(const KURL &url);
    void fxpTransfer(const KURL &source, const KURL &destination, Socket *client);
    QString getDefaultDir() { return m_defDir; }
    QString getLastDir() { return m_lastDir; }
    
    void stat(const KURL &url);
    
    bool isEncrypted() { return getConfig("using_ssl") == 1; }
    int getFeatures();
    void initConfig();
    
    void SSLSetClientCert(KSSLPKCS12 *cert);
    KSSL *SSLGetObject();
protected:
    int protoConnect();
    void protoDisconnect();
    
    int protoLogin();
    bool protoDirList(const KURL &url);
    int protoCwd(const QString &dir) { return ftpCwdCheck(dir); }
    int protoRmdir(const QString &dir) { return ftpRmdir(dir); }
    int protoDelete(const QString &path) { return ftpDelete(path); }
    void protoSendRawCommand(const QString &command) { ftpSendCommand(command); }
private:
    struct sockaddr_in m_socketAddress;
    int m_control;
    int m_data;
    int m_dataConn;
    
    /* Mutexes */
    QMutex m_connMutex;
    
    /* Directory navigation */
    QString m_defDir;
    QString m_lastDir;
    
    /* SSL variables */
    KSSL *m_controlSSL;
    KSSL *m_dataSSL;
    
    // Client certificate to be used before auth
    KSSLPKCS12 *m_clientCert;
    bool m_sslLastResult;

    /* Buffer */
    filesize_t m_lastSize;
    QString m_readBuf;
    QString m_lastMultiResp;
    
    QString m_dataReadBuf;
    
    /*** SSL methods ***/
    KSSL *SSLNewData(int fd);
    KSSL *SSLNewControl(int fd);
    
    void SSLClose();
    
    /*** Low-level FTP related methods ***/
    bool ftpCanRead();
    int ftpSendCommand(const QString &command, bool doRead = true);
    int ftpReadLine(char *buf, int max);
    char ftpReadResponse();
    bool ftpIsRespCode(const QString &code);
    void ftpChangeTLSDataProt(char prot);
    void ftpCloseDataSockets();
    
    /*** Connection type methods ***/
    bool ftpPASVConnection();
    bool ftpEPSVConnection();
    bool ftpEPRTConnection();
    bool ftpActiveConnection();
    bool ftpInitDataConnection();
    
    /*** Data connection methods ***/
    ssize_t ftpDataRead(void *buf, long size);
    ssize_t ftpDataWrite(void *buf, long size);
    int ftpDataReadLine(char *buf, int max);
    int ftpInitDataSocket(bool noTLS = false);
    bool ftpDataCommand(const QString &command, const QString &path, char mode, bool noTLS = false, filesize_t offset = 0);
    bool ftpCloseDataCommand();
    int ftpPrepareTLS(int fd);
    
    /*** Directory listing methods ***/
    FTPEntry ftpStat(const KURL &url);
    bool ftpDirChange(const QString &path, const QString &params = QString::null);
    bool ftpDirClose();
    bool ftpReadList(FTPEntry &entry);
    
    /*** FTP protocol command implementation ***/
    int ftpCwd(const QString &dir);
    int ftpCwdCheck(const QString &dir);
    int ftpSize(const QString &file, char mode);
    int ftpGet(const QString &source, const QString &destination);
    int ftpPut(const QString &source, const QString &destination);
    int ftpRename(const QString &source, const QString &destination);
    int ftpDelete(const QString &path);
    int ftpChmod(int mode, const QString &path);
    int ftpMkdir(const QString &dir);
    int ftpRmdir(const QString &dir);
    
    /*** FXP related methods ***/
    void ftpAbortFxp(FtpSocket *client);
    int ftpFxpTransfer(const QString &source, const QString &destination, FtpSocket *client);
    
    /*** Auxilary methods not FTP related ***/
    bool ftpCheckCRC(const QString &remote, const QString &local);
};

}

#endif
