/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#include "errorhandler.h"
#include "socket.h"

#define HANDLED_ERR(x) (x > -50)

namespace KFTPNetwork {

Error::Error()
{
}

Error::Error(ErrorHandler *handler, ErrorCode code, ErrorData data)
  : m_handler(handler), m_code(code), m_data(data), m_returnVal(0)
{
}

ErrorHandler::ErrorHandler(Socket *socket)
  : QObject(socket), m_socket(socket), m_lastId(0), m_enabled(true)
{
  // Get notified when socket disconnects
  connect(socket, SIGNAL(sigDisconnectDone()), this, SLOT(slotSocketDisconnected()));
}

ErrorHandler::~ErrorHandler()
{
}

long ErrorHandler::handlingError(Error error)
{
  // Add the error to the mapping if the error is unhandled
  if (!HANDLED_ERR(error.code())) {
    long id = m_lastId++;
    m_errorMap[id] = error;
    
    return id;
  }
  
  return -1;
}

void ErrorHandler::handlerReturnCode(long id, int code)
{
  m_errorMap[id].m_returnVal = code;
}

void ErrorHandler::abandonHandler(long id)
{
  // Remove the error from the mapping
  m_errorMap.remove(id);
}

bool ErrorHandler::dispatchError(ErrorCode code, ErrorData data)
{
  if (!m_enabled)
    return false;
    
  // Checking only makes sense for "unhandled errors"
  if (!HANDLED_ERR(code)) {
    // Check if this error has already been handled
    QMap<long, Error>::Iterator i;
    for (i = m_errorMap.begin(); i != m_errorMap.end(); ++i) {
      if (i.data().code() == code && i.data().data() == data) {
        return false;
      }
    }
  }
  
  // Execution has been interrupted, so reset the socket's state
  m_socket->m_stateInfo.resetState();

  // The error isn't being handled so we emit the error signal and return
  // true, so the action will abort.
  emit m_socket->sigError(Error(this, code, data));

  return true;
}

int ErrorHandler::returnCode(ErrorCode code, ErrorData data)
{
  if (!HANDLED_ERR(code)) {
    // Check if this error has already been handled
    QMap<long, Error>::Iterator i;
    for (i = m_errorMap.begin(); i != m_errorMap.end(); ++i) {
      if (i.data().code() == code && i.data().data() == data) {
        return i.data().returnVal();
      }
    }
  }
  
  return 0;
}

void ErrorHandler::errorDone(ErrorCode code, ErrorData data)
{
  if (!HANDLED_ERR(code)) {
    // Check if this error has already been handled
    QMap<long, Error>::Iterator i;
    for (i = m_errorMap.begin(); i != m_errorMap.end(); ++i) {
      if (i.data().code() == code && i.data().data() == data) {
        // Found it. We silently remove this entry (since it is now invalid)
        m_errorMap.remove(i);
        return;
      }
    }
  }
}

void ErrorHandler::slotSocketDisconnected()
{
  // Socket has been disconnected, so all error handlers are invalid
  m_errorMap.clear();
}

}

#include "errorhandler.moc"
