/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Handle communications with flickr.com
 *
 * Copyright 2005,2007 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#ifndef FLICKRCOMM_H
#define FLICKRCOMM_H

#include <QMap>
#include <QString>
#include <QObject>
#include <QBuffer>
#include <QSqlRecord>
#include <QByteArray>
#include <QStringList>

#include <kjob.h>
#include <kio/job.h>

class KMD5;
class QFile;
class Photo;
class QTextStream;


/**
 * This class provides the necessary inputs and signals to allow for easy
 * communication with flickr.com via http requests.
 * @short Provides flickr.com communications.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class FlickrComm : public QObject
{
Q_OBJECT

private:
typedef QMap<QString,QString> ArgMap;

public:
typedef QMap<QString,QString> SetData;
typedef QMap<QString,QString> LicenseData;

public:
    FlickrComm(QObject *parent = 0);
    ~FlickrComm();

    void setUserToken(const QString &);

    void doWebAuthentication(const QString &frob);

    void sendFROBRequest();
    void abortCurrentRequest();
    void sendUpStatusRequest(const QString &);
    void sendTokenRequest(const QString &frob);
    void sendPhotosetsRequest(const QString &token, const QString &);

    /**
     * Request a list of the available licenses and their IDs.
     */
    void sendLicensesRequest();

    void sendTagsRequest(const QString &token, const QString &);

    /** Sends a photo to flickr.
     *  @param The user's auth token
     *  @param photo Pointer fo Photo to send
     *  @return pointer to the KIO::TransferJob handling the transfer, or 0 on failure
     */
    KIO::TransferJob* sendPhoto(const QString &token, const QSqlRecord& rec);

    void createPhotoset(const QString &, const QString &, const QString &);
    void addPhoto2Photoset(const QString &, const QString &, const QString &);

    /**
     * Set the license for a photograph.
     * @param token The users authentication token
     * @param license The license string
     * @param id The photo's ID on flickr
     */
    void setPhotoLicense(const QString& token, const QString& license, const QString& photoID);

signals:
    void commError(const QString &);
    void returnedFrob(const QString &);
    void returnedTags(const QStringList &);
    void returnedUploadedOK(const QString &);
    void returnedUploadStatus(const QString &);
    void returnedPhotosets(const QStringList &, const QString &);
    void returnedToken(const QString &, const QString &, const QString &);

    /**
     * Emit after we have received a response to our licenses request.
     * @param licenses List of license strings in the order of their IDs.
     */
    void returnedLicenses(const QStringList& licenses);

private:
    QString generateMD5(const ArgMap &);
    QString assembleArgs(const ArgMap &);
    KIO::TransferJob* sendRequest(ArgMap &);
    QString validateHTTPResponse(const QString &);

    void handleFrobResponse(const QString &);
    void handleTagsResponse(const QString &);
    void handleTokenResponse(const QString &);
    void handleStatusResponse(const QString &);
    void handleUploadResponse(const QString &);
    void handlePhotosetResponse(const QString &);
    void hanldeCreatePhotosetResponse(const QString &);

    /**
     * Handle the response from a licenses request.
     * @param str The response string
     */
    void handleLicensesResponse(const QString& str);

    /** Writes a photo that has been manipulated by size or rotation
     *  ensuring that the EXIF data provided is re-associated with
     *  the photo.
     *  @param out Output buffer being used to write the photo to
     *  @param in Image
     *  @param exif The EXIF data to put with the image
     */
    void writePhotoWithEXIF(QBuffer &out, QImage &in, QByteArray &exif);

    QString m_APIKey;
    KMD5 *m_MD5Context;
    QByteArray m_publicKey;
    QString m_userToken;
    SetData m_photoSets;
    LicenseData m_licenseIDs;

    enum ResponseType
    {
        NO_REQ,
        FROB_REQ,
        TOKEN_REQ,
        AUTH_REQ,
        TAGS_REQ,
        STATUS_REQ,
        PHOTOSET_REQ,
        ADD_PHOTO2SET_REQ,
        CREATE_PHOTOSET_REQ,
        LICENSES_REQ,
        SET_LICENSE_REQ,
        FILE_UPLOAD
    };
    QMap<KIO::TransferJob*,ResponseType> m_requests;
    QMap<KIO::TransferJob*,QString> m_incomingData;

private slots:
    void jobResult(KJob *job);
    void jobData(KIO::Job *job, const QByteArray &data);
};

#endif
