/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : The main widget, show photo properties
 * 
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#include <kdialog.h>
#include <klocale.h>

#include <qlabel.h>
#include <iostream>

#include "kflickrwidget.h"
#include "photolistview.h"
#include "authcomplete.h"
#include "photopropsdlg.h"
#include "authquestiondlg.h"

#include <krun.h>
#include <qvbox.h>
#include <qlabel.h>
#include <qframe.h>
#include <qevent.h>
#include <qlayout.h>
#include <kconfig.h>
#include <qstring.h>
#include <qheader.h>
#include <kglobal.h>
#include <kguiitem.h>
#include <qvariant.h>
#include <qcombobox.h>
#include <kprogress.h>
#include <qvaluelist.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <qscrollview.h>
#include <qdragobject.h>
#include <qpushbutton.h>
#include <kpushbutton.h>
#include <qmessagebox.h>
#include <kapplication.h>

kflickrWidget::kflickrWidget(QWidget* parent, const char* name, WFlags fl)
        : QWidget(parent,name,fl),
    m_currentUserIndex(-1),
    m_progressDlg(0L),
    m_uploadInProgress(false)
{
    QValueList<int> widths;
    
    // setup to retrieve saved settings
    KConfig *config = kapp->config();
   
    // config settings group 
    config->setGroup("KFlickrWidget");
    
    // main layout manager
    QVBoxLayout* kflickrwidgetLayout = new QVBoxLayout(this, 11, 6); 
    
    // Users and Authentication widgets frame
    QFrame* userFrame = new QFrame(this);
    userFrame->setFrameShadow(QFrame::Raised);
    userFrame->setFrameShape(QFrame::StyledPanel);
    QGridLayout* userLayout = new QGridLayout(userFrame, 1, 1, 11, 6);
    kflickrwidgetLayout->addWidget(userFrame);
    
    // The text labels
    userLayout->addWidget(new QLabel(i18n("User:"), userFrame), 0, 0);
    userLayout->addWidget(new QLabel(i18n("Photoset:"), userFrame), 1, 0);    

    // The user and photoset combo boxes    
    m_users = new QComboBox(userFrame);
    m_users->setMinimumSize(QSize(300, 26));
    m_users->insertStringList(config->readListEntry("users", QStringList()));
    userLayout->addWidget(m_users, 0, 1);
    
    m_photosets = new QComboBox(userFrame);
    m_photosets->setEditable(true);
    m_photosets->setMinimumSize(QSize(300, 26));
    userLayout->addWidget(m_photosets, 1, 1);    
    
    // Some Spacers
    userLayout->addItem(new QSpacerItem(280, 30, QSizePolicy::Expanding, QSizePolicy::Minimum), 0, 2);    
    userLayout->addItem(new QSpacerItem(280, 30, QSizePolicy::Expanding, QSizePolicy::Minimum), 1, 2);
    
    // New user button and another spacer
    m_authNew = new QPushButton(i18n("Authenticate New User..."), userFrame);
    userLayout->addWidget(m_authNew, 0, 3);
    userLayout->addItem(new QSpacerItem(180, 30, QSizePolicy::Expanding, QSizePolicy::Minimum), 1, 3);
   
    // User authorization tokens
    m_userNSIDs= config->readListEntry( "user_nsids", QStringList() );
    m_userTokens = config->readListEntry( "user_tokens", QStringList() );
    
    // Photo list
    widths = config->readIntListEntry( "column_widths" );
    if( !widths.size() )
    {
        // default column widths
        widths.append( 175 );
        widths += 110;
        widths += 150;
        widths += 100;
        widths += 225;
        widths += 225;
    }
    
    // initial list setup
    m_photolist = new PhotoListView( this, "m_photolist" );    
    
    m_photolist->addColumn( i18n("Photo"), widths[0] );
    m_photolist->addColumn( i18n("Public"), widths[1] );
    m_photolist->addColumn( i18n("Title"), widths[2] );
    m_photolist->addColumn( i18n("Size"), widths[3] );
    m_photolist->addColumn( i18n("Tags"), widths[4] );
    m_photolist->addColumn( i18n("Description"), widths[5] );
   
    m_photolist->header()->setResizeEnabled( false, 0 );
    m_photolist->setAllColumnsShowFocus( TRUE );
    m_photolist->setAcceptDrops( TRUE );
    m_photolist->setDropVisualizer( TRUE );
    m_photolist->setSelectionMode( QListView::Extended );
    
    kflickrwidgetLayout->addWidget( m_photolist );
    
    // button frame
    QFrame* btnFrame = new QFrame( this, "btnFrame");
    btnFrame->setFrameShape(QFrame::StyledPanel);
    btnFrame->setFrameShadow(QFrame::Raised);
    QHBoxLayout* btnFrameLayout = new QHBoxLayout(btnFrame, 11, 6, "btnFrameLayout"); 

    // add button
    m_addBtn = new KPushButton(KGuiItem(i18n("&Add"), "add"), btnFrame);
    btnFrameLayout->addWidget(m_addBtn);

    // remove button
    m_removeBtn = new KPushButton(KGuiItem(i18n("&Remove"), "remove"), btnFrame);
    btnFrameLayout->addWidget(m_removeBtn);

    // edit button
    m_editBtn = new KPushButton(KGuiItem(i18n("&Edit"), "edit"), btnFrame);
    btnFrameLayout->addWidget(m_editBtn);
    
    QSpacerItem* spacer = new QSpacerItem(320, 30, QSizePolicy::Expanding, QSizePolicy::Minimum);
    btnFrameLayout->addItem(spacer);

    // upload button
    m_uploadBtn = new KPushButton(KGuiItem(i18n("&Upload"), "up"), btnFrame);
    btnFrameLayout->addWidget(m_uploadBtn);
    
    kflickrwidgetLayout->addWidget(btnFrame);
    
    // widget signals and slots connections
    connect(m_addBtn, SIGNAL(clicked()), this, SLOT(addPhotos()));
    connect(m_removeBtn, SIGNAL(clicked()), this, SLOT(removePhotos()));
    connect(m_uploadBtn, SIGNAL(clicked()), this, SLOT(uploadPhotos()));
    connect(m_authNew, SIGNAL(clicked()), this, SLOT(authorizeNewUser()));
    connect(m_editBtn, SIGNAL(clicked()), this, SLOT(onEditSelected()));
    connect(this, SIGNAL(enableEdit(bool)), m_editBtn, SLOT(setEnabled(bool)));
    connect(this, SIGNAL(enableRemove(bool)), m_removeBtn, SLOT(setEnabled(bool)));
    connect(this, SIGNAL(enableUpload(bool)), m_uploadBtn, SLOT(setEnabled(bool)));
    connect(m_photolist, SIGNAL(selectionChanged()), this, SLOT(updateActionStates()));
    connect(m_users, SIGNAL(activated(const QString &)), this, SLOT(setActiveUser(const QString &)));
    connect(m_photolist, SIGNAL(dropped(QDropEvent*,QListViewItem*)),
            this, SLOT(dropSlot(QDropEvent*,QListViewItem*)) );
                
    // Communications signals and slots connections
    connect(&m_comm, SIGNAL(commError(const QString &)), this, SLOT(handleCommError(const QString &)));
    connect(&m_comm, SIGNAL(returnedFrob(const QString &)), this, SLOT(doUserAuthentication(const QString &)));
    connect(&m_comm, SIGNAL(returnedToken(const QString &, const QString &, const QString &)),
            this, SLOT(addUser(const QString &, const QString &, const QString &)));
    connect(&m_comm, SIGNAL(returnedTags(const QStringList &)),
            this, SLOT(setPersistentTags(const QStringList &)));
    connect(&m_comm, SIGNAL(returnedPhotosets(const QStringList &, const QString &)),
            this, SLOT(setPhotosets(const QStringList &, const QString &)));
    connect(&m_comm, SIGNAL(returnedUploadStatus(const QString &)),
            parent,SLOT(newBandwidthValue(const QString &)));
    connect(&m_comm, SIGNAL(returnedUploadedOK(const QString &)), this, SLOT(photoUploadedOK(const QString &)));
                
    // Finally set the active user if one already authorized
    if( config->readEntry( "current_user", QString( "" ) ) != "" )
        setActiveUser( config->readEntry( "current_user" ) );
}

kflickrWidget::~kflickrWidget()
{
    QStringList users;
    QValueList<int> widths;
    KConfig *config = kapp->config();
    QHeader *hdr = m_photolist->header();
    
    // set the group
    config->setGroup("KFlickrWidget");
    
    // save list column widths
    for( int i = 0; i < hdr->count(); ++i )
        widths.append( hdr->sectionSize( i ) );
    config->writeEntry( "column_widths", widths );
    
    // save users
    for( int i = 0; i < m_users->count(); ++i )
        users += m_users->text( i );
    config->writeEntry( "users", users );
    
    // save user tokens
    config->writeEntry( "user_nsids", m_userNSIDs );
    config->writeEntry( "user_tokens", m_userTokens );
    
    // save current user
    config->writeEntry( "current_user", m_users->currentText() );
    
    // delete stuff
    if( m_progressDlg != 0L )
        delete m_progressDlg;
}

void kflickrWidget::addPhotos()
{
    KURL::List urls;
        
    // allow user to select some jpeg files to add
    urls =  KFileDialog::getOpenURLs( ":OpenPhoto", "*.jpg *.png *.gif|Photo Files",
                                       this, i18n("Select Image(s)") );
    
    // ensure at least one selection
    if( !urls.empty() )
    {
        KURL::List::iterator it;
        
        // add each selection to the photo list
        for( it = urls.begin(); it != urls.end(); ++it )
        {
            PhotoListViewItem *photo = new PhotoListViewItem( m_photolist, (*it) );
            
            // use the image filename as the default title
            photo->setTitle( (*it).filename() );
        }
    }
    
    // get the previews
    m_photolist->getPreviews(urls);
    
    // update the button states
    updateActionStates();
}


/*!
    \fn kflickrWidget::uploadPhotos()
 */
void kflickrWidget::uploadPhotos()
{
    if( m_photolist->childCount() > 0 )
    {
        m_uploadInProgress = true;
        
        // display the progress dialog
        showUploadProgress(m_photolist->childCount());
        
        // upload the first photos one at a time
        uploadNextPhoto();
    }
}


/*!
    \fn kflickrWidget::uploadNextPhoto()
 */
void kflickrWidget::uploadNextPhoto()
{
    PhotoListViewItem *photo;
    
    // get last photo in list
    if( (photo = m_photolist->lastPhoto()) != 0L )
    {
        // upload the next photo
        m_comm.sendPhoto(m_userTokens[m_currentUserIndex], photo);
        updateUploadProgress(i18n("Sending Photo - %1").arg(photo->url().filename()));
    }
    else
    {
        // The progress dialog will sometime pop back up if
        // it is left with steps to do, so update to make %100
        updateUploadProgress(i18n("Done"));
        m_uploadInProgress = false;
        hideUploadProgress();
    }
}


/*!
    \fn kflickrWidget::photoUploadedOK(const QString &photoID)
 */
void kflickrWidget::photoUploadedOK(const QString &photoID)
{
    PhotoListViewItem *photo;
        
    // update unused bandwidth
    m_comm.sendUpStatusRequest(m_userTokens[m_currentUserIndex]);    
    
    // remove the picture from the list
    if( (photo = m_photolist->lastPhoto()) != 0L )
    {
        QString photoset;
        
        // Add photo to a photoset
        if( (photoset = m_photosets->currentText()) != i18n("<photostream only>") )
        {
            m_comm.addPhoto2Photoset(m_userTokens[m_currentUserIndex], photoset, photoID);
        }
        
        delete photo;
    }
    
    // upload the next one
    uploadNextPhoto();
}


/*!
    \fn kflickrWidget::cancelUpload()
 */
void kflickrWidget::cancelUpload()
{
    m_uploadInProgress = false;
    m_comm.abortCurrentRequest();
    hideUploadProgress();
}


/*!
    \fn kflickrWidget::slotAuthorizeNewUser()
 */
void kflickrWidget::authorizeNewUser()
{
    AuthQuestionDlg dlg( this );
    
    // inform the user and check if we should continue
    if( dlg.exec() != QDialog::Accepted )
    {
        return;
    }
    
    // Get our FROB from flickr.com
    m_comm.sendFROBRequest();
}


/*!
    \fn kflickrWidget::doUserAuthentication( const QString &frob )
 */
void kflickrWidget::doUserAuthentication( const QString &frob )
{
    AuthCompleteDlg dlg( this );
    
    // Open browser etc... for web authentication
    m_comm.doWebAuthentication( frob );
    
    // Wait for user to login and such at Flickr.com
    if( dlg.exec() != QDialog::Accepted )
    {
        return;
    }
    
    // Request our TOKEN from flickr.com
    m_comm.sendTokenRequest( frob );
}


/*!
    \fn kflickrWidget::addUser( const QString &name, const QString &token )
 */
void kflickrWidget::addUser( const QString &name, const QString &token, const QString &nsid )
{
    int i;
    
    // check if the user already exists
    for( i = 0; i < m_users->count(); ++i )
    {
        if( name == m_users->text( i ) )
        {
            break;
        }
    }
    
    // brand new user, existing users retain their current token
    if( i == m_users->count() )
    {
        m_userNSIDs.append( nsid );
        m_users->insertItem( name );
        m_userTokens.append( token );        
    }
    
    // make the active user
    setActiveUser( name );
    
    // Notify user
    QMessageBox::information( this, i18n("New User"),
                i18n("A new user '%1' has been added successfully.").arg(name) );
}


/*!
    \fn kflickrWidget::setActiveUser( const QString & )
 */
void kflickrWidget::setActiveUser( const QString &name )
{
    // update combobox
    if( m_users->currentText() != name )
        m_users->setCurrentText( name );
        
    // update current user index, must always do this here
    m_currentUserIndex = m_users->currentItem();
    
    // request users tags
    m_comm.sendTagsRequest(m_userTokens[m_currentUserIndex], m_userNSIDs[m_currentUserIndex]);
    
    // request user photosets
    m_comm.sendPhotosetsRequest(m_userNSIDs[m_currentUserIndex]);
    
    // update unused bandwidth
    m_comm.sendUpStatusRequest(m_userTokens[m_currentUserIndex]);
    
    
    // update the button states
    updateActionStates();
}

void kflickrWidget::removePhotos()
{
    // remove selected photos
    m_photolist->removeSelected();
    
    // update the button states
    updateActionStates();
}


/*!
    \fn kflickrWidget::dropSlot( QDropEvent *e, QListViewItem *after )
 */
void kflickrWidget::dropSlot( QDropEvent *e, QListViewItem *after )
{
    QString text;
    KURL::List urls;
    QStringList parts;
    
    // make sure the drag is of the right type
    if ( QTextDrag::decode( e, text ) )
    {
        // there might be more than one file, split it up
        parts = QStringList::split( "\n", text );
        
        for ( QStringList::Iterator it = parts.begin(); it != parts.end(); ++it )
        {
            // turn into a URL
            KURL url( (*it).stripWhiteSpace() );
            
            // ensure it is a jpeg type file
            if( (url.protocol() == "file") &&
                ( (url.filename()).endsWith( ".jpg", false ) ||
                  (url.filename()).endsWith( ".png", false ) ||
                  (url.filename()).endsWith( ".gif", false ) ) )
            {
                // add it to the photo list
                PhotoListViewItem *photo = new PhotoListViewItem( m_photolist, url, after );
                
                // use the filename as the default title
                photo->setTitle( url.filename() );
                
                // add it for preview
                urls.append(url);
             }
        }
    }
    
    // get the previews
    if(!urls.isEmpty())
        m_photolist->getPreviews(urls);
        
    // update the button states
    updateActionStates();
}


/*!
    \fn kflickrWidget::onEditSelected()
 */
void kflickrWidget::onEditSelected()
{
    if( m_photolist->numSelected() == 1 )
        editSelectedItem();
    else
        editSelectedItems();
}


/*!
    \fn kflickrWidget::editSelectedItems()
 */
void kflickrWidget::editSelectedItems()
{
    PhotoPropsDlg* dlg = new PhotoPropsDlg(this);    
    
    // Setup the dialog
    dlg->editPhotoBatch();
    dlg->setAvailableTags(m_persistTags);
    
    // handle an "OK" response from the dialog
    if( dlg->exec() == QDialog::Accepted )
    {
        PhotoListViewItem *photo;
                
        // starting with the first selected photo,
        // edit the selected photos attributes
        photo = m_photolist->firstSelectedPhoto();
        while( photo != 0L )
        {
            if(!dlg->title().isEmpty())         photo->setTitle(dlg->title());
            if(!dlg->description().isEmpty())   photo->setDescription(dlg->description());
            if(!dlg->tags().empty())            photo->setTags(dlg->tags());
            if(!dlg->photoSize().isEmpty())     photo->setSize(dlg->photoSize());
            
            // Only update privacy if user has selected a privacy level
            if( dlg->isPublic() || dlg->isFamily() || dlg->isFriends() || dlg->isPrivate() )
            {
                photo->setPublic(dlg->isPublic());
                photo->setFamily(dlg->isFamily());
                photo->setFriends(dlg->isFriends());
            }
            
            // move on to next selected photo
            photo = m_photolist->nextSelectedPhoto(photo);
        }    
        m_persistTags = dlg->availableTags();
    }
    delete dlg;   
}


/*!
    \fn kflickrWidget::editSelectedItem()
 */
void kflickrWidget::editSelectedItem()
{
    PhotoListViewItem *photo;
        
    if( (photo = m_photolist->firstSelectedPhoto()) != 0L )
    {
        PhotoPropsDlg* dlg = new PhotoPropsDlg( this );
        
        
        // setup the dialog
        dlg->editSinglePhoto(photo);
        dlg->setAvailableTags(m_persistTags);
        
        // handle an 'OK' response in the dialog
        if( dlg->exec() == QDialog::Accepted )
        {
            dlg->updateActivePhotoInfo();
            m_persistTags = dlg->availableTags();
        }
        delete dlg;   
    }
}


/*
    \fn kflickrWidget::persistentTags() const
 */
QStringList kflickrWidget::persistentTags() const
{
    return m_persistTags;
}


/*!
    \fn kflickrWidget::setPersistentTags(const QStringList &lst)
 */
void kflickrWidget::setPersistentTags(const QStringList &lst)
{
    m_persistTags = lst;
}


/*!
    \fn kflickrWidget::setPhotosets(const QStringList &titles, const QString &sel)
 */
void kflickrWidget::setPhotosets(const QStringList &titles, const QString &sel)
{
    // update the photosets combobox
    m_photosets->clear();
    m_photosets->insertStringList(titles);
    m_photosets->insertItem(i18n("<photostream only>"), 0);
    
    if( sel == QString::null )
        m_photosets->setCurrentText(i18n("<photostream only>"));
    else
        m_photosets->setCurrentText(sel);
        
}


/*!
    \fn kflickrWidget::hideUploadProgress()
 */
void kflickrWidget::hideUploadProgress()
{
    // hide the progress dialog
    if( m_progressDlg != 0L && m_progressDlg->isVisible() )
    {
        m_progressDlg->hide();
    }
}


/*!
    \fn kflickrWidget::showUploadProgress(int numPhotos)
 */
void kflickrWidget::showUploadProgress(int numPhotos)
{
    // create the progress dialog if it does not yet exist    
    if( m_progressDlg == 0L )
    {
        m_progressDlg = new KProgressDialog(this, "m_progressDlg");    
        m_progressDlg->progressBar()->setTextEnabled(false);
        m_progressDlg->setCaption(i18n("Upload Progress"));
        m_progressDlg->setAutoClose(false);
        m_progressDlg->setModal(true);
            
        // Progress dialog connections
        connect(m_progressDlg, SIGNAL(cancelClicked()), this, SLOT(cancelUpload()));
    }
    
    // tune the progress dialog accordingly
    m_progressDlg->progressBar()->setTotalSteps(numPhotos);
    m_progressDlg->progressBar()->setValue(0);
    m_progressDlg->show();
}


/*!
    \fn kflickrWidget::updateUploadProgress(const QString &str)
 */
void kflickrWidget::updateUploadProgress(const QString &str)
{
    // set text
    m_progressDlg->setLabel(str);
    
    // move progress bar
    if( m_progressDlg->progressBar()->totalSteps() != m_photolist->childCount())
        m_progressDlg->progressBar()->advance(1);
}


/*!
    \fn kflickrWidget::handleCommError(const QString &)
 */
void kflickrWidget::handleCommError(const QString &errstr)
{
    QMessageBox::critical(this, i18n("Error"), errstr);
    
    // Are we in the middle of an upload
    if( m_uploadInProgress )
    {
        // Reset upload things
        m_uploadInProgress = false;
        hideUploadProgress();

        // Inform the user as to what has happenned and how to continue
        QMessageBox::information(this, i18n("Information"),
            i18n("Due to an unexpected communications error your upload has been terminated.") +
            i18n("\nYou may press the \"Upload\" button to retry uploading the unsent photos.") +
            i18n("\nYou may also want to check your flickr site to ensure no partial photos exist."));
    }
}


/*!
    \fn kflickrWidget::updateActionStates()
 */
void kflickrWidget::updateActionStates()
{
    // Upload button enabled if we have a valid user and at least one photo
    if( m_photolist->firstChild() != 0L && m_currentUserIndex != -1 )
    {
        emit enableUpload(true);
    }
    else
    {
        emit enableUpload(false);
    }
    
    // remove and edit only available if at least one photo selected
    if( m_photolist->numSelected() > 0 )
    {
        emit enableRemove(true);
        emit enableEdit(true);
    }
    else
    {
        emit enableRemove(false);
        emit enableEdit(false);
    }
}


#include "kflickrwidget.moc"
