/*
 * itemviews.h
 *
 * Copyright (c) 2007 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#ifndef ITEMVIEWS_H
#define ITEMVIEWS_H

#include <QGraphicsItemAnimation>
#include <QGraphicsRectItem>
#include <QList>
#include <QUrl>
#include <QObject>

struct NewsItem
{
    QString text;
    QString url;
    QString description;
};

class KNewsTicker;

class NewsItemView : public QObject, public QGraphicsRectItem
{
    Q_OBJECT
public:
    NewsItemView( KNewsTicker *parent );

    const QList<NewsItem> &items() const;
    virtual void setItems( const QList<NewsItem> &items );
    virtual void reloadSettings();

Q_SIGNALS:
    void itemActivated( const QString &url );

protected:
    KNewsTicker *newsTicker() { return m_newsTicker; }

private:
    QList<NewsItem> m_items;
    KNewsTicker *m_newsTicker;
};

class HyperlinkItem : public QObject, public QGraphicsSimpleTextItem
{
    Q_OBJECT
public:
    static const int Type = UserType + 1;

    HyperlinkItem( const QString &text, const QUrl &url,
                  QGraphicsItem *parentItem );

    virtual int type() const { return Type; }

    const QUrl &url() const { return m_url; }
    void setUrl( const QUrl &url ) { m_url = url; }

Q_SIGNALS:
    void activated( const QString &url );

protected:
    virtual void hoverEnterEvent( QGraphicsSceneHoverEvent *e );
    virtual void hoverLeaveEvent( QGraphicsSceneHoverEvent *e );

    // empty reimplementation so that we get release events
    virtual void mousePressEvent( QGraphicsSceneMouseEvent * ) { }

    virtual void mouseReleaseEvent( QGraphicsSceneMouseEvent *e );

private:
    QUrl m_url;
};

class SeparatorItem : public QGraphicsSimpleTextItem
{
public:
    static const int Type = UserType + 2;

    SeparatorItem( QGraphicsItem *parentItem );

    virtual int type() const { return Type; }
};

class NewsTickerItem : public QGraphicsItemGroup
{
public:
    static const int Type = UserType + 3;

    NewsTickerItem( const QString &text, const QUrl &url,
                    const QString &description,
                    QGraphicsItem *parentItem );

    virtual int type() const { return Type; }

    const HyperlinkItem *headlineItem() const { return m_headlineItem; }
    HyperlinkItem *headlineItem() { return m_headlineItem; }

    const SeparatorItem *separatorItem() const { return m_separatorItem; }
    SeparatorItem *separatorItem() { return m_separatorItem; }

private:
    HyperlinkItem *m_headlineItem;
    SeparatorItem *m_separatorItem;
};

class ScrollingItemView : public NewsItemView
{
    Q_OBJECT
    using NewsItemView::advance;
public:
    ScrollingItemView( KNewsTicker *parent );

    virtual void setItems( const QList<NewsItem> &items );
    virtual void reloadSettings();

private Q_SLOTS:
    void advance();

private:
    void relayoutItems();
    void addItemToLayout( const NewsItem &item, qreal *xpos, qreal *ypos );
    void configureScrollTimer();

    QTimer *m_scrollTimer;
    QList<NewsTickerItem *> m_graphicsItems;
    qreal m_steppingWidth;
};

class PagingItemView : public NewsItemView
{
    Q_OBJECT
public:
    PagingItemView( KNewsTicker *parent = 0 );
    virtual ~PagingItemView();

    virtual void setItems( const QList<NewsItem> &items );
    virtual void reloadSettings();

private Q_SLOTS:
    void moveItemIn();
    void itemMovedIn();
    void moveItemOut();
    void itemMovedOut();

private:
    void updateMoveAnimations();
    void updateLinkItem();

    QList<NewsItem>::ConstIterator m_currentItem;
    HyperlinkItem *m_linkItem;
    QGraphicsItemAnimation *m_moveInAnimation;
    QGraphicsItemAnimation *m_moveOutAnimation;
    bool m_needToReloadSettings;
};

#endif // !defined(ITEMVIEWS_H)

