 /*
    This file is part of kdepim.

    Copyright (c) 2003 - 2004 Tobias Koenig <tokoe@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KCAL_RESOURCEXMLRPC_H
#define KCAL_RESOURCEXMLRPC_H

#include <QString>
#include <QList>

#include <kurl.h>


#include <kcal/calendarlocal.h>
#include <kcal/incidence.h>
#include <kcal/resourcecached.h>
#include "todostatemapper.h"
#include "egroupware_export.h"

namespace KXMLRPC {
class Server;
}

class Synchronizer;

namespace KCal {

class EGroupwarePrefs;

/**
  This class provides access to php/eGroupware calendar via XML-RPC.
*/
class KCAL_XMLRPC_EXPORT ResourceXMLRPC : public ResourceCached
{
  Q_OBJECT

  public:
    ResourceXMLRPC();
    explicit ResourceXMLRPC( const KConfigGroup &group );
    virtual ~ResourceXMLRPC();

    void readConfig( const KConfigGroup &group );
    void writeConfig( KConfigGroup &group );

    EGroupwarePrefs *prefs() const { return mPrefs; }

    bool isSaving();

    KABC::Lock *lock();

    /**
      Add Event to calendar.
     */
    bool addEvent( Event* event );

    /**
      Deletes an event from this calendar.
     */
    bool deleteEvent( Event* );

    /**
      Retrieves an event on the basis of the unique string ID.
     */
    Event *event( const QString& uid );

    /**
      Return unfiltered list of all events in calendar.
     */
    Event::List rawEvents();

    /**
      Builds and then returns a list of all events that match for the
      date specified. useful for dayView, etc. etc.
     */
    Event::List rawEventsForDate(
      const QDate& date,
      const KDateTime::Spec& timespec = KDateTime::Spec(),
      EventSortField sortField=EventSortUnsorted,
      SortDirection sortDirection=SortDirectionAscending );

    /**
      Get unfiltered events for date \a dt.
     */
    Event::List rawEventsForDate( const KDateTime& dt );

    /**
      Get unfiltered events in a range of dates. If inclusive is set to true,
      only events are returned, which are completely included in the range.
     */
    Event::List rawEvents( const QDate& start, const QDate& end,
                           const KDateTime::Spec& timespec = KDateTime::Spec(),
                           bool inclusive = false );


    /**
      Add a todo to the todolist.
     */
    bool addTodo( Todo* todo );

    /**
      Remove a todo from the todolist.
     */
    bool deleteTodo( Todo* todo );

    /**
      Searches todolist for an event with this unique string identifier,
      returns a pointer or null.
     */
    Todo *todo( const QString& uid );

    /**
      Return list of all todos.
     */
    Todo::List rawTodos();

    /**
      Returns list of todos due on the specified date.
     */
    Todo::List rawTodosForDate( const QDate& date );

    /**
      Add a Journal entry to calendar
     */
    virtual bool addJournal( Journal* journal );

    /**
      Remove journal from the calendar.
     */
    bool deleteJournal( Journal* journal );

    /**
      Return Journals for given date
     */
    virtual Journal::List journals( const QDate& );

    /**
      Return Journal with given UID
     */
    virtual Journal *journal( const QString& uid );

    /**
      Return all alarms, which occur in the given time interval.
     */
    Alarm::List alarms( const KDateTime& from, const KDateTime& to );

    /**
      Return all alarms, which occur before given date.
     */
    Alarm::List alarmsTo( const KDateTime& to );

    /**
      Public because needed in MultiCalendar::load()
     */
    bool doOpen();
    void doClose();

    void dump() const;

    void setTimeZoneId( const QString& ) {}

  protected slots:
    void loginFinished( const QList<QVariant>&, const QVariant& );
    void logoutFinished( const QList<QVariant>&, const QVariant& );

    void listEventsFinished( const QList<QVariant>&, const QVariant& );
    void addEventFinished( const QList<QVariant>&, const QVariant& );
    void updateEventFinished( const QList<QVariant>&, const QVariant& );
    void deleteEventFinished( const QList<QVariant>&, const QVariant& );
    void loadEventCategoriesFinished( const QList<QVariant>&, const QVariant& );

    void listTodosFinished( const QList<QVariant>&, const QVariant& );
    void addTodoFinished( const QList<QVariant>&, const QVariant& );
    void updateTodoFinished( const QList<QVariant>&, const QVariant& );
    void deleteTodoFinished( const QList<QVariant>&, const QVariant& );
    void loadTodoCategoriesFinished( const QList<QVariant>&, const QVariant& );

    void fault( int, const QString&, const QVariant& );

  protected:
    bool doLoad( bool syncCache );
    bool doSave( bool syncCache );

  private slots:
    void reload();

  private:
    void init();
    void initEGroupware();

    void writeEvent( Event*, QMap<QString, QVariant>& );
    void readEvent( const QMap<QString, QVariant>&, Event*, QString& );

    void writeTodo( Todo*, QMap<QString, QVariant>& );
    void readTodo( const QMap<QString, QVariant>&, Todo*, QString& );

    void checkLoadingFinished();

    KXMLRPC::Server *mServer;

    EGroupwarePrefs *mPrefs;

    QString mSessionID;
    QString mKp3;

    QMap<QString, int> mEventCategoryMap;
    QMap<QString, int> mTodoCategoryMap;

    TodoStateMapper mTodoStateMapper;

    Synchronizer *mSynchronizer;

    KABC::Lock *mLock;
    int mLoaded;
};

}

#endif
