/*
	client.cpp - Kopete Oscar Protocol

	Copyright (c) 2004-2005 Matt Rogers <mattr@kde.org>
    Copyright (c) 2007 Roman Jarosz <kedgedev@centrum.cz>

	Based on code Copyright (c) 2004 SuSE Linux AG <http://www.suse.com>
	Based on Iris, Copyright (C) 2003  Justin Karneges

	Kopete (c) 2002-2007 by the Kopete developers <kopete-devel@kde.org>

	*************************************************************************
	*                                                                       *
	* This library is free software; you can redistribute it and/or         *
	* modify it under the terms of the GNU Lesser General Public            *
	* License as published by the Free Software Foundation; either          *
	* version 2 of the License, or (at your option) any later version.      *
	*                                                                       *
	*************************************************************************
*/

#include "client.h"

#include <qtimer.h>
#include <QList>
#include <QByteArray>
#include <qtextcodec.h>
#include <QtNetwork/QTcpSocket>

#include <kdebug.h> //for kDebug()
#include <klocale.h>

#include "filetransfertask.h"
#include "buddyicontask.h"
#include "clientreadytask.h"
#include "connectionhandler.h"
#include "changevisibilitytask.h"
#include "chatnavservicetask.h"
#include "errortask.h"
#include "icquserinfo.h"
#include "icquserinfotask.h"
#include "logintask.h"
#include "connection.h"
#include "messagereceivertask.h"
#include "onlinenotifiertask.h"
#include "oscarclientstream.h"
#include "oscarsettings.h"
#include "oscarutils.h"
#include "ownuserinfotask.h"
#include "profiletask.h"
#include "senddcinfotask.h"
#include "sendmessagetask.h"
#include "serverredirecttask.h"
#include "servicesetuptask.h"
#include "contactmanager.h"
#include "ssimodifytask.h"
#include "ssiauthtask.h"
#include "offlinemessagestask.h"
#include "task.h"
#include "typingnotifytask.h"
#include "userinfotask.h"
#include "usersearchtask.h"
#include "warningtask.h"
#include "chatservicetask.h"
#include "rateclassmanager.h"
#include "icquserinfoupdatetask.h"
#include "icqchangepasswordtask.h"
#include "oscarmessageplugin.h"
#include "xtrazxtraznotify.h"
#include "xtrazxawayservice.h"
#include "closeconnectiontask.h"
#include "icqtlvinforequesttask.h"
#include "icqtlvinfoupdatetask.h"


namespace
{
	class DefaultCodecProvider : public Client::CodecProvider
	{
	public:
		virtual QTextCodec* codecForContact( const QString& ) const
		{
			return QTextCodec::codecForMib( 4 );
		}
		virtual QTextCodec* codecForAccount() const
		{
			return QTextCodec::codecForMib( 4 );
		}
	};

	DefaultCodecProvider defaultCodecProvider;
}

class Client::ClientPrivate
{
public:
	ClientPrivate() {}

	QString host, user, pass;
	uint port;
	int tzoffset;
	bool active;

	enum { StageOne, StageTwo };
	int stage;

	//Protocol specific data
	bool isIcq;
	bool redirectRequested;
	QList<Oscar::WORD> redirectionServices;
    Oscar::WORD currentRedirect;
	QByteArray cookie;
	Oscar::Settings* settings;

	//Tasks
	ErrorTask* errorTask;
	OnlineNotifierTask* onlineNotifier;
	OwnUserInfoTask* ownStatusTask;
	MessageReceiverTask* messageReceiverTask;
	SSIAuthTask* ssiAuthTask;
	ICQUserInfoRequestTask* icqInfoTask;
	ICQTlvInfoRequestTask* icqTlvInfoTask;
	UserInfoTask* userInfoTask;
	TypingNotifyTask * typingNotifyTask;
	SSIModifyTask* ssiModifyTask;
	//Managers
	ContactManager* ssiManager;
	ConnectionHandler connections;

	//Our Userinfo
	UserDetails ourDetails;

    //Infos
    QList<int> exchanges;

	struct Status
	{
		Oscar::DWORD status;
		QString message;     // for away-,DND-message etc., and for Xtraz status
		int xtraz;           // Xtraz status
		QString description; // Xtraz description
		bool sent;
	} status;

	//away messages
	struct AwayMsgRequest
	{
		QString contact;
		ICQStatus contactStatus;
	};
	QList<AwayMsgRequest> awayMsgRequestQueue;
	QTimer* awayMsgRequestTimer;
	CodecProvider* codecProvider;
	
	const Oscar::ClientVersion* version;
	Guid versionCap;
};

Client::Client( QObject* parent )
:QObject( parent )
{
	setObjectName( "oscarclient" );
	m_loginTask = 0L;
	m_loginTaskTwo = 0L;

	d = new ClientPrivate;
	d->tzoffset = 0;
	d->active = false;
	d->isIcq = false; //default to AIM
	d->redirectRequested = false;
    d->currentRedirect = 0;
	d->status.status = 0x0; // default to online
	d->status.xtraz = -1; // default to no Xtraz
	d->status.sent = false;
	d->ssiManager = new ContactManager( this );
	d->settings = new Oscar::Settings();
	d->errorTask = 0L;
	d->onlineNotifier = 0L;
	d->ownStatusTask = 0L;
	d->messageReceiverTask = 0L;
	d->ssiAuthTask = 0L;
	d->icqInfoTask = 0L;
	d->icqTlvInfoTask = 0L;
	d->userInfoTask = 0L;
	d->stage = ClientPrivate::StageOne;
	d->typingNotifyTask = 0L;
	d->ssiModifyTask = 0L;
	d->awayMsgRequestTimer = new QTimer();
	d->codecProvider = &defaultCodecProvider;

	connect( this, SIGNAL( redirectionFinished( Oscar::WORD ) ),
	         this, SLOT( checkRedirectionQueue( Oscar::WORD ) ) );
	connect( d->awayMsgRequestTimer, SIGNAL( timeout() ),
	         this, SLOT( nextICQAwayMessageRequest() ) );
}

Client::~Client()
{

	//delete the connections differently than in deleteConnections()
	//deleteLater() seems to cause destruction order issues
	deleteStaticTasks();
    delete d->settings;
	delete d->ssiManager;
	delete d->awayMsgRequestTimer;
	delete d;
}

Oscar::Settings* Client::clientSettings() const
{
	return d->settings;
}

void Client::connectToServer( Connection *c, const QString& host, quint16 port, bool auth )
{
	d->connections.append( c );
	if ( auth == true )
	{
		m_loginTask = new StageOneLoginTask( c->rootTask() );
		connect( m_loginTask, SIGNAL( finished() ), this, SLOT( lt_loginFinished() ) );
	}

	connect( c, SIGNAL( socketError( int, const QString& ) ), this, SLOT( determineDisconnection( int, const QString& ) ) );
	c->connectToServer( host, port );
}

void Client::start( const QString &host, const uint port, const QString &userId, const QString &pass )
{
	Q_UNUSED( host );
	Q_UNUSED( port );
	d->user = userId;
	d->pass = pass;
	d->stage = ClientPrivate::StageOne;
	d->active = false;
}

void Client::close()
{
	QList<Connection*> cList = d->connections.connections();
	for ( int i = 0; i < cList.size(); i++ )
		(new CloseConnectionTask( cList.at(i)->rootTask() ))->go( Task::AutoDelete );

	d->active = false;
	d->awayMsgRequestTimer->stop();
	d->awayMsgRequestQueue.clear();
	d->connections.clear();
	deleteStaticTasks();

	//don't clear the stored status between stage one and two
	if ( d->stage == ClientPrivate::StageTwo )
	{
		d->status.status = 0x0;
		d->status.xtraz = -1;
		d->status.sent = false;
		d->status.message.clear();
		d->status.description.clear();
	}

    d->exchanges.clear();
    d->redirectRequested = false;
    d->currentRedirect = 0;
    d->redirectionServices.clear();
    d->ssiManager->clear();
}

void Client::setStatus( Oscar::DWORD status, const QString &message, int xtraz, const QString &description )
{
	kDebug(OSCAR_RAW_DEBUG) << "Setting status message to "<< message;

	// remember the values to reply with, when requested
	bool xtrazChanged = (xtraz > -1 || d->status.xtraz != xtraz);
	bool statusInfoChanged = ( !d->status.sent || message != d->status.message || description != d->status.description );
	d->status.status = status;
	d->status.message = message;
	d->status.xtraz = xtraz;
	d->status.description = description;
	d->status.sent = false;

	if ( d->active )
	{
		if ( d->isIcq )
		{
			//the first connection is always the BOS connection
			Connection* c = d->connections.connectionForFamily( 0x0013 );
			if ( !c )
				return; //TODO trigger an error of some sort?

			ChangeVisibilityTask* cvt = new ChangeVisibilityTask( c->rootTask() );
			if ( ( status & 0x0100 ) == 0x0100 )
			{
				kDebug(OSCAR_RAW_DEBUG) << "Setting invisible";
				cvt->setVisible( false );
			}
			else
			{
				kDebug(OSCAR_RAW_DEBUG) << "Setting visible";
				cvt->setVisible( true );
			}
			cvt->go( Task::AutoDelete );
		}
		
		Connection* c = d->connections.connectionForFamily( 0x0002 );
		if ( !c )
			return;

		SendDCInfoTask* sdcit = new SendDCInfoTask( c->rootTask(), status );
		sdcit->go( Task::AutoDelete ); //autodelete

		QString msg;
		// AIM: you're away exactly when your away message isn't empty.
		// can't use QString() as a message either; ProfileTask
		// interprets null as "don't change".
		if ( (status & 0xFF) == 0x00 ) //is status online?
		{
			msg = QString::fromAscii("");
		}
		else
		{
			if ( message.isEmpty() )
				msg = QString::fromAscii(" ");
			else
				msg = message;
		}

		ProfileTask* pt = new ProfileTask( c->rootTask() );
		pt->setAwayMessage( msg );

		if ( d->isIcq && xtrazChanged )
			pt->setXtrazStatus( xtraz );

		pt->go( Task::AutoDelete );

		if ( d->isIcq && statusInfoChanged )
		{
			ICQFullInfo info( false );
			info.statusDescription.set( description.toUtf8() );

			ICQTlvInfoUpdateTask* infoUpdateTask = new ICQTlvInfoUpdateTask( c->rootTask() );
			infoUpdateTask->setInfo( info );
			infoUpdateTask->go( Task::AutoDelete );
		}
		d->status.sent = true;
	}
}

UserDetails Client::ourInfo() const
{
	return d->ourDetails;
}

QString Client::host()
{
	return d->host;
}

int Client::port()
{
	return d->port;
}

ContactManager* Client::ssiManager() const
{
	return d->ssiManager;
}

const Oscar::ClientVersion* Client::version() const
{
	return d->version;
}

Guid Client::versionCap() const
{
	return d->versionCap;
}

// SLOTS //

void Client::streamConnected()
{
	kDebug(OSCAR_RAW_DEBUG) ;
	d->stage = ClientPrivate::StageTwo;
	if ( m_loginTaskTwo )
		m_loginTaskTwo->go();
}

void Client::lt_loginFinished()
{
	/* Check for stage two login first, since we create the stage two
	 * task when we finish stage one
	 */
	if ( d->stage == ClientPrivate::StageTwo )
	{
		//we've finished logging in. start the services setup
		kDebug(OSCAR_RAW_DEBUG) << "stage two done. setting up services";
		initializeStaticTasks();
		ServiceSetupTask* ssTask = new ServiceSetupTask( d->connections.defaultConnection()->rootTask() );
		connect( ssTask, SIGNAL( finished() ), this, SLOT( serviceSetupFinished() ) );
		ssTask->go( Task::AutoDelete ); //fire and forget
		m_loginTaskTwo->deleteLater();
		m_loginTaskTwo = 0;
	}
	else if ( d->stage == ClientPrivate::StageOne )
	{
		kDebug(OSCAR_RAW_DEBUG) << "stage one login done";
		disconnect( m_loginTask, SIGNAL( finished() ), this, SLOT( lt_loginFinished() ) );

		if ( m_loginTask->statusCode() == 0 ) //we can start stage two
		{
			kDebug(OSCAR_RAW_DEBUG) << "no errors from stage one. moving to stage two";

			//cache these values since they'll be deleted when we close the connections (which deletes the tasks)
			d->host = m_loginTask->bosServer();
			d->port = m_loginTask->bosPort().toUInt();
			d->cookie = m_loginTask->loginCookie();
			close();
			QTimer::singleShot( 100, this, SLOT(startStageTwo() ) );
		}
		else
		{
			kDebug(OSCAR_RAW_DEBUG) << "errors reported. not moving to stage two";
			close(); //deletes the connections for us
		}

		m_loginTask->deleteLater();
		m_loginTask = 0;
	}

}

void Client::startStageTwo()
{
	//create a new connection and set it up
	Connection* c = createConnection();
	new CloseConnectionTask( c->rootTask() );

	//create the new login task
	m_loginTaskTwo = new StageTwoLoginTask( c->rootTask() );
	m_loginTaskTwo->setCookie( d->cookie );
	QObject::connect( m_loginTaskTwo, SIGNAL( finished() ), this, SLOT( lt_loginFinished() ) );


	//connect
	QObject::connect( c, SIGNAL( connected() ), this, SLOT( streamConnected() ) );
	connectToServer( c, d->host, d->port, false ) ;

}

void Client::serviceSetupFinished()
{
	d->active = true;

	setStatus( d->status.status, d->status.message, d->status.xtraz, d->status.description );
	d->ownStatusTask->go();

	if ( isIcq() )
	{
		//retrieve offline messages
		Connection* c = d->connections.connectionForFamily( 0x0015 );
		if ( !c )
			return;

		OfflineMessagesTask *offlineMsgTask = new OfflineMessagesTask( c->rootTask() );
		connect( offlineMsgTask, SIGNAL( receivedOfflineMessage(const Oscar::Message& ) ),
				this, SIGNAL( messageReceived(const Oscar::Message& ) ) );
		offlineMsgTask->go( Task::AutoDelete );
	}

	emit haveContactList();
	emit loggedIn();
}

void Client::receivedIcqInfo( const QString& contact, unsigned int type )
{
	kDebug(OSCAR_RAW_DEBUG) << "received icq info for " << contact
		<< " of type " << type << endl;

	if ( type == ICQUserInfoRequestTask::Short )
		emit receivedIcqShortInfo( contact );
	else
		emit receivedIcqLongInfo( contact );
}

void Client::receivedInfo( quint16 sequence )
{
	UserDetails details = d->userInfoTask->getInfoFor( sequence );
	emit receivedUserInfo( details.userId(), details );
}

void Client::offlineUser( const QString& user, const UserDetails& )
{
	emit userIsOffline( user );
}

void Client::haveOwnUserInfo()
{
	kDebug( OSCAR_RAW_DEBUG ) ;
	UserDetails ud = d->ownStatusTask->getInfo();
	d->ourDetails = ud;
	emit haveOwnInfo();
}

void Client::setCodecProvider( Client::CodecProvider* codecProvider )
{
	d->codecProvider = codecProvider;
}

void Client::setVersion( const Oscar::ClientVersion* version )
{
	d->version = version;
}

void Client::setVersionCap( const QByteArray &cap )
{
	d->versionCap = Guid( cap );
}

// INTERNALS //

QString Client::userId() const
{
	return d->user;
}

QString Client::password() const
{
	return d->pass;
}

int Client::statusXtraz() const
{
	return d->status.xtraz;
}

QString Client::statusDescription() const
{
	return d->status.description;
}

QString Client::statusMessage() const
{
	return d->status.message;
}

void Client::setStatusMessage( const QString &message )
{
	d->status.message = message;
}

QByteArray Client::ipAddress() const
{
	//!TODO determine ip address
	return "127.0.0.1";
}

void Client::notifyTaskError( const Oscar::SNAC& s, int errCode, bool fatal )
{
	emit taskError( s, errCode, fatal );
}

void Client::notifySocketError( int errCode, const QString& msg )
{
	emit socketError( errCode, msg );
}

void Client::sendMessage( const Oscar::Message& msg, bool isAuto)
{
    Connection* c = 0L;
    if ( msg.channel() == 0x0003 )
    {
        c = d->connections.connectionForChatRoom( msg.exchange(), msg.chatRoom() );
        if ( !c )
            return;

        kDebug(OSCAR_RAW_DEBUG) << "sending message to chat room";
        ChatServiceTask* cst = new ChatServiceTask( c->rootTask(), msg.exchange(), msg.chatRoom() );
        cst->setMessage( msg );
        cst->setEncoding( d->codecProvider->codecForAccount()->name() );
        cst->go( Task::AutoDelete );
    }
    else
    {
        c = d->connections.connectionForFamily( 0x0004 );
        if ( !c )
            return;
        SendMessageTask *sendMsgTask = new SendMessageTask( c->rootTask() );
        // Set whether or not the message is an automated response
        sendMsgTask->setAutoResponse( isAuto );
        sendMsgTask->setMessage( msg );
        sendMsgTask->setIp( ourInfo().dcExternalIp().IPv4Addr() ); //TODO: switch to internal
        sendMsgTask->go( Task::AutoDelete );
    }
}

void Client::receivedMessage( const Oscar::Message& msg )
{
	if ( msg.channel() == 2 && !msg.hasProperty( Oscar::Message::AutoResponse ) )
	{
		// channel 2 message needs an autoresponse, regardless of type
		Connection* c = d->connections.connectionForFamily( 0x0004 );
		if ( !c )
			return;

		Oscar::Message response ( msg );
		if ( msg.hasProperty( Oscar::Message::StatusMessageRequest ) )
		{
			QTextCodec* codec = d->codecProvider->codecForContact( msg.sender() );
			response.setText( Oscar::Message::UserDefined, statusMessage(), codec );
			emit userReadsStatusMessage( msg.sender() );
		}
		else if ( msg.messageType() == Oscar::MessageType::Plugin )
		{
			Oscar::MessagePlugin::Types type = msg.plugin()->type();
			Oscar::WORD subType = msg.plugin()->subTypeId();
			if ( type == Oscar::MessagePlugin::XtrazScript )
			{
				if ( subType == Oscar::MessagePlugin::SubScriptNotify )
				{
					using namespace Xtraz;
					XtrazNotify xNotify;
					xNotify.handle( msg.plugin() );
					if ( xNotify.type() == XtrazNotify::Request && xNotify.pluginId() == "srvMng" )
					{
						if ( xNotify.findService( "cAwaySrv" ) )
						{
							XtrazNotify xNotifyResponse;
							xNotifyResponse.setSenderUni( userId() );
							response.setPlugin( xNotifyResponse.statusResponse( statusXtraz(), statusDescription(), statusMessage() ) );
							emit userReadsStatusMessage( msg.sender() );
						}
					}
				}
			}
			else if ( type == Oscar::MessagePlugin::StatusMsgExt )
			{
				Buffer buffer;

				QTextCodec* codec = d->codecProvider->codecForContact( msg.sender() );
				buffer.addLEDBlock( codec->fromUnicode( statusMessage() ) );
				//TODO: Change this to text/x-aolrtf
				buffer.addLEDBlock( "text/plain" );

				msg.plugin()->setData( buffer.buffer() );
				emit userReadsStatusMessage( msg.sender() );
			}
		}
		else
		{
			response.setEncoding( Oscar::Message::UserDefined );
			response.setTextArray( QByteArray() );
		}
		response.setReceiver( msg.sender() );
		response.addProperty( Oscar::Message::AutoResponse );
		SendMessageTask *sendMsgTask = new SendMessageTask( c->rootTask() );
		sendMsgTask->setMessage( response );
		sendMsgTask->go( Task::AutoDelete );
	}

	if ( msg.hasProperty( Oscar::Message::AutoResponse ) )
	{
		if ( msg.hasProperty( Oscar::Message::StatusMessageRequest ) )
		{
			// we got a response to a status message request.
			QString awayMessage( msg.text( d->codecProvider->codecForContact( msg.sender() ) ) );
			kDebug( OSCAR_RAW_DEBUG ) << "Received an away message: " << awayMessage;
			emit receivedAwayMessage( msg.sender(), awayMessage );
		}
		else if ( msg.messageType() == Oscar::MessageType::Plugin )
		{
			kDebug( OSCAR_RAW_DEBUG ) << "Received an plugin message response.";

			Oscar::MessagePlugin::Types type = msg.plugin()->type();
			Oscar::WORD subType = msg.plugin()->subTypeId();
			if ( type == Oscar::MessagePlugin::XtrazScript )
			{
				if ( subType == Oscar::MessagePlugin::SubScriptNotify )
				{
					using namespace Xtraz;
					XtrazNotify xNotify;
					xNotify.handle( msg.plugin() );
					if ( xNotify.type() == XtrazNotify::Response )
					{
						const Xtraz::XAwayService* service = dynamic_cast<const XAwayService*>(xNotify.findService( "cAwaySrv" ));
						if ( service )
							emit receivedXStatusMessage( service->senderId(), service->iconIndex(),
							                             service->description(), service->message() );
					}
				}
			}
			else if ( type == Oscar::MessagePlugin::StatusMsgExt )
			{
				// we got a response to a status message request.
				Buffer buffer( msg.plugin()->data() );

				QTextCodec* codec = d->codecProvider->codecForContact( msg.sender() );
				QString awayMessage = codec->toUnicode( buffer.getLEDBlock() );
				kDebug( OSCAR_RAW_DEBUG ) << "Received an away message: " << awayMessage;
				emit receivedAwayMessage( msg.sender(), awayMessage );
			}
		}
	}
	else
	{
		if ( msg.messageType() == Oscar::MessageType::Plugin )
		{
			kDebug( OSCAR_RAW_DEBUG ) << "Received a plugin message.";
		}
		else if ( !msg.hasProperty( Oscar::Message::StatusMessageRequest ) )
		{
			// let application handle it
			kDebug( OSCAR_RAW_DEBUG ) << "Emitting receivedMessage";
			emit messageReceived( msg );
		}
	}
}

void Client::fileMessage( const Oscar::Message& msg )
{
	kDebug( OSCAR_RAW_DEBUG ) << "internal ip: " << ourInfo().dcInternalIp().toString();
	kDebug( OSCAR_RAW_DEBUG ) << "external ip: " << ourInfo().dcExternalIp().toString();

	sendMessage( msg );
}

void Client::requestAuth( const QString& contactid, const QString& reason )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;
	d->ssiAuthTask->sendAuthRequest( contactid, reason );
}

void Client::sendAuth( const QString& contactid, const QString& reason, bool auth )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;
	d->ssiAuthTask->sendAuthReply( contactid, reason, auth );
}

bool Client::isActive() const
{
	return d->active;
}

bool Client::isIcq() const
{
	return d->isIcq;
}

void Client::setIsIcq( bool isIcq )
{
	d->isIcq = isIcq;
}

void Client::debug( const QString& str )
{
	Q_UNUSED(str);
//	qDebug( "CLIENT: %s", str.toAscii() );
}

void Client::initializeStaticTasks()
{
	//set up the extra tasks
	Connection* c = d->connections.defaultConnection();
	if ( !c )
		return;
	d->errorTask = new ErrorTask( c->rootTask() );
	d->onlineNotifier = new OnlineNotifierTask( c->rootTask() );
	d->ownStatusTask = new OwnUserInfoTask( c->rootTask() );
	d->messageReceiverTask = new MessageReceiverTask( c->rootTask() );
	d->ssiAuthTask = new SSIAuthTask( c->rootTask() );
	d->icqInfoTask = new ICQUserInfoRequestTask( c->rootTask() );
	d->icqTlvInfoTask = new ICQTlvInfoRequestTask( c->rootTask() );
	d->userInfoTask = new UserInfoTask( c->rootTask() );
	d->typingNotifyTask = new TypingNotifyTask( c->rootTask() );
	d->ssiModifyTask = new SSIModifyTask( c->rootTask(), true );

	connect( d->onlineNotifier, SIGNAL( userIsOnline( const QString&, const UserDetails& ) ),
	         this, SIGNAL( receivedUserInfo( const QString&, const UserDetails& ) ) );
	connect( d->onlineNotifier, SIGNAL( userIsOffline( const QString&, const UserDetails& ) ),
	         this, SLOT( offlineUser( const QString&, const UserDetails & ) ) );

	connect( d->ownStatusTask, SIGNAL( gotInfo() ), this, SLOT( haveOwnUserInfo() ) );
	connect( d->ownStatusTask, SIGNAL( buddyIconUploadRequested() ), this,
	         SIGNAL( iconNeedsUploading() ) );

	connect( d->messageReceiverTask, SIGNAL( receivedMessage( const Oscar::Message& ) ),
	         this, SLOT( receivedMessage( const Oscar::Message& ) ) );
	connect( d->messageReceiverTask, SIGNAL( fileMessage( int, const QString, const QByteArray, Buffer ) ),
	         this, SLOT( gotFileMessage( int, const QString, const QByteArray, Buffer ) ) );

	connect( d->ssiAuthTask, SIGNAL( authRequested( const QString&, const QString& ) ),
	         this, SIGNAL( authRequestReceived( const QString&, const QString& ) ) );
	connect( d->ssiAuthTask, SIGNAL( authReplied( const QString&, const QString&, bool ) ),
	         this, SIGNAL( authReplyReceived( const QString&, const QString&, bool ) ) );

	connect( d->icqInfoTask, SIGNAL( receivedInfoFor( const QString&, unsigned int ) ),
	         this, SLOT( receivedIcqInfo( const QString&, unsigned int ) ) );
	connect( d->icqTlvInfoTask, SIGNAL(receivedInfoFor(const QString&)),
	         this, SIGNAL(receivedIcqTlvInfo(const QString&)) );

	connect( d->userInfoTask, SIGNAL( receivedProfile( const QString&, const QString& ) ),
	         this, SIGNAL( receivedProfile( const QString&, const QString& ) ) );
	connect( d->userInfoTask, SIGNAL( receivedAwayMessage( const QString&, const QString& ) ),
	         this, SIGNAL( receivedAwayMessage( const QString&, const QString& ) ) );
	connect( d->typingNotifyTask, SIGNAL( typingStarted( const QString& ) ),
	         this, SIGNAL( userStartedTyping( const QString& ) ) );
	connect( d->typingNotifyTask, SIGNAL( typingFinished( const QString& ) ),
	         this, SIGNAL( userStoppedTyping( const QString& ) ) );
}

void Client::removeGroup( const QString& groupName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug( OSCAR_RAW_DEBUG ) << "Removing group " << groupName << " from Contact";
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->removeGroup( groupName ) )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::addGroup( const QString& groupName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug( OSCAR_RAW_DEBUG ) << "Adding group " << groupName << " to Contact";
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->addGroup( groupName ) )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::addContact( const QString& contactName, const QString& groupName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug( OSCAR_RAW_DEBUG ) << "Adding contact " << contactName << " to ssi in group " << groupName;
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->addContact( contactName, groupName )  )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::removeContact( const QString& contactName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug( OSCAR_RAW_DEBUG ) << "Removing contact " << contactName << " from ssi";
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->removeContact( contactName ) )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::renameGroup( const QString & oldGroupName, const QString & newGroupName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug( OSCAR_RAW_DEBUG ) << "Renaming group " << oldGroupName << " to " << newGroupName;
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->renameGroup( oldGroupName, newGroupName ) )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::modifyContactItem( const OContact& oldItem, const OContact& newItem )
{
	int action = 0; //0 modify, 1 add, 2 remove TODO cleanup!
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	if ( !oldItem && newItem )
		action = 1;
	if ( oldItem && !newItem )
		action = 2;

	kDebug(OSCAR_RAW_DEBUG) << "Add/Mod/Del item on server";
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	switch ( action )
	{
	case 0:
		if ( ssimt->modifyItem( oldItem, newItem ) )
			ssimt->go( Task::AutoDelete );
		else
			delete ssimt;
		break;
	case 1:
		if ( ssimt->addItem( newItem ) )
			ssimt->go( Task::AutoDelete );
		else
			delete ssimt;
		break;
	case 2:
		if ( ssimt->removeItem( oldItem ) )
			ssimt->go( Task::AutoDelete );
		else
			delete ssimt;
		break;
	}
}

void Client::changeContactGroup( const QString& contact, const QString& newGroupName )
{
	Connection* c = d->connections.connectionForFamily( 0x0013 );
	if ( !c )
		return;

	kDebug(OSCAR_RAW_DEBUG) << "Changing " << contact << "'s group to "
		<< newGroupName << endl;
	SSIModifyTask* ssimt = new SSIModifyTask( c->rootTask() );
	if ( ssimt->changeGroup( contact, newGroupName ) )
		ssimt->go( Task::AutoDelete );
	else
		delete ssimt;
}

void Client::requestShortTlvInfo( const QString& contactId, const QByteArray &metaInfoId )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;

	d->icqTlvInfoTask->setUser( Oscar::normalize( contactId ) );
	d->icqTlvInfoTask->setMetaInfoId( metaInfoId );
	d->icqTlvInfoTask->setType( ICQTlvInfoRequestTask::Short );
	d->icqTlvInfoTask->go();
}

void Client::requestMediumTlvInfo( const QString& contactId, const QByteArray &metaInfoId )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;

	d->icqTlvInfoTask->setUser( Oscar::normalize( contactId ) );
	d->icqTlvInfoTask->setMetaInfoId( metaInfoId );
	d->icqTlvInfoTask->setType( ICQTlvInfoRequestTask::Medium );
	d->icqTlvInfoTask->go();
}

void Client::requestLongTlvInfo( const QString& contactId, const QByteArray &metaInfoId )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;

	d->icqTlvInfoTask->setUser( Oscar::normalize( contactId ) );
	d->icqTlvInfoTask->setMetaInfoId( metaInfoId );
	d->icqTlvInfoTask->setType( ICQTlvInfoRequestTask::Long );
	d->icqTlvInfoTask->go();
}

void Client::requestFullInfo( const QString& contactId )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;
	d->icqInfoTask->setUser( contactId );
	d->icqInfoTask->setType( ICQUserInfoRequestTask::Long );
	d->icqInfoTask->go();
}

void Client::requestShortInfo( const QString& contactId )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;
	d->icqInfoTask->setUser( contactId );
	d->icqInfoTask->setType( ICQUserInfoRequestTask::Short );
	d->icqInfoTask->go();
}

void Client::sendWarning( const QString& contact, bool anonymous )
{
	Connection* c = d->connections.connectionForFamily( 0x0004 );
	if ( !c )
		return;
	WarningTask* warnTask = new WarningTask( c->rootTask() );
	warnTask->setContact( contact );
	warnTask->setAnonymous( anonymous );
	QObject::connect( warnTask, SIGNAL( userWarned( const QString&, quint16, quint16 ) ),
	                  this, SIGNAL( userWarned( const QString&, quint16, quint16 ) ) );
	warnTask->go( Task::AutoDelete );
}

bool Client::changeICQPassword( const QString& password )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return false;

	ICQChangePasswordTask* task = new ICQChangePasswordTask( c->rootTask() );
	QObject::connect( task, SIGNAL(finished()), this, SLOT(changeICQPasswordFinished()) );
	task->setPassword( password );
	task->go( Task::AutoDelete );
	return true;
}

void Client::changeICQPasswordFinished()
{
	ICQChangePasswordTask* task = (ICQChangePasswordTask*)sender();
	if ( task->success() )
		d->pass = task->password();

	emit icqPasswordChanged( !task->success() );
}

ICQFullInfo Client::getFullInfo( const QString& contact )
{
	return d->icqTlvInfoTask->fullInfoFor( contact );
}

ICQGeneralUserInfo Client::getGeneralInfo( const QString& contact )
{
	return d->icqInfoTask->generalInfoFor( contact );
}

ICQWorkUserInfo Client::getWorkInfo( const QString& contact )
{
	return d->icqInfoTask->workInfoFor( contact );
}

ICQEmailInfo Client::getEmailInfo( const QString& contact )
{
	return d->icqInfoTask->emailInfoFor( contact );
}

ICQNotesInfo Client::getNotesInfo( const QString& contact )
{
	return d->icqInfoTask->notesInfoFor( contact );
}

ICQMoreUserInfo Client::getMoreInfo( const QString& contact )
{
	return d->icqInfoTask->moreInfoFor( contact );
}

ICQInterestInfo Client::getInterestInfo( const QString& contact )
{
	return d->icqInfoTask->interestInfoFor( contact );
}

ICQOrgAffInfo Client::getOrgAffInfo( const QString& contact )
{
	return d->icqInfoTask->orgAffInfoFor( contact );
}

ICQShortInfo Client::getShortInfo( const QString& contact )
{
	return d->icqInfoTask->shortInfoFor( contact );
}

QList<int> Client::chatExchangeList() const
{
    return d->exchanges;
}

void Client::setChatExchangeList( const QList<int>& exchanges )
{
    d->exchanges = exchanges;
}

void Client::requestAIMProfile( const QString& contact )
{
	d->userInfoTask->requestInfoFor( contact, UserInfoTask::Profile );
}

void Client::requestAIMAwayMessage( const QString& contact )
{
	d->userInfoTask->requestInfoFor( contact, UserInfoTask::AwayMessage );
}

void Client::requestICQAwayMessage( const QString& contact, ICQStatus contactStatus )
{
	kDebug(OSCAR_RAW_DEBUG) << "requesting away message for " << contact;
	Oscar::Message msg;
	msg.setChannel( 2 );
	msg.setReceiver( contact );

	if ( (contactStatus & ICQXStatus) == ICQXStatus )
	{
		Xtraz::XtrazNotify xNotify;
		xNotify.setSenderUni( userId() );

		msg.setMessageType( Oscar::MessageType::Plugin ); // plugin message
		msg.setPlugin( xNotify.statusRequest() );
	}
	else if ( (contactStatus & ICQPluginStatus) == ICQPluginStatus )
	{
		Oscar::WORD subTypeId = 0xFFFF;
		QByteArray subTypeText;

		switch ( contactStatus & ICQStatusMask )
		{
		case ICQOnline:
		case ICQFreeForChat:
		case ICQAway:
			subTypeId = 1;
			subTypeText = "Away Status Message";
			break;
		case ICQOccupied:
		case ICQDoNotDisturb:
			subTypeId = 2;
			subTypeText = "Busy Status Message";
			break;
		case ICQNotAvailable:
			subTypeId = 3;
			subTypeText = "N/A Status Message";
			break;
		default:
			// may be a good way to deal with possible error and lack of online status message?
			emit receivedAwayMessage( contact, "Sorry, this protocol does not support this type of status message" );
			return;
		}

		Oscar::MessagePlugin *plugin = new Oscar::MessagePlugin();
		plugin->setType( Oscar::MessagePlugin::StatusMsgExt );
		plugin->setSubTypeId( subTypeId );
		plugin->setSubTypeText( subTypeText );

		Buffer buffer;
		buffer.addLEDWord( 0x00000000 );
		//TODO: Change this to text/x-aolrtf
		buffer.addLEDBlock( "text/plain" );
		plugin->setData( buffer.buffer() );

		msg.setMessageType( Oscar::MessageType::Plugin ); // plugin message
		msg.setPlugin( plugin );
	}
	else
	{
		msg.addProperty( Oscar::Message::StatusMessageRequest );
		switch ( contactStatus & ICQStatusMask )
		{
		case ICQAway:
			msg.setMessageType( Oscar::MessageType::AutoAway ); // away
			break;
		case ICQOccupied:
			msg.setMessageType( Oscar::MessageType::AutoBusy ); // occupied
			break;
		case ICQNotAvailable:
			msg.setMessageType( Oscar::MessageType::AutoNA ); // not awailable
			break;
		case ICQDoNotDisturb:
			msg.setMessageType( Oscar::MessageType::AutoDND ); // do not disturb
			break;
		case ICQFreeForChat:
			msg.setMessageType( Oscar::MessageType::AutoFFC ); // free for chat
			break;
		default:
			// may be a good way to deal with possible error and lack of online status message?
			emit receivedAwayMessage( contact, "Sorry, this protocol does not support this type of status message" );
			return;
		}
	}
	sendMessage( msg );
}

void Client::addICQAwayMessageRequest( const QString& contact, ICQStatus contactStatus )
{
	kDebug(OSCAR_RAW_DEBUG) << "adding away message request for "
	                         << contact << " to queue" << endl;

	//remove old request if still exists
	removeICQAwayMessageRequest( contact );

	ClientPrivate::AwayMsgRequest amr = { contact, contactStatus };
	d->awayMsgRequestQueue.prepend( amr );

	if ( !d->awayMsgRequestTimer->isActive() )
		d->awayMsgRequestTimer->start( 1000 );
}

void Client::removeICQAwayMessageRequest( const QString& contact )
{
	kDebug(OSCAR_RAW_DEBUG) << "removing away message request for "
	                         << contact << " from queue" << endl;

	QList<ClientPrivate::AwayMsgRequest>::iterator it = d->awayMsgRequestQueue.begin();
	while ( it != d->awayMsgRequestQueue.end() )
	{
		if ( (*it).contact == contact )
			it = d->awayMsgRequestQueue.erase( it );
		else
			it++;
	}
}

void Client::nextICQAwayMessageRequest()
{
	kDebug(OSCAR_RAW_DEBUG) << "request queue count " << d->awayMsgRequestQueue.count();

	if ( d->awayMsgRequestQueue.empty() )
	{
		d->awayMsgRequestTimer->stop();
		return;
	}
	else
	{
		Connection* c = d->connections.connectionForFamily( 0x0004 );
		if ( !c )
			return;

		SNAC s = { 0x0004, 0x0006, 0x0000, 0x00000000 };
		//get time needed to restore level to initial
		//for some reason when we are long under initial level
		//icq server will start to block our messages
		int time = c->rateManager()->timeToInitialLevel( s );
		if ( time > 0 )
		{
			d->awayMsgRequestTimer->start( time );
			return;
		}
		else
		{
			d->awayMsgRequestTimer->start( 5000 );
		}
	}

	ClientPrivate::AwayMsgRequest amr;

	amr = d->awayMsgRequestQueue.back();
	d->awayMsgRequestQueue.pop_back();
	requestICQAwayMessage( amr.contact, amr.contactStatus );
}

void Client::requestStatusInfo( const QString& contact )
{
	d->userInfoTask->requestInfoFor( contact, UserInfoTask::General );
}

void Client::whitePagesSearch( const ICQWPSearchInfo& info )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;
	UserSearchTask* ust = new UserSearchTask( c->rootTask() );
	connect( ust, SIGNAL( foundUser( const ICQSearchResult& ) ),
	         this, SIGNAL( gotSearchResults( const ICQSearchResult& ) ) );
	connect( ust, SIGNAL( searchFinished( int ) ), this, SIGNAL( endOfSearch( int ) ) );
	ust->go( Task::AutoDelete ); //onGo does nothing in this task. This is just here so autodelete works
	ust->searchWhitePages( info );
}

void Client::uinSearch( const QString& uin )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return;
	UserSearchTask* ust = new UserSearchTask( c->rootTask() );
	connect( ust, SIGNAL( foundUser( const ICQSearchResult& ) ),
	         this, SIGNAL( gotSearchResults( const ICQSearchResult& ) ) );
	connect( ust, SIGNAL( searchFinished( int ) ), this, SIGNAL( endOfSearch( int ) ) );
	ust->go( Task::AutoDelete ); //onGo does nothing in this task. This is just here so autodelete works
	ust->searchUserByUIN( uin );
}

void Client::updateProfile( const QString& profile )
{
	Connection* c = d->connections.connectionForFamily( 0x0002 );
	if ( !c )
		return;
	ProfileTask* pt = new ProfileTask( c->rootTask() );
	pt->setProfileText( profile );
	pt->go( Task::AutoDelete );
}

bool Client::updateProfile( const QList<ICQInfoBase*>& infoList )
{
	Connection* c = d->connections.connectionForFamily( 0x0015 );
	if ( !c )
		return false;

	ICQUserInfoUpdateTask* ui = new ICQUserInfoUpdateTask( c->rootTask() );
	ui->setInfo( infoList );
	ui->go( Task::AutoDelete );
	return true;
}

void Client::sendTyping( const QString & contact, bool typing )
{
	Connection* c = d->connections.connectionForFamily( 0x0004 );
	if ( !c )
		return;
	d->typingNotifyTask->setParams( contact, ( typing ? TypingNotifyTask::Begin : TypingNotifyTask::Finished ) );
	d->typingNotifyTask->go(); 	// don't delete the task after sending
}

void Client::connectToIconServer()
{
	Connection* c = d->connections.connectionForFamily( 0x0010 );
	if ( c )
		return;

	requestServerRedirect( 0x0010 );
	return;
}

void Client::setIgnore( const QString& user, bool ignore )
{
	OContact item = ssiManager()->findItem( user,  ROSTER_IGNORE );
	if ( item && !ignore )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Removing " << user << " from ignore list";
		this->modifyContactItem( item, OContact() );
	}
	else if ( !item && ignore )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Adding " << user << " to ignore list";
		OContact s( user, 0, ssiManager()->nextContactId(), ROSTER_IGNORE, QList<TLV>() );
		this->modifyContactItem( OContact(), s );
	}
}

void Client::setVisibleTo( const QString& user, bool visible )
{
	OContact item = ssiManager()->findItem( user,  ROSTER_VISIBLE );
	if ( item && !visible )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Removing " << user << " from visible list";
		this->modifyContactItem( item, OContact() );
	}
	else if ( !item && visible )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Adding " << user << " to visible list";
		OContact s( user, 0, ssiManager()->nextContactId(), ROSTER_VISIBLE, QList<TLV>() );
		this->modifyContactItem( OContact(), s );
	}
}

void Client::setInvisibleTo( const QString& user, bool invisible )
{
	OContact item = ssiManager()->findItem( user,  ROSTER_INVISIBLE );
	if ( item && !invisible )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Removing " << user << " from invisible list";
		this->modifyContactItem( item, OContact() );
	}
	else if ( !item && invisible )
	{
		kDebug(OSCAR_RAW_DEBUG) << "Adding " << user << " to invisible list";
		OContact s( user, 0, ssiManager()->nextContactId(), ROSTER_INVISIBLE, QList<TLV>() );
		this->modifyContactItem( OContact(), s );
	}
}

void Client::requestBuddyIcon( const QString& user, const QByteArray& hash, Oscar::BYTE hashType )
{
	Connection* c = d->connections.connectionForFamily( 0x0010 );
	if ( !c )
		return;

	BuddyIconTask* bit = new BuddyIconTask( c->rootTask() );
	connect( bit, SIGNAL( haveIcon( const QString&, QByteArray ) ),
	         this, SIGNAL( haveIconForContact( const QString&, QByteArray ) ) );
	bit->requestIconFor( user );
	bit->setHashType( hashType );
	bit->setHash( hash );
	bit->go( Task::AutoDelete );
}

void Client::requestServerRedirect( Oscar::WORD family, Oscar::WORD exchange,
                                    QByteArray cookie, Oscar::WORD instance,
                                    const QString& room )
{
	//making the assumption that family 2 will always be the BOS connection
	//use it instead since we can't query for family 1
	Connection* c = d->connections.connectionForFamily( family );
	if ( c && family != 0x000E )
		return; //we already have the connection

	c = d->connections.connectionForFamily( 0x0002 );
	if ( !c )
		return;

    if ( d->redirectionServices.indexOf( family ) == -1 )
        d->redirectionServices.append( family ); //don't add families twice

    if ( d->currentRedirect != 0 )
        return; //we're already doing one redirection

	d->currentRedirect = family;

    //FIXME. this won't work if we have to defer the connection because we're
    //already connecting to something
	ServerRedirectTask* srt = new ServerRedirectTask( c->rootTask() );
    if ( family == 0x000E )
    {
        srt->setChatParams( exchange, cookie, instance );
        srt->setChatRoom( room );
    }

	connect( srt, SIGNAL( haveServer( const QString&, const QByteArray&, Oscar::WORD ) ),
	         this, SLOT( haveServerForRedirect( const QString&, const QByteArray&, Oscar::WORD ) ) );
	srt->setService( family );
	srt->go( Task::AutoDelete );
}

void Client::haveServerForRedirect( const QString& host, const QByteArray& cookie, Oscar::WORD )
{
    //nasty sender() usage to get the task with chat room info
	QObject* o = const_cast<QObject*>( sender() );
    ServerRedirectTask* srt = dynamic_cast<ServerRedirectTask*>( o );

	//create a new connection and set it up
	int colonPos = host.indexOf(':');
	QString realHost, realPort;
	if ( colonPos != -1 )
	{
		realHost = host.left( colonPos );
		realPort = host.right(4); //we only need 4 bytes
	}
	else
	{
		realHost = host;
		realPort = QString::fromLatin1("5190");
	}

	Connection* c = createConnection();
	//create the new login task
	m_loginTaskTwo = new StageTwoLoginTask( c->rootTask() );
	m_loginTaskTwo->setCookie( cookie );
	QObject::connect( m_loginTaskTwo, SIGNAL( finished() ), this, SLOT( serverRedirectFinished() ) );

	//connect
	connectToServer( c, realHost, realPort.toInt(), false );
  	QObject::connect( c, SIGNAL( connected() ), this, SLOT( streamConnected() ) );

    if ( srt )
        d->connections.addChatInfoForConnection( c, srt->chatExchange(), srt->chatRoomName() );
}

void Client::serverRedirectFinished()
{
	if ( m_loginTaskTwo &&  m_loginTaskTwo->statusCode() == 0 )
	{ //stage two was successful
		Connection* c = d->connections.connectionForFamily( d->currentRedirect );
		if ( !c )
			return;
		ClientReadyTask* crt = new ClientReadyTask( c->rootTask() );
		crt->setFamilies( c->supportedFamilies() );
		crt->go( Task::AutoDelete );
	}

	kDebug(OSCAR_RAW_DEBUG) << "redirection finished for service "
	                         << d->currentRedirect << endl;

	if ( d->currentRedirect == 0x0010 )
		emit iconServerConnected();

	if ( d->currentRedirect == 0x000D )
	{
		connect( this, SIGNAL( chatNavigationConnected() ),
				 this, SLOT( requestChatNavLimits() ) );
		emit chatNavigationConnected();
	}

    if ( d->currentRedirect == 0x000E )
    {
        //HACK! such abuse! think of a better way
        if ( !m_loginTaskTwo )
        {
            kWarning(OSCAR_RAW_DEBUG) << "no login task to get connection from!";
            emit redirectionFinished( d->currentRedirect );
            return;
        }

        Connection* c = m_loginTaskTwo->client();
        QString roomName = d->connections.chatRoomForConnection( c );
        Oscar::WORD exchange = d->connections.exchangeForConnection( c );
        if ( c )
        {
            kDebug(OSCAR_RAW_DEBUG) << "setting up chat connection";
            ChatServiceTask* cst = new ChatServiceTask( c->rootTask(), exchange, roomName );
            connect( cst, SIGNAL( userJoinedChat( Oscar::Oscar::WORD, const QString&, const QString& ) ),
                     this, SIGNAL( userJoinedChat( Oscar::Oscar::WORD, const QString&, const QString& ) ) );
            connect( cst, SIGNAL( userLeftChat( Oscar::Oscar::WORD, const QString&, const QString& ) ),
                     this, SIGNAL( userLeftChat( Oscar::Oscar::WORD, const QString&, const QString& ) ) );
            connect( cst, SIGNAL( newChatMessage( const Oscar::Message& ) ),
                     this, SIGNAL( messageReceived( const Oscar::Message& ) ) );
        }
        emit chatRoomConnected( exchange, roomName );
    }

	emit redirectionFinished( d->currentRedirect );

}

void Client::checkRedirectionQueue( Oscar::WORD family )
{
	kDebug(OSCAR_RAW_DEBUG) << "checking redirection queue";
	d->redirectionServices.removeAll( family );
    d->currentRedirect = 0;
	if ( !d->redirectionServices.isEmpty() )
	{
		kDebug(OSCAR_RAW_DEBUG) << "scheduling new redirection";
		requestServerRedirect( d->redirectionServices.front() );
	}
}


void Client::requestChatNavLimits()
{
	Connection* c = d->connections.connectionForFamily( 0x000D );
	if ( !c )
		return;

	kDebug(OSCAR_RAW_DEBUG) << "requesting chat nav service limits";
	ChatNavServiceTask* cnst = new ChatNavServiceTask( c->rootTask() );
    cnst->setRequestType( ChatNavServiceTask::Limits );
    QObject::connect( cnst, SIGNAL( haveChatExchanges( const QList<int>& ) ),
                      this, SLOT( setChatExchangeList( const QList<int>& ) ) );
	cnst->go( Task::AutoDelete ); //autodelete

}

void Client::determineDisconnection( int code, const QString& string )
{
    if ( !sender() )
        return;

    //yay for the sender() hack!
    QObject* obj = const_cast<QObject*>( sender() );
    Connection* c = dynamic_cast<Connection*>( obj );
    if ( !c )
        return;

	if ( c->isSupported( 0x0002 ) ||
	     d->stage == ClientPrivate::StageOne ) //emit on login
	{
		emit socketError( code, string );
	}

    //connection is deleted. deleteLater() is used
    d->connections.remove( c );
    c = 0;
}

void Client::sendBuddyIcon( const QByteArray& iconData )
{
	Connection* c = d->connections.connectionForFamily( 0x0010 );
	if ( !c )
		return;

	kDebug(OSCAR_RAW_DEBUG) << "icon length is " << iconData.size();
	BuddyIconTask* bit = new BuddyIconTask( c->rootTask() );
	bit->uploadIcon( iconData.size(), iconData );
	bit->go( Task::AutoDelete );
}

void Client::joinChatRoom( const QString& roomName, int exchange )
{
    Connection* c = d->connections.connectionForFamily( 0x000D );
    if ( !c )
        return;

    kDebug(OSCAR_RAW_DEBUG) << "joining the chat room '" << roomName
                             << "' on exchange " << exchange << endl;
    ChatNavServiceTask* cnst = new ChatNavServiceTask( c->rootTask() );
    connect( cnst, SIGNAL( connectChat( Oscar::WORD, QByteArray, Oscar::WORD, const QString& ) ),
             this, SLOT( setupChatConnection( Oscar::WORD, QByteArray, Oscar::WORD, const QString& ) ) );
    cnst->createRoom( exchange, roomName );

}

void Client::setupChatConnection( Oscar::WORD exchange, QByteArray cookie, Oscar::WORD instance, const QString& room )
{
    kDebug(OSCAR_RAW_DEBUG) << "cookie is:" << cookie;
    QByteArray realCookie( cookie );
    kDebug(OSCAR_RAW_DEBUG) << "connection to chat room";
    requestServerRedirect( 0x000E, exchange, realCookie, instance, room );
}

void Client::disconnectChatRoom( Oscar::WORD exchange, const QString& room )
{
    Connection* c = d->connections.connectionForChatRoom( exchange, room );
    if ( !c )
        return;

    d->connections.remove( c );
    c = 0;
}

Connection* Client::createConnection()
{
	ClientStream* cs = new ClientStream( new QTcpSocket(), 0 );
	cs->setNoopTime( 60000 );
	Connection* c = new Connection( cs, "BOS" );
	cs->setConnection( c );
	c->setClient( this );
	return c;
}

void Client::deleteStaticTasks()
{
	delete d->errorTask;
	delete d->onlineNotifier;
	delete d->ownStatusTask;
	delete d->messageReceiverTask;
	delete d->ssiAuthTask;
	delete d->icqInfoTask;
	delete d->icqTlvInfoTask;
	delete d->userInfoTask;
	delete d->typingNotifyTask;
	delete d->ssiModifyTask;

	d->errorTask = 0;
	d->onlineNotifier = 0;
	d->ownStatusTask = 0;
	d->messageReceiverTask = 0;
	d->ssiAuthTask = 0;
	d->icqInfoTask = 0;
	d->icqTlvInfoTask = 0;
	d->userInfoTask = 0;
	d->typingNotifyTask = 0;
	d->ssiModifyTask = 0;
}

bool Client::hasIconConnection( ) const
{
	Connection* c = d->connections.connectionForFamily( 0x0010 );
	return c;
}

void Client::sendFiles( const QString& contact, const QStringList& files, Kopete::Transfer *t )
{
	Connection* c = d->connections.connectionForFamily( 0x0004 );
	if ( !c )
		return;

	FileTransferTask *ft = new FileTransferTask( c->rootTask(), contact, ourInfo().userId(), files, t );
	connect( ft, SIGNAL( sendMessage( const Oscar::Message& ) ),
	         this, SLOT( fileMessage( const Oscar::Message& ) ) );
	ft->go( Task::AutoDelete );
}

void Client::gotFileMessage( int type, const QString from, const QByteArray cookie, Buffer buf)
{
	Connection* c = d->connections.connectionForFamily( 0x0004 );
	if ( !c )
		return;
	//pass the message to the matching task if we can
	const QList<FileTransferTask*> p = c->rootTask()->findChildren<FileTransferTask*>();
	foreach( FileTransferTask *t, p)
	{
		if ( t->take( type, cookie, buf ) )
		{
			return;
		}
	}
	//maybe it's a new request!
	if ( type == 0 )
	{
		kDebug(14151) << "new request :)";
		FileTransferTask *ft = new FileTransferTask( c->rootTask(), from, ourInfo().userId(), cookie, buf );
		connect( ft, SIGNAL( getTransferManager( Kopete::TransferManager ** ) ),
				SIGNAL( getTransferManager( Kopete::TransferManager ** ) ) );
		connect( ft, SIGNAL( askIncoming( QString, QString, Oscar::DWORD, QString, QString ) ),
				SIGNAL( askIncoming( QString, QString, Oscar::DWORD, QString, QString ) ) );
		connect( ft, SIGNAL( sendMessage( const Oscar::Message& ) ),
				this, SLOT( fileMessage( const Oscar::Message& ) ) );
		ft->go( Task::AutoDelete );
		return;
	}

	kDebug(14151) << "nobody wants it :(";
}

#include "client.moc"
//kate: tab-width 4; indent-mode csands; space-indent off; replace-tabs off;
