/*
 * newssourcedlgimpl.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "newssourcedlgimpl.h"
#include "xmlnewsaccess.h"
#include "configaccess.h"
#include "newsiconmgr.h"

#include <kcombobox.h>
#include <klineedit.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kurlrequester.h>
#include <kvbox.h>

#include <qcheckbox.h>
#include <qlabel.h>
#include <QProgressBar>
#include <qtimer.h>
#include <QPixmap>

SuggestProgressDlg::SuggestProgressDlg(const KUrl &url, QWidget *parent, const char *name)
	: KDialog(parent),
	m_gotSourceFile(false),
	m_gotIcon(false)
{
    setCaption(  i18n("Downloading Data") );
    setButtons( Cancel );
    setDefaultButton( Cancel );
    setModal( true );
    KVBox *mainWidget = new KVBox( this );
    setMainWidget( mainWidget);

    new QLabel(i18n("<qt>Please wait while KNewsTicker is downloading some "
                    "data necessary to suggest reasonable values.<br/>"
                    "<br/>"
                    "This will not take longer than one minute.</qt>" ),
               mainWidget);

    m_progressBar = new QProgressBar(mainWidget);
    m_progressBar->setMaximum(60);
    m_progressBar->setTextVisible(false);

    m_timeoutTimer = new QTimer(this);
    connect(m_timeoutTimer, SIGNAL(timeout()), this, SLOT(slotTimeoutTick()));
    m_timeoutTimer->start(1000);

    m_xmlSrc = new XMLNewsSource;
    connect(m_xmlSrc, SIGNAL(loadComplete(XMLNewsSource *, bool)),
            this, SLOT(slotLoadComplete(XMLNewsSource *, bool)));
    m_xmlSrc->loadFrom(url);

    connect(NewsIconMgr::self(), SIGNAL(gotIcon(const KUrl &, const QPixmap &)),
            this, SLOT(slotGotIcon(const KUrl &, const QPixmap &)));
    KUrl u = url;
    if (url.isLocalFile())
        u = QString();
    else
        u.setEncodedPathAndQuery(QLatin1String("/favicon.ico"));
    NewsIconMgr::self()->getIcon(u);
}

SuggestProgressDlg::~SuggestProgressDlg()
{
	delete m_xmlSrc;
}

void SuggestProgressDlg::slotTimeoutTick()
{
	if (m_progressBar->value() == m_progressBar->maximum()) {
		m_timeoutTimer->stop();
		KMessageBox::error(this, i18n("Could not retrieve the specified source file."));
		reject();
		return;
	}
	m_progressBar->setValue(m_progressBar->value() + 1);
}

void SuggestProgressDlg::slotLoadComplete(XMLNewsSource *, bool succeeded)
{
	m_gotSourceFile = true;
	m_succeeded = succeeded;

	if (m_gotIcon)
		done(succeeded ? QDialog::Accepted : QDialog::Rejected);
}

void SuggestProgressDlg::slotGotIcon(const KUrl &url, const QPixmap &pixmap)
{
	m_gotIcon = true;
	m_icon = pixmap;
	m_iconURL = url;

	if (m_gotIcon)
		done(m_succeeded ? QDialog::Accepted : QDialog::Rejected);
}

NewsSourceDlgImpl::NewsSourceDlgImpl(QWidget *parent,  const char *name, bool modal, Qt::WFlags fl)
	: NewsSourceDlg(parent, name, modal, fl),
	m_modified(false)
{
	connect(NewsIconMgr::self(), SIGNAL(gotIcon(const KUrl &, const QPixmap &)),
	        this, SLOT(slotGotIcon(const KUrl &, const QPixmap &)));

	for (unsigned int i = 0; i < DEFAULT_SUBJECTS; i++)
		comboCategory->addItem(
				NewsSourceBase::subjectText(static_cast<NewsSourceBase::Subject>(i)));

}

void NewsSourceDlgImpl::slotCancelClicked()
{
	close();
}

void NewsSourceDlgImpl::slotOkClicked()
{
	KUrl url (polishedURL(KUrl(urlSourceFile->url())));

	if (!validateURL(url))
		return;

	if (leName->text().isEmpty()) {
		KMessageBox::error(this, i18n("You have to specify a name for this news"
					" source to be able to use it."), i18n("No Name Specified"));
		return;
	}

	// This finds out which subject is selected in the 'Subject' combo box.
	NewsSourceBase::Subject subject = NewsSourceBase::Computers;
	for (unsigned int i = 0; i < DEFAULT_SUBJECTS; i++) {
		NewsSourceBase::Subject thisSubj = static_cast<NewsSourceBase::Subject>(i);
		if (comboCategory->currentText() == NewsSourceBase::subjectText(thisSubj)) {
			subject = thisSubj;
			break;
		}
	}

	KUrl iconURL ( leIcon->text() );
	if (iconURL.protocol().isEmpty())
		if (iconURL.host().startsWith(QLatin1String("ftp.")))
			iconURL.setProtocol(QLatin1String("ftp"));
		else if (iconURL.host().startsWith(QLatin1String("www.")))
			iconURL.setProtocol(QLatin1String("http"));
		else
			iconURL.setProtocol(QLatin1String("file"));

	NewsSourceBase::Data nsd(leName->text(), url.url(), iconURL.url(), subject,
			sbMaxArticles->value(), true, cbProgram->isChecked());

	emit newsSource(nsd);

	close();
}

void NewsSourceDlgImpl::slotSourceFileChanged()
{
	bSuggest->setEnabled(!urlSourceFile->url().isEmpty());
}

void NewsSourceDlgImpl::slotSuggestClicked()
{
	KUrl url ( polishedURL(KUrl( urlSourceFile->url() )) );

	if (!validateURL(url))
		return;

	SuggestProgressDlg dlg(url, this);
	if (dlg.exec() == QDialog::Accepted) {
		pixmapIcon->setPixmap(dlg.icon());
		if (NewsIconMgr::self()->isStdIcon(dlg.icon()))
			leIcon->clear();
		else
			leIcon->setText(dlg.iconURL().url());
		cbProgram->setChecked(false);
		leName->setText(dlg.xmlSrc()->newsSourceName());
		sbMaxArticles->setValue(dlg.xmlSrc()->articles().count());
	}
}

void NewsSourceDlgImpl::slotModified()
{
	m_modified = true;
}

void NewsSourceDlgImpl::setup(const NewsSourceBase::Data &nsd, bool modify)
{
	leName->setText(nsd.name);
	urlSourceFile->setUrl(nsd.sourceFile);
	cbProgram->setChecked(nsd.isProgram);
	comboCategory->setCurrentIndex(nsd.subject);
	sbMaxArticles->setValue(nsd.maxArticles);
	KUrl iconURL ( nsd.icon );
	if (iconURL.protocol() == QLatin1String("file"))
		iconURL.setProtocol(QString::null);	//krazy:exclude=nullstrassign for old broken gcc
	leIcon->setText(iconURL.url());
	NewsIconMgr::self()->getIcon(iconURL);
	if (modify == true) {
		setWindowTitle(i18n("Edit News Source"));
	}
}

KUrl NewsSourceDlgImpl::polishedURL(const KUrl &url) const
{
	KUrl newURL = url;

	if (url.protocol().isEmpty())
		if (url.url().startsWith(QLatin1String("ftp")))
			newURL = QLatin1String("ftp://") + url.url();
		else
			newURL = QLatin1String("http://") + url.url();

	return newURL;
}

bool NewsSourceDlgImpl::validateURL(const KUrl &url)
{
	if (url.isEmpty()) {
		KMessageBox::error(this, i18n("You have to specify the source file for this"
					" news source to be able to use it."), i18n("No Source File"
					" Specified"));
		return false;
	}

	if (!url.isValid() || !url.hasPath() || url.encodedPathAndQuery() == QLatin1String("/")) {
		KMessageBox::error(this, i18n("KNewsTicker needs a valid RDF or RSS file to"
					" suggest sensible values. The specified source file is invalid."),
					i18n("Invalid Source File"));
		return false;
	}

	return true;
}

void NewsSourceDlgImpl::slotGotIcon(const KUrl &, const QPixmap &pixmap)
{
	pixmapIcon->setPixmap(pixmap);
}

#include "newssourcedlgimpl.moc"
