/*
 * xmlnewsaccess.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "xmlnewsaccess.h"

#include <kcharsets.h>
#include <kdebug.h>
#include <kglobal.h>

#include <qbuffer.h>
#include <qdom.h>
#include <qregexp.h>

XMLNewsArticle::XMLNewsArticle(const QString &headline, const KUrl &address)
	: m_headline(headline),
	m_address(address)
{
}

XMLNewsArticle &XMLNewsArticle::operator=(const XMLNewsArticle &other)
{
	m_headline = other.m_headline;
	m_address = other.m_address;
	return *this;
}

bool XMLNewsArticle::operator==(const XMLNewsArticle &a)
{
	return m_headline == a.headline() && m_address == a.address();
}

XMLNewsSource::XMLNewsSource() : QObject(),
 	m_name(QString::null),	//krazy:exclude=nullstrassign for old broken gcc
	m_link(QString::null),	//krazy:exclude=nullstrassign for old broken gcc
	m_description(QString::null),	//krazy:exclude=nullstrassign for old broken gcc
	m_downloadData(0)
{
}

XMLNewsSource::~XMLNewsSource()
{
	delete m_downloadData; // Might exist if we are in the middle of a download
}

void XMLNewsSource::loadFrom(const KUrl &url)
{
	if ( m_downloadData != 0 ) {
		kDebug( 5005 ) << "XMLNewsSource::loadFrom(): Busy, ignoring load "
		                   "request for " << url << endl;
		return;
	}
	m_downloadData = new QBuffer;
	m_downloadData->open(QIODevice::WriteOnly);

	KIO::Job *job = KIO::get(url, KIO::NoReload, KIO::HideProgressInfo);
	job->addMetaData(QLatin1String("UserAgent"),
	                 QLatin1String("KNewsTicker v0.2"));
	connect(job, SIGNAL(data(KIO::Job *, const QByteArray &)),
			SLOT(slotData(KIO::Job *, const QByteArray &)));
	connect(job, SIGNAL(result(KJob *)), SLOT(slotResult(KJob *)));
}

void XMLNewsSource::slotData(KIO::Job *, const QByteArray &data)
{
	m_downloadData->write(data.data(), data.size());
}

void XMLNewsSource::slotResult(KJob *job)
{
	kDebug(5005) << "XMLNewsSource::slotResult(): Finished downloading data (" << job->error() << ").";
	processData(m_downloadData->buffer(), !job->error());
	delete m_downloadData;
	m_downloadData = 0;
}

void XMLNewsSource::processData(const QByteArray &data, bool okSoFar)
{
	bool validContent = okSoFar;
	kDebug(5005) << "XMLNewsSource::processData(): validContent = " << validContent;

	if (okSoFar) {
		/*
		 * Some servers prepend whitespace before the <?xml...?> declaration.
		 * Since QDom doesn't like that we strip this first.
		 */
		QDomDocument domDoc;

		const char *charData = data.data();
		int len = data.count();

		while (len && (*charData == ' ' || *charData == '\n' || *charData == '\t' || *charData == '\r') ) {
			len--;
			charData++;
		}

		QByteArray tmpData = QByteArray::fromRawData(charData, len);

		if (validContent = domDoc.setContent(tmpData)) {
			QDomNode channelNode = domDoc.documentElement().namedItem(QLatin1String("channel"));

			m_name = channelNode.namedItem(QLatin1String("title")).toElement().text().simplified();
			kDebug(5005) << "XMLNewsSource::processData(): Successfully updated " << m_name;
			m_link = channelNode.namedItem(QLatin1String("link")).toElement().text().simplified();
			m_description = channelNode.namedItem(QLatin1String("description")).toElement().text().simplified();

			QDomNodeList items = domDoc.elementsByTagName(QLatin1String("item"));
			m_articles.clear();
			QDomNode itemNode;
			QString headline, address;
			for (int i = 0; i < items.count(); i++) {
				itemNode = items.item(i);
				headline = KCharsets::resolveEntities(itemNode.namedItem(QLatin1String("title")).toElement().text().simplified());
				address = KCharsets::resolveEntities(itemNode.namedItem(QLatin1String("link")).toElement().text().simplified());
				m_articles.append(XMLNewsArticle(headline, KUrl( address )));
			}
		}
		kDebug(5005) << "XMLNewsSource::processData(): validContent = " << validContent;
		tmpData.clear();
	}

	emit loadComplete(this, validContent);
}

#include "xmlnewsaccess.moc"
