/*
  KPF - Public fileserver for KDE

  Copyright 2001 Rik Hemsley (rikkus) <rik@kde.org>

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <qlabel.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qfileinfo.h>
#include <qcursor.h>
//Added by qt3to4:
#include <QMouseEvent>
#include <QDragEnterEvent>
#include <QVBoxLayout>
#include <QEvent>
#include <QDropEvent>

#include <kiconloader.h>
#include <klocale.h>
#include <kmenu.h>
#include <kurl.h>
#include "Defines.h"
#include "AppletItem.h"
#include "WebServerManager.h"
#include "WebServer.h"
#include "BandwidthGraph.h"
#include "ActiveMonitorWindow.h"
#include "SingleServerConfigDialog.h"

namespace KPF
{
  AppletItem::AppletItem(WebServer * server, QWidget * parent)
    : QWidget       (parent, "KPF::AppletItem"),
      server_       (server),
      configDialog_ (0L),
      monitorWindow_      (0L),
      graph_        (0L),
      popup_        (0L)
  {
    setAcceptDrops(true);

    graph_ = new BandwidthGraph(server_, BandwidthGraph::UseOverlays, this);

    graph_->setAcceptDrops(true);

    graph_->installEventFilter(this);

    (new QVBoxLayout(this))->addWidget(graph_);

    QString popupTitle(i18n("kpf - %1", server_->root()));

    popup_ = new KMenu(this);

    popup_->addTitle
      (SmallIcon("kpf"),          popupTitle);

    newFile = popup_->addAction(SmallIcon("filenew"),i18n("New Server..."));

    popup_->addSeparator();

    monitor = popup_->addAction(SmallIcon("viewmag"), i18n("Monitor"));

    configure = popup_->addAction(SmallIcon("configure"),    i18n("Preferences..."));

    remove = popup_->addAction(SmallIcon("remove"),i18n("Remove"));

    reload = popup_->addAction(SmallIcon("reload"),i18n("Restart"));

    pause = popup_->addAction(SmallIcon("player_pause"), i18n("Pause"));

    monitorWindow_ = new ActiveMonitorWindow(server_);

    connect
      (
       monitorWindow_,
       SIGNAL(dying(ActiveMonitorWindow *)),
       SLOT(slotActiveMonitorWindowDying(ActiveMonitorWindow *))
      );
  }

  AppletItem::~AppletItem()
  {
    delete configDialog_;
    configDialog_ = 0;

    delete monitorWindow_;
    monitorWindow_ = 0;
  }

    bool
  AppletItem::eventFilter(QObject *, QEvent * ev)
  {
    switch (ev->type())
    {

      case QEvent::MouseButtonRelease:
        {
          QMouseEvent * e = static_cast<QMouseEvent *>(ev);

          if (0 == e)
          {
            kpfDebug
              << "Hmm, should have been able to static_cast here." << endl;
            break;
          }

          if (!rect().contains(e->pos()))
          {
            break;
          }

          if (Qt::LeftButton == e->button())
          {
            // Show monitor.

            if (0 == monitorWindow_)
              monitorServer();

            else
            {
              if (monitorWindow_->isVisible())
                monitorWindow_->hide();
              else
                monitorWindow_->show();
            }
          }

          return true;
        }
        break;

      case QEvent::MouseButtonPress:
        {
          QMouseEvent * e = static_cast<QMouseEvent *>(ev);

          if (0 == e)
          {
            kpfDebug
              << "Hmm, should have been able to static_cast here." << endl;
            break;
          }

          if (Qt::RightButton != e->button() && Qt::LeftButton != e->button())
            break;

          if (server_->paused())
          {
            pause->setIcon(SmallIcon("1rightarrow"));
            pause->setText(i18n("Unpause"));
          }
          else
          {
            pause->setIcon(SmallIcon("player_pause"));
            pause->setText(i18n("Pause"));

          }
          QAction *ret = popup_->exec(QCursor::pos() );
          if ( ret == newFile )
              emit(newServer());
          else if ( ret == monitor )
              monitorServer();
          else if ( ret == configure )
              configureServer();
          else if ( ret == remove )
              removeServer();
          else if ( ret == reload )
               restartServer();
          else if ( ret == pause )
              pauseServer();

          return true;
        }
        break;

      case QEvent::DragEnter:
        {
          QDragEnterEvent * e = static_cast<QDragEnterEvent *>(ev);

          if (0 == e)
          {
            kpfDebug
              << "Hmm, should have been able to static_cast here." << endl;
            break;
          }

          KUrl::List l=KUrl::List::fromMimeData( e->mimeData() );

          if (!l.isEmpty())
            break;

          if (l.count() != 1)
            break;

          const KUrl &url = l[0];

          if (!url.isLocalFile() || !QFileInfo(url.path()).isDir())
            break;

          e->accept();
          return true;
        }
        break;

      case QEvent::Drop:
        {
          QDropEvent * e = static_cast<QDropEvent *>(ev);

          if (0 == e)
          {
            kpfDebug
              << "Hmm, should have been able to static_cast here." << endl;
            break;
          }

          KUrl::List l = KUrl::List::fromMimeData( e->mimeData() );


          if (!l.isEmpty())
            break;

          if (l.count() != 1)
            break;

          const KUrl &url = l[0];

          if (!url.isLocalFile() || !QFileInfo(url.path()).isDir())
            break;

          e->accept();
          emit(newServerAtLocation(url.path()));
          return true;
        }
        break;

      default:
        break;
    }

    return false;
  }

    void
  AppletItem::slotActiveMonitorWindowDying(ActiveMonitorWindow *)
  {
    // We used to delete it here, but let's not. See if this is a CPU hog.
#if 0
    delete monitorWindow_;
    monitorWindow_ = 0;
#endif
    monitorWindow_->hide();
  }

    void
  AppletItem::slotConfigDialogDying(SingleServerConfigDialog *)
  {
    graph_->setTooltip();

    configDialog_->delayedDestruct();
    configDialog_ = 0;
  }

    void
  AppletItem::slotNewServer()
  {
    emit(newServer());
  }

    void
  AppletItem::monitorServer()
  {
    // We used to delete it here, but let's not. See if this is a CPU hog.
#if 0
    if (0 != monitorWindow_)
    {
      monitorWindow_->show();
      return;
    }

    monitorWindow_ = new ActiveMonitorWindow(server_);

    connect
      (
       monitorWindow_,
       SIGNAL(dying(ActiveMonitorWindow *)),
       SLOT(slotActiveMonitorWindowDying(ActiveMonitorWindow *))
      );
#else
    if (!monitorWindow_)
	return;
#endif

    monitorWindow_->show();
    monitorWindow_->raise();
  }

    void
  AppletItem::removeServer()
  {
    QTimer::singleShot(0, this, SLOT(slotSuicide()));
  }

    void
  AppletItem::configureServer()
  {
    if (0 != configDialog_)
    {
      configDialog_->show();
      return;
    }

    configDialog_ = new SingleServerConfigDialog(server_, 0);

    connect
      (
       configDialog_,
       SIGNAL(dying(SingleServerConfigDialog *)),
       SLOT(slotConfigDialogDying(SingleServerConfigDialog *))
      );

    configDialog_->show();
  }

    void
  AppletItem::slotSuicide()
  {
    WebServerManager::instance()->disableServer(server_->root());
  }

    void
  AppletItem::restartServer()
  {
    server_->restart();
  }

    void
  AppletItem::pauseServer()
  {
    server_->pause(!server_->paused());
  }

    WebServer *
  AppletItem::server()
  {
    return server_;
  }
}

#include "AppletItem.moc"
// vim:ts=2:sw=2:tw=78:et
